/*  Cabecera obligatoria: NO BORRAR NI MODIFICAR este bloque inicial en ningún fichero.
    Archivo: scripts/panel/detect.js – Rol: Detección de sesión en Wallapop (watchdog, handshake, listeners)
    Descripcion de todo el proyecto:
         MitikLive Wallapop (MV3): panel lateral en Wallapop para gestionar anuncios conectado a FastAPI
            (login JWT, export/import, backups/renovaciones).
    Aviso: NO BORRAR los comentarios descriptivos situados encima de cada función, solo cambiarlos si se modifica la funcion.
    En Rol (linea mas arriba): si está vacía o el fichero se modifica o reestructura, modificar esa linea de rol
*/

import { retryWithBackoff } from '../utils.js';
import { logger } from './logger.js';

import { 
  LAST_GOOD, NOLOGIN_HOLD, NOLOGIN_HOLD_MS,
  set_LAST_GOOD, set_NOLOGIN_HOLD
} from './state.js';
import { 
  paintConnected, paintNoLogin, paintNotDetected, 
  setBadgeLoading, setBadgeTitle 
} from './ui.js';

let CS_WATCHDOG = null;

/* ===========================
   Helpers de detección Walla
   =========================== */

/** Extrae alias de una URL de perfil de Wallapop */
export function aliasFromProfileUrl(u) {
  try {
    const last = new URL(u).pathname.split('/').filter(Boolean).pop() || '';
    return last.replace(/-\d+$/, '');
  } catch {
    return null;
  }
}

/** Espera a que una pestaña complete su carga */
export async function waitTabComplete(tabId, timeout = 20000) {
  try {
    const t = await chrome.tabs.get(tabId);
    if (t.status === 'complete') return;
  } catch {}

  return await new Promise((res) => {
    const to = setTimeout(() => { cleanup(); res(); }, timeout);
    const cleanup = () => {
      clearTimeout(to);
      chrome.tabs.onUpdated.removeListener(onUpd);
    };
    const onUpd = (id, info) => {
      if (id === tabId && info.status === 'complete') {
        cleanup();
        res();
      }
    };
    chrome.tabs.onUpdated.addListener(onUpd);
  });
}

/** Ping rápido al content script para verificar si está activo */
export function pingContentScript(tabId, timeout = 400) {
  return new Promise((resolve) => {
    let done = false;
    try {
      chrome.tabs.sendMessage(tabId, { type: 'SYS.PING' }, (r) => {
        // ✅ v10.5.82: Silenciar lastError para evitar ruido en consola
        void chrome.runtime.lastError;
        if (done) return;
        done = true;
        resolve(!!(r && r.ok));
      });
    } catch {
      if (!done) { done = true; resolve(false); }
    }
    setTimeout(() => { 
      if (!done) { done = true; resolve(false); } 
    }, timeout);
  });
}

/** Verifica si el usuario está logueado en Wallapop (con reintentos) */
export async function isLoggedInWallapopTab(tabId, retries = 3) {
  try {
    return await retryWithBackoff(
      async () => {
        // ✅ v10.5.82: Envolver en Promise para manejar lastError silenciosamente
        const r = await new Promise((resolve) => {
          chrome.tabs.sendMessage(tabId, { type: 'DOM.IS_LOGGED_IN' }, (response) => {
            void chrome.runtime.lastError; // Silenciar error
            resolve(response);
          });
        });
        if (r && r.logged === true) return true;
        if (r && r.logged === false) return false;
        throw new Error('no_response');
      },
      { 
        maxRetries: retries, 
        baseDelay: 250,
      }
    );
  } catch {
    return false;
  }
}

/** Encuentra la mejor pestaña de Wallapop (prioriza activa) */
export async function findBestWallapopTab() {
  const tabs = await chrome.tabs.query({ 
    url: ['https://*.wallapop.com/*', 'https://*.wallapop.es/*'] 
  });
  if (!tabs || !tabs.length) return null;
  const active = tabs.find(t => t.active);
  return active || tabs[0];
}

/** Cancela el hold de degradación a "no login" */
function cancelNoLoginHold() {
  const hold = NOLOGIN_HOLD;
  if (hold) {
    clearTimeout(hold);
    set_NOLOGIN_HOLD(null);
  }
}

/** Programa degradación diferida a "no login" (evita flickering) */
function scheduleNoLogin(_reason = '') {
  cancelNoLoginHold();
  const hold = setTimeout(() => {
    paintNoLogin();
    set_LAST_GOOD({ alias: '', avatar: '' });
    set_NOLOGIN_HOLD(null);
  }, NOLOGIN_HOLD_MS);
  set_NOLOGIN_HOLD(hold);
}

/* ============================
   Detección principal (única)
   ============================ */

/** 
 * Detecta si hay sesión activa en Wallapop.
 * Retorna { ok, reason?, alias? }
 */
export async function detectLoggedWalla() {
  const tab = await findBestWallapopTab();

  // 1) No hay pestañas de Wallapop
  if (!tab) {
    cancelNoLoginHold();
    set_LAST_GOOD({ alias: '', avatar: '' });
    paintNotDetected();
    return { ok: false, reason: 'NO_TAB' };
  }

  // 2) Hay pestaña: comprobar login
  try {
    const logged = await isLoggedInWallapopTab(tab.id);

    if (!logged) {
      // No degradar de inmediato; mantenemos lo último bueno
      scheduleNoLogin('detect');
      return { ok: false, reason: 'PENDING_NOLOGIN' };
    }

    // 3) Logueado: pedir alias + avatar
    let info = null;
    try {
      // ✅ v10.5.82: Envolver para silenciar lastError
      info = await new Promise((resolve) => {
        chrome.tabs.sendMessage(tab.id, { type: 'DOM.CURRENT_PROFILE' }, (response) => {
          void chrome.runtime.lastError;
          resolve(response);
        });
      });
    } catch {}

    const rawAvatar = info?.avatar || '';
    const rawAlias = (info?.alias && info.alias.trim()) || '';
    const lastGood = LAST_GOOD;
    const nextAlias = rawAlias || 
                      (info?.profileUrl ? (aliasFromProfileUrl(info.profileUrl) || '') : '') || 
                      lastGood.alias || 
                      '';
    const nextAvatar = rawAvatar || lastGood.avatar || '';

    cancelNoLoginHold();
    set_LAST_GOOD({ alias: nextAlias, avatar: nextAvatar });
    paintConnected(nextAlias, nextAvatar);
    setBadgeTitle(nextAlias ? 'Detectado desde pestaña de Wallapop.' : 'Wallapop abierto y con sesión.');

    return { ok: true, alias: nextAlias };
  } catch {
    cancelNoLoginHold();
    paintNotDetected();
    return { ok: false, reason: 'ERROR' };
  }
}

/* ===============================
   Listeners de pestañas (eventos)
   =============================== */

let _initedListeners = false;

/** Inicializa listeners de pestañas (onUpdated, onRemoved) */
export function initTabListeners() {
  if (_initedListeners) return;
  _initedListeners = true;

  let debounce = null;

  const kick = (tab) => {
    clearTimeout(debounce);
    debounce = setTimeout(async () => {
      // 1) Watchdog por si no llega push
      if (CS_WATCHDOG) clearTimeout(CS_WATCHDOG);
      CS_WATCHDOG = setTimeout(() => {
        detectLoggedWalla().catch(err => { console.error(`[${filepath.split('/').pop()}] Error:`, err); });
        CS_WATCHDOG = null;
      }, 2000);

      // 2) Handshake: si el CS aún no está listo tras F5, no fuerza nada
      const ready = tab?.id ? await pingContentScript(tab.id, 350) : false;
      if (!ready) return;

      // 3) Si ya teníamos alias, evita spinner y confía en el push; si no, hace una detección
      const needSpinner = !LAST_GOOD.alias;
      if (needSpinner) setBadgeLoading(true);
      try {
        await detectLoggedWalla();
      } finally {
        if (needSpinner) setBadgeLoading(false);
        if (CS_WATCHDOG) {
          clearTimeout(CS_WATCHDOG);
          CS_WATCHDOG = null;
        }
      }
    }, 250);
  };

  // onUpdated: detectar navegación/carga en Wallapop
  chrome.tabs.onUpdated.addListener(async (tabId, info, tab) => {
    try {
      if (info.status === 'complete') {
        const host = new URL(tab?.url || '', 'https://x').hostname || '';
        const isWalla = /wallapop\.(com|es)$/i.test(host);

        if (isWalla) {
          kick(tab);
        } else {
          // Acabas de aterrizar en un dominio NO wallapop: ¿quedan pestañas wallapop?
          const tabs = await chrome.tabs.query({ 
            url: ['https://*.wallapop.com/*', 'https://*.wallapop.es/*'] 
          });
          if (!tabs.length) {
            cancelNoLoginHold();
            if (CS_WATCHDOG) {
              clearTimeout(CS_WATCHDOG);
              CS_WATCHDOG = null;
            }
            set_LAST_GOOD({ alias: '', avatar: '' });
            paintNotDetected();
          }
        }
      }
    } catch {}
  });

  // onRemoved: si cierras la última pestaña de Wallapop
  chrome.tabs.onRemoved.addListener(async () => {
    try {
      const tabs = await chrome.tabs.query({ 
        url: ['https://*.wallapop.com/*', 'https://*.wallapop.es/*'] 
      });
      if (!tabs.length) {
        cancelNoLoginHold();
        if (CS_WATCHDOG) {
          clearTimeout(CS_WATCHDOG);
          CS_WATCHDOG = null;
        }
        set_LAST_GOOD({ alias: '', avatar: '' });
        paintNotDetected();
      }
    } catch {}
  });
}

/** Init general del módulo */
export function init() {
  initTabListeners();
}