/**
 * ============================================================================
 * delete-handler.js - Eliminación masiva de anuncios
 * ============================================================================
 * 
 * RESPONSABILIDAD:
 * - Maneja la lógica completa de eliminación de anuncios
 * - Usa barra de progreso centralizada (UI.updateProgressBar)
 * - Coordina con window.LISTINGS (no import para evitar circular)
 * 
 * ARQUITECTURA:
 * - Import de UI para progreso (ui.js)
 * - window.LISTINGS para funciones de tabla
 * - logger para debugging
 * 
 * EXPORTS:
 * - handleDeleteSelected() - Única función exportada
 * 
 * NO DUPLICAR:
 * - Barra de progreso está en ui.js (reutilizar)
 * - Funciones de tabla están en listings/index.js (usar window.LISTINGS)
 */

import { deleteOne } from '../delete.js';
import { logger } from './logger.js';
import { toast, sleep } from '../utils.js';  // ✅ v6.4.4: Centralizar delays
import { getDbStats } from './state-manager.js';
// ✅ v10.5.82: selectAllByFilter movido a module-state.js
import { getSelectAllByFilter, clearSelectAllByFilter } from './listings/module-state.js';

// ---------------------------------------------------------
// SW helper
// ---------------------------------------------------------
async function sw(msg) {
  return new Promise((resolve) => {
    try {
      chrome.runtime.sendMessage(msg, (response) => {
        resolve(response || { ok: false });
      });
    } catch {
      resolve({ ok: false });
    }
  });
}

// Referencias DOM
let modal = null;


// ==============================================
// DELETE HANDLER ORIGINAL
// ==============================================

function initDOM() {
  if (modal) {
    return;
  }
  
  
  modal = document.getElementById('delete-confirm-modal');
  
  if (!modal) {
    logger.error('[DELETE] ❌ Modal delete-confirm-modal NO existe en HTML');
    return;
  }
  
  // Botones del modal
  const btnCancel = document.getElementById('btn-cancel-delete');
  const btnConfirm = document.getElementById('btn-confirm-delete');
  
  
  if (!btnCancel || !btnConfirm) {
    logger.error('[DELETE] ❌ Botones del modal no encontrados');
    return;
  }
  
  if (btnCancel) {
    btnCancel.addEventListener('click', hideModal);
  }
  
  if (btnConfirm) {
    btnConfirm.addEventListener('click', () => {
      hideModal();
      executeDelete();
    });
  }
  
  // Click fuera del modal para cerrar
  if (modal) {
    modal.addEventListener('click', (e) => {
      if (e.target === modal) {
        hideModal();
      }
    });
  }
  
  // v4.59.0: Listener para botón ELIMINAR
  const btnDeleteSelected = document.getElementById('btn-delete-selected');
  if (btnDeleteSelected) {
    btnDeleteSelected.addEventListener('click', handleDeleteSelected);
  }
  
  // v4.59.0: Listener para botón PUBLICAR (nuevo, separado)
  const btnPublishDeleted = document.getElementById('btn-publish-deleted');
  if (btnPublishDeleted) {
    btnPublishDeleted.addEventListener('click', handlePublishDeletedSelected);
  }
  
}

// ✅ v6.4.4: Renombrado para evitar conflicto con showModal de utils.js
function showDeleteConfirmModal(count, selectedIds) {
  
  initDOM();
  
  if (!modal) {
    logger.error('[DELETE] ❌ Modal es null, no se puede mostrar');
    return;
  }
  
  // 🔧 Guardar selectedIds en variable de módulo para que executeDelete la use
  window._DELETE_SELECTED_IDS = selectedIds;
  
  const modalCount = document.getElementById('modal-delete-count');
  if (modalCount) {
    modalCount.textContent = count;
  } else {
  }
  
  modal.style.display = 'flex';
}

function hideModal() {
  if (modal) {
    modal.style.display = 'none';
  }
}

async function executeDelete() {
  const { requireWallaActive } = await import('./auth.js');
  const { showModal: showAuthModal, showLoader, hideLoader } = await import('../utils.js');
  
  // 🔧 FIX: Leer selectedIds desde window
  const selectedIds = window._DELETE_SELECTED_IDS;
  
  if (!Array.isArray(selectedIds) || selectedIds.length === 0) {
    logger.error('[DELETE] selectedIds inválido o vacío');
    toast('Error: No hay anuncios seleccionados', 'error');
    return;
  }
  
  // 🔄 v4.46.0: LOADER con timeout de 30s
  showLoader('Verificando Wallapop...', { 
    timeout: 30000,
    onTimeout: () => {
      toast('⏱️ Tiempo agotado verificando Wallapop', 'error', 5000);
    }
  });
  
  const wallaCheck = await requireWallaActive();
  
  hideLoader();
  
  if (!wallaCheck.ok) {
    return;
  }
  
  
  // 🔄 LOADER mientras valida cuenta
  showLoader('Validando cuenta...', {
    timeout: 30000,
    onTimeout: () => toast('⏱️ Tiempo agotado validando cuenta', 'error', 5000)
  });
  
  // ✅ v4.44.0: Validación ROBUSTA de cuenta
  const selAccount = document.getElementById('sel-account');
  const selectedAccountId = selAccount?.value;
  const wallapopAlias = wallaCheck.wallapopAccount?.alias;
  
  
  // 🔒 VALIDACIÓN: Si tenemos alias de Wallapop, verificar
  if (wallapopAlias) {
    // Si hay cuenta seleccionada, comparar
    if (selectedAccountId) {
      const selectedOption = selAccount?.selectedOptions?.[0];
      const selectedAlias = selectedOption?.dataset?.alias || selectedOption?.textContent?.split(' ')[0];
      
      const normalizeAlias = (str) => String(str || '').trim().toLowerCase();
      
      if (normalizeAlias(selectedAlias) !== normalizeAlias(wallapopAlias)) {
        // ⚠️ MISMATCH
        hideLoader();
        
        await showAuthModal({
          title: '⚠️ No puedes continuar',
          html: `
            <p style="margin-bottom: 16px; line-height: 1.6;">
              <strong>Panel seleccionado:</strong> ${selectedAlias}<br>
              <strong>Wallapop abierto:</strong> ${wallapopAlias}
            </p>
            <p class="modal-text-error">
              ⚠️ <strong>Las cuentas no coinciden.</strong><br>
              No puedes eliminar anuncios de una cuenta diferente.
            </p>
            <p class="modal-text-muted">
              Para continuar, ve a Wallapop y:
            </p>
            <ul class="modal-list">
              <li>Cierra sesión</li>
              <li>Inicia sesión con: <strong>${selectedAlias}</strong></li>
              <li>Vuelve aquí e intenta de nuevo</li>
            </ul>
          `,
          buttons: [{ text: 'Entendido', value: 'ok', primary: true }]
        });
        
        toast('Cambia de cuenta en Wallapop', 'warning', 4000);
        return;
      }
    }
    // Si NO hay cuenta seleccionada pero SÍ hay alias de Wallapop
    // → Continuar (permitir uso sin cuenta seleccionada)
  } else {
    // NO hay alias de Wallapop → Advertir que puede fallar
    hideLoader();
    
    const choice = await showAuthModal({
      title: '⚠️ Advertencia',
      html: `
        <p style="line-height: 1.6; margin-bottom: 16px;">
          No se pudo verificar la cuenta activa en Wallapop.
        </p>
        <p class="modal-text-warning">
          Si estás logueado con una cuenta diferente,<br>
          la eliminación fallará.
        </p>
        <p class="modal-text-muted">
          ¿Quieres continuar de todas formas?
        </p>
      `,
      buttons: [
        { text: 'Cancelar', value: 'cancel' },
        { text: 'Continuar', value: 'continue', primary: true }
      ]
    });
    
    if (choice !== 'continue') {
      toast('Operación cancelada', 'info');
      return;
    }
    
    showLoader('Preparando eliminación...', {
      timeout: 30000,
      onTimeout: () => toast('⏱️ Tiempo agotado preparando eliminación', 'error', 5000)
    });
  }
  
  hideLoader();
  
  // Obtener funciones desde window.LISTINGS (evita import circular)
  if (!window.LISTINGS) {
    logger.error('[DELETE] window.LISTINGS no disponible');
    toast('Error: Sistema no inicializado', 'err');
    return;
  }
  
  const { getSelectedListingIds, getCurrentListings, clearSelection, refreshListings } = window.LISTINGS;
  
  // selectedIds ya está disponible desde window._DELETE_SELECTED_IDS (línea 141)
  
  if (selectedIds.length === 0) {
    toast('⚠️ No hay anuncios seleccionados', 'warning');
    return;
  }
  
  // Obtener datos completos de listings
  const allListings = getCurrentListings();
  const toDelete = allListings.filter(l => selectedIds.includes(l.listing_id));
  
  const total = toDelete.length;
  
  // onDeleteStart removed (was no-op)
  
  // ✅ Importar UI para barra de progreso
  const UI = await import('./ui.js');
  
  let completed = 0;
  let reallyDeleted = 0; // Anuncios que SÍ existían y se eliminaron
  let notExisted = 0;    // Anuncios que NO existían en Wallapop
  let failed = 0;
  
  const notExistedList = []; // Para el resumen
  const deletedList = [];
  const failedList = [];
  
  // Importar funciones de animación (import dinámico OK)
  const { moveRowToTop, setRowState, clearRowState } = await import('./listings/index.js');
  
  // ✅ v6.8.2: Obtener accountId para aislamiento de barra
  const currentAccountId = window.AUTH?.getActiveAccountId?.() || null;
  
  // ✅ Mostrar barra de progreso al inicio con accountId
  UI.updateProgressBar({
    operation: 'delete',
    current: 0,
    total: total,
    active: true,
    accountId: currentAccountId  // ✅ Aislar por cuenta
  });
  
  for (const listing of toDelete) {
    const listingId = String(listing.listing_id);
    
    // Mover fila arriba y marcar como ELIMINANDO (rojo)
    moveRowToTop(listingId);
    setRowState(listingId, 'is-deleting');
    
    // updateProgress removed (was no-op)
    
    // ⚡ v4.41.1: NO esperar antes de intentar (más rápido)
    
    // Eliminar
    try {
      const result = await deleteOne({ id: listing.id_wallapop });
      
      if (result.ok) {
        // Distinguir entre realmente eliminado vs ya no existía
        if (result.method === 'idempotent') {
          // Ya estaba eliminado (404/410)
          notExisted++;
          notExistedList.push({
            id: listing.id_wallapop,
            title: listing.title || 'Sin título'
          });
          setRowState(listingId, 'is-warning'); // Amarillo para "no existía"
          
          // ⚡ Si NO existía → limpiar inmediatamente (0ms)
          await sleep(100); // Solo 100ms visual
          clearRowState(listingId);
        } else {
          // Realmente eliminado
          reallyDeleted++;
          deletedList.push({
            id: listing.id_wallapop,
            title: listing.title || 'Sin título'
          });
          setRowState(listingId, 'is-success');
          
          // ✅ Si SÍ se eliminó → mostrar 1s para feedback visual
          await sleep(1000);
          clearRowState(listingId);
        }
      } else {
        // ❌ ERROR
        
        // 🔒 v4.44.0: Si es 401 → DETENER TODO inmediatamente
        if (result.http === 401 || result.error === 'not_authorized') {
          logger.error('[DELETE] 🚨 401 DETECTADO - Deteniendo operación');
          
          setRowState(listingId, 'is-error');
          
          // ✅ Ocultar barra de progreso al detener
          UI.updateProgressBar({ operation: null });
          
          // Detener progreso
          // onOperationComplete removed (was no-op)
          
          // Modal explicativa
          await showAuthModal({
            title: '🚨 Cuenta incorrecta',
            html: `
              <p style="margin-bottom: 16px; line-height: 1.6;">
                <strong class="text-error">Error 401: No autorizado</strong>
              </p>
              <p style="font-size: 14px; line-height: 1.6; margin-bottom: 12px;">
                Estás intentando eliminar anuncios de una cuenta diferente a la que tienes abierta en Wallapop.
              </p>
              <p class="modal-text-muted">
                <strong>Procesados:</strong> ${completed} de ${total}<br>
                <strong class="text-success">✅ Eliminados:</strong> ${reallyDeleted}<br>
                <strong class="text-warning">⚠️ No existían:</strong> ${notExisted}<br>
                <strong class="text-error">❌ Fallaron:</strong> ${failed + 1}
              </p>
              <p class="modal-text-muted">
                Cambia de cuenta en Wallapop y vuelve a intentarlo.
              </p>
            `,
            buttons: [{ text: 'Entendido', value: 'ok', primary: true }]
          });
          
          toast('❌ Operación detenida: cuenta incorrecta', 'error', 5000);
          
          // Refrescar tabla
          await refreshListings();
          
          return; // 🛑 DETENER LOOP
        }
        
        // Otros errores (no 401)
        failed++;
        failedList.push({
          id: listing.id_wallapop,
          title: listing.title || 'Sin título',
          error: result.error || 'unknown'
        });
        setRowState(listingId, 'is-error');
        logger.error(`[DELETE] ❌ Error: ${listing.id_wallapop}`, result.error);
        
        // Mostrar error por 2 segundos
        await sleep(2000);
        clearRowState(listingId);
      }
    } catch (error) {
      failed++;
      failedList.push({
        id: listing.id_wallapop,
        title: listing.title || 'Sin título',
        error: String(error)
      });
      setRowState(listingId, 'is-error');
      logger.error(`[DELETE] ❌ Exception: ${listing.id_wallapop}`, error);
      
      // Mostrar error por 2 segundos
      await sleep(2000);
      clearRowState(listingId);
    }
    
    completed++;
    
    // ✅ v6.8.2: Actualizar barra de progreso con accountId
    UI.updateProgressBar({
      operation: 'delete',
      current: completed,
      total: total,
      active: true,
      accountId: currentAccountId  // ✅ Aislar por cuenta
    });
  }
  
  // updateProgress removed (was no-op)
  
  // Esperar 1s para que el usuario vea el 100%
  await sleep(1000);
  
  // ✅ Ocultar barra de progreso
  UI.updateProgressBar({ operation: null });
  
  // onOperationComplete removed (was no-op)
  
  clearSelection();
  
  // Refrescar tabla
  await refreshListings();
  
  // ✅ MODAL RESUMEN (solo si hay algo que reportar)
  if (notExisted > 0 || failed > 0) {
    
    try {
      await showDeleteSummary({
        total,
        reallyDeleted,
        notExisted,
        failed,
        notExistedList,
        failedList
      });
    } catch (error) {
      logger.error('[DELETE] Error mostrando resumen:', error);
      // Fallback: toast con resumen
      toast(`✅ ${reallyDeleted} eliminados | ⚠️ ${notExisted} no existían | ❌ ${failed} errores`, 'warning', 6000);
    }
  } else {
    // Todo OK, toast simple
    toast(`✅ ${reallyDeleted} anuncio(s) eliminado(s) correctamente`, 'success', 4000);
  }
}

export async function handleDeleteSelected() {
  
  // Obtener funciones de LISTINGS desde window (evita import circular)
  if (!window.LISTINGS) {
    logger.error('[DELETE] window.LISTINGS no disponible');
    toast('Error: Sistema no inicializado', 'err');
    return;
  }
  
  // ✅ v6.0.0: Verificar si está activo "seleccionar todos por filtro"
  const selectAllByFilter = getSelectAllByFilter();
  const dbStats = getDbStats();
  
  let selectedIds = [];
  let totalToDelete = 0;
  
  if (selectAllByFilter) {
    // Modo "seleccionar todos": obtener TODOS los IDs del backend
    totalToDelete = dbStats.active || 0;
    
    if (totalToDelete === 0) {
      toast('❌ No hay anuncios activos para eliminar', 'warn');
      clearSelectAllByFilter();
      return;
    }
    
    // ✅ Obtener todos los IDs activos desde el backend
    try {
      const accountId = window.AUTH?.getActiveAccountId?.();
      if (!accountId) {
        toast('❌ No hay cuenta activa', 'warn');
        clearSelectAllByFilter();
        return;
      }
      
      toast('🔄 Obteniendo lista completa de anuncios...', 'info');
      
      const response = await sw({
        type: 'API.FETCH_JSON',
        url: `/api/walla/publish2/active-ids/${accountId}`,
        method: 'GET'
      });
      
      if (!response?.ok || !response.data?.ok) {
        throw new Error(response?.error || 'Error obteniendo IDs');
      }
      
      // Convertir a formato esperado por el sistema de eliminación
      const items = response.data.items || [];
      selectedIds = items.map(item => item.listing_id);
      totalToDelete = selectedIds.length;
      
      
    } catch (err) {
      logger.error('[DELETE] Error obteniendo IDs del backend:', err);
      toast('❌ Error obteniendo lista de anuncios', 'error');
      clearSelectAllByFilter();
      return;
    }
  } else {
    // Modo normal: obtener IDs de checkboxes seleccionados
    const { getSelectedListingIds } = window.LISTINGS;
    selectedIds = getSelectedListingIds();
    
    // 🔧 FIX: Convertir Set a Array si es necesario
    if (selectedIds instanceof Set) {
      selectedIds = Array.from(selectedIds);
    }
    
    
    if (selectedIds.length === 0) {
      toast('⚠️ Selecciona al menos un anuncio para eliminar', 'warning');
      return;
    }
    
    totalToDelete = selectedIds.length;
  }
  
  // ✅ Mostrar modal de confirmación directamente (sin modal intermedia)
  showDeleteConfirmModal(totalToDelete, selectedIds);
  
  // ✅ v6.0.0: Limpiar flag después de usar
  if (selectAllByFilter) {
    clearSelectAllByFilter();
  }
}

/**
 * Mostrar modal resumen de eliminación
 */
async function showDeleteSummary({ total, reallyDeleted, notExisted, failed, notExistedList, failedList }) {
  
  try {
    const { showModal: showAuthModal } = await import('../utils.js');
    
    let html = `<div style="text-align: left; font-size: 14px;">`;
    
    // Resumen general
    html += `<p style="margin-bottom: 16px;">
      <strong>Total procesados:</strong> ${total}<br>
      <strong class="text-success">✅ Eliminados:</strong> ${reallyDeleted}<br>`;
    
    if (notExisted > 0) {
      html += `<strong class="text-warning">⚠️ No existían:</strong> ${notExisted}<br>`;
    }
    
    if (failed > 0) {
      html += `<strong class="text-error">❌ Errores:</strong> ${failed}`;
    }
    
    html += `</p>`;
    
    // Detalles de los que NO existían
    if (notExisted > 0 && notExistedList.length > 0) {
      html += `<details style="margin-top: 12px; cursor: pointer;">
        <summary style="color: #f59e0b; font-weight: 600;">⚠️ Anuncios que NO existían en Wallapop (${notExisted})</summary>
        <ul style="margin: 8px 0; padding-left: 20px; max-height: 150px; overflow-y: auto;">`;
      
      notExistedList.forEach(item => {
        html += `<li style="margin: 4px 0; font-size: 13px;">
          <code>${item.id}</code> - ${item.title.substring(0, 50)}${item.title.length > 50 ? '...' : ''}
        </li>`;
      });
      
      html += `</ul></details>`;
    }
    
    // Detalles de errores
    if (failed > 0 && failedList.length > 0) {
      html += `<details style="margin-top: 12px; cursor: pointer;">
        <summary style="color: #ef4444; font-weight: 600;">❌ Errores al eliminar (${failed})</summary>
        <ul style="margin: 8px 0; padding-left: 20px; max-height: 150px; overflow-y: auto;">`;
      
      failedList.forEach(item => {
        html += `<li style="margin: 4px 0; font-size: 13px;">
          <code>${item.id}</code> - ${item.error}
        </li>`;
      });
      
      html += `</ul></details>`;
    }
    
    html += `</div>`;
    
    
    await showAuthModal({
      title: '📊 Resumen de Eliminación',
      html,
      buttons: [
        { text: 'Cerrar', value: 'ok', primary: true }
      ]
    });
    
  } catch (error) {
    logger.error('[DELETE] Error en showDeleteSummary:', error);
    throw error;
  }
}

// Inicializar al cargar
if (typeof document !== 'undefined') {
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initDOM);
  } else {
    initDOM();
  }
}
