/*  Cabecera obligatoria: NO BORRAR NI MODIFICAR este bloque inicial en ningún fichero.
    Archivo: scripts/panel/delay-slider.js — Rol: Componente centralizado para slider de delay entre publicaciones
    Descripcion de todo el proyecto: 
         MitikLive Wallapop (MV3): panel lateral en Wallapop para gestionar anuncios conectado a FastAPI 
            (login JWT, export/import, backups/renovaciones).  
    Aviso: NO BORRAR los comentarios descriptivos situados encima de cada función, solo cambiarlos si se modifica la funcion.
    En Rol (linea mas arriba): si está vacía o el fichero se modifica o reestructura, modificar esa linea de rol 
    
    ✅ v10.5.34: Componente creado para centralizar la lógica del slider de delay
    - Elimina duplicación entre config.js y progress-tab.js
    - Sincroniza ambos sliders en tiempo real via Bus
    - Maneja cancelación de alarms y guardado en BD
*/

import { toast } from '../utils.js';
import { logger } from './logger.js';
import { Bus } from './bus.js';
import * as ButtonManager from './publish-button-manager.js'; // ✅ v1.4.0: Para verificar publicación activa

// =============================================================================
// ESTADO INTERNO
// =============================================================================

/** Valor actual del delay en segundos */
let currentDelay = 30;

/** Flag para evitar loops infinitos de sincronización */
let isSyncing = false;

/** Timeout para debounce del guardado */
let saveTimeout = null;

/** Sliders registrados para sincronización */
const registeredSliders = new Map();

// =============================================================================
// EVENTO DEL BUS
// =============================================================================

/** Nombre del evento para sincronización entre sliders */
const DELAY_CHANGED_EVENT = 'delay:changed';

// =============================================================================
// FUNCIONES PÚBLICAS
// =============================================================================

/**
 * Inicializa un slider de delay
 * Puede ser llamado múltiples veces para diferentes sliders (Config, Progress)
 * 
 * @param {Object} options - Opciones de configuración
 * @param {string} options.sliderId - ID del input range
 * @param {string} options.displayId - ID del elemento que muestra el valor
 * @param {string} [options.messageId] - ID del mensaje de confirmación (opcional)
 * @param {boolean} [options.showToast=false] - Mostrar toast al guardar
 * @param {boolean} [options.autoSave=true] - Guardar automáticamente al cambiar
 * @returns {Function} Función de cleanup para remover listeners
 */
export function initDelaySlider(options) {
  const {
    sliderId,
    displayId,
    messageId = null,
    showToast = false,
    autoSave = true
  } = options;

  const slider = document.getElementById(sliderId);
  const display = document.getElementById(displayId);
  const message = messageId ? document.getElementById(messageId) : null;

  if (!slider || !display) {
    logger.warn(`[DelaySlider] No se encontró slider (${sliderId}) o display (${displayId})`);
    return () => {};
  }

  // Registrar slider para sincronización
  registeredSliders.set(sliderId, { slider, display, message });

  // Aplicar valor actual
  slider.value = currentDelay;
  display.textContent = `${currentDelay}s`;

  // Handler para actualización inmediata del display
  const handleInput = (e) => {
    const value = parseInt(e.target.value, 10);
    display.textContent = `${value}s`;
  };

  // Handler para guardar cambios (al soltar el slider)
  const handleChange = async (e) => {
    const value = parseInt(e.target.value, 10);
    
    if (autoSave) {
      await saveDelay(value, { showToast, messageEl: message });
    }
  };

  // Añadir listeners
  slider.addEventListener('input', handleInput);
  slider.addEventListener('change', handleChange);

  // Suscribirse al Bus para sincronización
  const unsubscribe = Bus.on(DELAY_CHANGED_EVENT, (payload) => {
    if (isSyncing) return;
    
    // Actualizar este slider si el cambio vino de otro
    if (payload.source !== sliderId) {
      slider.value = payload.value;
      display.textContent = `${payload.value}s`;
    }
  });

  // Retornar función de cleanup
  return () => {
    slider.removeEventListener('input', handleInput);
    slider.removeEventListener('change', handleChange);
    unsubscribe();
    registeredSliders.delete(sliderId);
  };
}

/**
 * Carga el delay actual desde la BD
 * Debe llamarse después del login
 * 
 * @returns {Promise<number>} El delay actual en segundos
 */
export async function loadDelayFromDB() {
  try {
    const response = await chrome.runtime.sendMessage({
      type: 'API.FETCH_JSON',
      url: '/api/user/preferences',
      method: 'GET'
    });

    if (response?.ok && response.data) {
      const preferences = response.data.ui_preferences || {};
      currentDelay = preferences.publish_delay_seconds ?? 30;
    } else {
      currentDelay = 30;
    }

    // Actualizar todos los sliders registrados
    syncAllSliders(currentDelay, null);

    return currentDelay;
  } catch (error) {
    logger.error('[DelaySlider] Error cargando delay:', error);
    return currentDelay;
  }
}

/**
 * Guarda el delay en la BD con debounce
 * 
 * @param {number} delaySeconds - Nuevo valor en segundos
 * @param {Object} [options] - Opciones
 * @param {boolean} [options.showToast=false] - Mostrar toast de confirmación
 * @param {HTMLElement} [options.messageEl=null] - Elemento para mostrar "✓"
 * @returns {Promise<boolean>} true si se guardó correctamente
 */
export async function saveDelay(delaySeconds, options = {}) {
  const { showToast: shouldShowToast = false, messageEl = null } = options;

  // Actualizar valor local inmediatamente
  currentDelay = delaySeconds;

  // Sincronizar todos los sliders
  syncAllSliders(delaySeconds, null);

  // Limpiar timeout anterior (debounce)
  if (saveTimeout) {
    clearTimeout(saveTimeout);
  }

  // Guardar con debounce de 500ms
  return new Promise((resolve) => {
    saveTimeout = setTimeout(async () => {
      try {
        const response = await chrome.runtime.sendMessage({
          type: 'API.FETCH_JSON',
          url: '/api/user/preferences',
          method: 'PATCH',
          body: {
            ui_preferences: {
              publish_delay_seconds: delaySeconds
            },
            extension_version: chrome.runtime.getManifest().version
          }
        });

        if (response?.ok) {
          // ✅ v1.4.0: NO cancelar alarms si hay publicación activa
          // Esto evita que el cambio de delay interrumpa la publicación en curso
          const bmState = ButtonManager.getState();
          if (bmState.processingCount === 0) {
            await cancelPublishAlarms();
          } else {
            logger.debug('[DelaySlider] No se cancelan alarms - publicación activa');
          }

          // Mostrar confirmación
          if (shouldShowToast) {
            toast('✅ Configuración guardada', 'ok');
          }
          
          if (messageEl) {
            messageEl.classList.remove('hidden');
            setTimeout(() => messageEl.classList.add('hidden'), 2000);
          }

          resolve(true);
        } else {
          logger.error('[DelaySlider] Error guardando:', response?.error);
          if (shouldShowToast) {
            toast('❌ Error guardando configuración', 'error');
          }
          resolve(false);
        }
      } catch (error) {
        logger.error('[DelaySlider] Error guardando delay:', error);
        if (shouldShowToast) {
          toast('❌ Error guardando configuración', 'error');
        }
        resolve(false);
      }
    }, 500);
  });
}

/**
 * Obtiene el delay actual
 * 
 * @returns {number} Delay actual en segundos
 */
export function getCurrentDelay() {
  return currentDelay;
}

/**
 * Establece el delay sin guardar en BD (para uso interno)
 * 
 * @param {number} value - Nuevo valor
 */
export function setCurrentDelay(value) {
  currentDelay = value;
  syncAllSliders(value, null);
}

// =============================================================================
// FUNCIONES INTERNAS
// =============================================================================

/**
 * Sincroniza todos los sliders registrados con un nuevo valor
 * 
 * @param {number} value - Nuevo valor
 * @param {string|null} sourceId - ID del slider origen (para evitar loops)
 */
function syncAllSliders(value, sourceId) {
  isSyncing = true;

  // Actualizar sliders via DOM
  for (const [sliderId, elements] of registeredSliders) {
    if (sliderId !== sourceId) {
      elements.slider.value = value;
      elements.display.textContent = `${value}s`;
    }
  }

  // Emitir evento via Bus para sliders que no estén en el Map
  Bus.emit(DELAY_CHANGED_EVENT, { value, source: sourceId });

  isSyncing = false;
}

/**
 * Cancela todas las alarms de publicación activas
 * Necesario cuando se cambia el delay para evitar que sigan con el delay anterior
 */
async function cancelPublishAlarms() {
  try {
    const allAlarms = await chrome.alarms.getAll();
    const publishAlarms = allAlarms.filter(a => a.name.startsWith('publish-next-'));

    if (publishAlarms.length > 0) {
      for (const alarm of publishAlarms) {
        await chrome.alarms.clear(alarm.name);
      }
      logger.debug(`[DelaySlider] Canceladas ${publishAlarms.length} alarms de publicación`);
    }
  } catch (err) {
    logger.error('[DelaySlider] Error cancelando alarms:', err);
  }
}
