/**
 * credits.js
 * Sistema de gestión de créditos y billing
 * Integración con el backend de MitikLive
 * 
 * ✅ v10.5.33: API_BASE ahora se obtiene de chrome.storage (centralizado)
 */

import * as ButtonManager from './publish-button-manager.js';
import { logger } from './logger.js';
import { sleep } from '../utils.js';  // ✅ v6.4.5: Centralizar delays

// ✅ v6.5.0: Usa logger centralizado de logger.js (eliminado logger local duplicado)

const CreditsManager = (() => {
  // Configuración - API_BASE se carga dinámicamente desde storage
  const CONFIG = {
    API_BASE: 'https://www.mitiklive.com/fa',  // Fallback si storage vacío
    // ✅ v4.65.2: Eliminado polling - Solo WebSocket + actualización manual
    CACHE_DURATION: 30 * 1000,       // 30 segundos (cache corto para datos en tiempo real)
    WARNING_THRESHOLD: 10,            // Mostrar warning si <= 10 créditos
    DANGER_THRESHOLD: 0               // Mostrar danger si = 0 créditos
  };
  
  /**
   * ✅ v10.5.33: Carga API_BASE desde chrome.storage (fuente única de verdad)
   * Si no existe aún (primer arranque), usa fallback silenciosamente
   */
  async function loadApiBase() {
    try {
      const result = await chrome.storage.local.get(['cfg.API_BASE']);
      if (result['cfg.API_BASE']) {
        CONFIG.API_BASE = result['cfg.API_BASE'];
      }
      // Si no existe, usa fallback silenciosamente (es normal en primer arranque)
    } catch (err) {
      // Error silencioso - usa fallback hardcodeado
    }
  }

  // Estado
  let state = {
    credits: null,
    plan: null,
    publications_used: 0,
    publications_limit: 0,
    is_unlimited: false,  // 🆕 Añadir is_unlimited
    lastUpdate: null,
    isUpdating: false
  };

  // Elementos del DOM
  let elements = {
    container: null,
    count: null,
    tooltip: null
  };

  // ✅ v4.65.2: Eliminado updateTimer - No más polling

  /**
   * Inicializar el módulo
   */
  async function init() {
    
    // ✅ v10.5.33: Cargar API_BASE desde storage primero
    await loadApiBase();
    
    // ✅ Limpiar estado previo por si acaso
    state = {
      credits: null,
      plan: null,
      publications_used: 0,
      publications_limit: 0,
      is_unlimited: false,
      lastUpdate: null,
      isUpdating: false
    };
    
    // Obtener elementos del DOM
    elements.container = document.getElementById('ml-credits-display');
    elements.count = document.getElementById('credit-count');
    
    if (!elements.container || !elements.count) {
      logger.error('[Credits] No se encontraron elementos del DOM');
      return;
    }

    // 🆕 FORZAR visibilidad del contador desde el inicio
    elements.container.style.display = 'flex';
    elements.container.style.visibility = 'visible';
    elements.count.textContent = '...'; // Mostrar "cargando"

    // Event listeners
    elements.container.addEventListener('click', handleClick);
    
    // Verificar si hay usuario logueado con reintentos
    let userId = null;
    const MAX_USER_RETRIES = 3;
    
    for (let i = 0; i < MAX_USER_RETRIES; i++) {
      userId = await getUserId();
      if (userId) break;
      
      await sleep(500);
    }
    
    if (!userId) {
      setError('No logueado');
      return;
    }
    
    
    // Cargar créditos inicial (FORZADO - no usar cache)
    await fetchCredits(true);
    
    // ✅ v4.65.2: NO programar polling automático
    
  }

  /**
   * Obtener créditos del servidor
   */
  async function fetchCredits(force = false) {
    // Si ya está actualizando, no hacer nada
    if (state.isUpdating && !force) {
      return;
    }

    // Si hay cache reciente y no es forzado, usar cache
    if (!force && state.lastUpdate && (Date.now() - state.lastUpdate < CONFIG.CACHE_DURATION)) {
      return state;
    }

    state.isUpdating = true;
    setLoading(true);

    try {
      // Obtener user_id de la sesión
      const userId = await getUserId();
      
      if (!userId) {
        logger.error('[Credits] ❌ No se pudo obtener user_id');
        setError('No logueado');
        return null;
      }


      // Obtener token de autenticación desde service worker
      const tokenResponse = await chrome.runtime.sendMessage({ 
        type: 'AUTH.GET_TOKEN' 
      });
      
      if (!tokenResponse?.ok || !tokenResponse?.data?.accessToken) {
        logger.error('[Credits] ❌ No hay token de autenticación');
        setError('No autenticado');
        return null;
      }
      
      const token = tokenResponse.data.accessToken;

      // Llamar a la API con token
      const url = `${CONFIG.API_BASE}/api/billing/check-credits?user_id=${userId}`;
      
      const response = await fetch(url, {
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json',
          'X-Extension-Version': chrome.runtime.getManifest().version
        }
      });
      
      
      if (!response.ok) {
        const errorText = await response.text().catch(() => 'Sin detalles');
        logger.error('[Credits] ❌ Error HTTP:', response.status, errorText);
        throw new Error(`HTTP ${response.status}: ${response.statusText}`);
      }

      const data = await response.json();
      
      // Actualizar estado
      state.credits = data.credits_available || 0;
      state.plan = data.plan_name || 'Free';
      state.publications_used = data.publications_used || 0;
      state.publications_limit = data.publications_limit || 0;
      state.is_unlimited = data.is_unlimited || false;
      state.lastUpdate = Date.now();

      // ✅ Informar créditos al ButtonManager
      ButtonManager.updateState({
        availableCredits: state.credits
      });

      // Actualizar UI
      updateUI();

      
      return state;

    } catch (error) {
      logger.error('[Credits] ❌ Error al obtener créditos:', error);
      setError('Error');
      return null;
    } finally {
      state.isUpdating = false;
      setLoading(false);
    }
  }

  /**
   * Descontar un crédito después de publicar
   */
  async function deductCredit(wallaAccountId = null, listingId = null) {
    try {
      const userId = await getUserId();
      
      if (!userId) {
        logger.error('[Credits] No se pudo obtener user_id para deducción');
        return false;
      }

      const response = await fetch(`${CONFIG.API_BASE}/api/billing/deduct-credit`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'X-Extension-Version': chrome.runtime.getManifest().version
        },
        body: JSON.stringify({
          user_id: userId,
          walla_account_id: wallaAccountId,
          listing_id: listingId
        })
      });

      if (!response.ok) {
        throw new Error(`HTTP ${response.status}`);
      }

      const data = await response.json();
      
      if (data.success) {
        // Actualizar estado local
        if (state.credits !== null && state.credits > 0) {
          state.credits--;
        }
        if (state.publications_used !== null) {
          state.publications_used++;
        }
        
        // Actualizar UI
        updateUI();
        
        return true;
      }

      return false;

    } catch (error) {
      logger.error('[Credits] Error al deducir crédito:', error);
      return false;
    }
  }

  /**
   * Verificar si hay créditos disponibles
   */
  function hasCredits() {
    if (state.is_unlimited) {
      return true; // Plan ilimitado
    }
    
    if (state.publications_used < state.publications_limit) {
      return true; // Tiene publicaciones del plan
    }
    
    return state.credits > 0; // Tiene créditos extra
  }

  /**
   * Obtener user_id desde el JWT access token
   */
  async function getUserId() {
    return new Promise((resolve) => {
      // 🔧 CLAVES CORRECTAS: session.access, session.user (no jwt.access, user)
      chrome.storage.local.get(['session.access', 'session.user', 'jwt.access', 'user'], (data) => {
        // Opción 1: Desde session.user (arquitectura actual)
        if (data['session.user'] && typeof data['session.user'] === 'object' && data['session.user'].id) {
          const userId = parseInt(data['session.user'].id);
          resolve(userId);
          return;
        }
        
        // Opción 2: Desde user (legacy/fallback)
        if (data.user && typeof data.user === 'object' && data.user.id) {
          const userId = parseInt(data.user.id);
          resolve(userId);
          return;
        }
        
        // Opción 3: Decodificar JWT session.access (arquitectura actual)
        let jwt = data['session.access'];
        
        // Opción 4: Fallback a jwt.access (legacy)
        if (!jwt) {
          jwt = data['jwt.access'];
        }
        
        if (jwt) {
          try {
            // Decodificar JWT (payload es la segunda parte)
            const parts = jwt.split('.');
            if (parts.length === 3) {
              const payload = JSON.parse(atob(parts[1]));
              
              // El user_id puede estar en 'sub', 'user_id' o 'id'
              const userIdStr = payload.sub || payload.user_id || payload.id;
              
              if (userIdStr) {
                // Convertir a número (puede venir como string "1")
                const userId = parseInt(userIdStr);
                
                if (!isNaN(userId) && userId > 0) {
                  resolve(userId);
                  return;
                }
              }
              
              logger.error('[Credits] JWT payload sin user_id válido:', payload);
            }
          } catch (e) {
            logger.error('[Credits] Error decodificando JWT:', e);
          }
        }
        
        logger.error('[Credits] No se pudo encontrar user_id en storage');
        resolve(null);
      });
    });
  }

  /**
   * Actualizar UI con los datos actuales
   */
  function updateUI() {
    
    // ✅ v6.5.1: Re-buscar elementos si no existen (lazy init)
    if (!elements.count) {
      elements.count = document.getElementById('credit-count');
    }
    if (!elements.container) {
      elements.container = document.getElementById('ml-credits-display');
    }
    
    // ✅ v10.5.144: Buscar también el wrapper container y el count de config
    const creditsWrapper = document.getElementById('ml-credits-container');
    const configCreditCount = document.getElementById('config-credit-count');
    
    if (!elements.count) {
      return;
    }
    
    if (!elements.container) {
      return;
    }

    // ✅ v10.5.144: Mostrar el wrapper
    if (creditsWrapper) {
      creditsWrapper.style.display = 'flex';
    }

    // Actualizar contador según el tipo de plan
    if (state.is_unlimited) {
      // Plan ilimitado → Mostrar "Ilimitado"
      elements.count.textContent = 'Ilimitado';
      if (configCreditCount) configCreditCount.textContent = '∞';
      
      // ✅ v4.0.5: Actualizar precio en euros
      const configCreditPrice = document.getElementById('config-credit-price');
      if (configCreditPrice) configCreditPrice.textContent = '—';
      
      elements.container.classList.remove('warning', 'danger', 'no-credits');
    } else if (state.plan === 'pay_as_you_go' || state.publications_limit === 0) {
      // Sistema sin planes, solo créditos (pay as you go)
      const creditCount = state.credits || 0;
      elements.count.textContent = `${creditCount}`;
      if (configCreditCount) configCreditCount.textContent = `${creditCount}`;
      
      // ✅ v4.0.5: Calcular y mostrar precio en euros (0.05€ por crédito)
      const configCreditPrice = document.getElementById('config-credit-price');
      if (configCreditPrice) {
        const priceInEuros = (creditCount * 0.05).toFixed(2);
        configCreditPrice.textContent = `${priceInEuros}€`;
      }
      
      // ✅ v10.5.144: Actualizar clases - usar no-credits en lugar de danger
      elements.container.classList.remove('warning', 'danger', 'no-credits');
      if (state.credits === 0) {
        elements.container.classList.add('no-credits');
      } else if (state.credits <= CONFIG.WARNING_THRESHOLD) {
        elements.container.classList.add('warning');
      }
    } else {
      // Sistema con planes (legacy)
      const remaining = Math.max(0, state.publications_limit - state.publications_used);
      
      
      if (remaining > 0) {
        // Tiene publicaciones del plan
        if (state.credits > 0) {
          // Mostrar ambos: publicaciones del plan + créditos extra
          elements.count.textContent = `${remaining} +${state.credits}`;
          if (configCreditCount) configCreditCount.textContent = `${remaining} +${state.credits}`;
        } else {
          // Solo publicaciones del plan
          elements.count.textContent = `${remaining}`;
          if (configCreditCount) configCreditCount.textContent = `${remaining}`;
        }
      } else {
        // Solo tiene créditos extra (o ninguno)
        const creditCount = state.credits || 0;
        elements.count.textContent = `${creditCount}`;
        if (configCreditCount) configCreditCount.textContent = `${creditCount}`;
      }

      // ✅ v10.5.144: Actualizar clases - usar no-credits en lugar de danger
      elements.container.classList.remove('warning', 'danger', 'no-credits');
      
      if (remaining === 0 && state.credits === 0) {
        elements.container.classList.add('no-credits');
      } else if (remaining === 0 && state.credits <= CONFIG.WARNING_THRESHOLD) {
        elements.container.classList.add('warning');
      }
    }

    // Actualizar tooltip
    updateTooltip();
  }

  /**
   * Actualizar tooltip con detalles
   */
  function updateTooltip() {
    if (!elements.container) return;
    
    const planLabel = state.plan || 'Pay as you go';
    const creditsAvailable = state.credits || 0;
    
    // Sistema sin planes (solo créditos)
    if (state.plan === 'pay_as_you_go' || state.publications_limit === 0) {
      elements.container.title = 
        `Créditos: ${creditsAvailable}\n` +
        `1 crédito = 1 publicación\n` +
        `\nClick para actualizar`;
      return;
    }
    
    // Sistema con planes (legacy)
    const pubUsed = state.publications_used || 0;
    const pubLimit = state.publications_limit || 0;
    
    let publicationsText;
    
    if (state.is_unlimited) {
      // Plan ilimitado
      publicationsText = 'Ilimitadas';
    } else {
      // Plan con límite
      const remaining = Math.max(0, pubLimit - pubUsed);
      if (remaining > 0) {
        publicationsText = `${remaining}/${pubLimit} disponibles`;
      } else {
        publicationsText = `${pubUsed}/${pubLimit} (agotadas)`;
      }
    }

    elements.container.title = 
      `Plan: ${planLabel}\n` +
      `Publicaciones: ${publicationsText}\n` +
      `Créditos extra: ${creditsAvailable}\n` +
      `\nClick para actualizar`;
  }

  /**
   * Establecer estado de carga
   */
  function setLoading(loading) {
    if (!elements.container) return;
    
    if (loading) {
      elements.container.classList.add('loading');
      elements.count.textContent = '';
    } else {
      elements.container.classList.remove('loading');
    }
  }

  /**
   * Establecer estado de error
   */
  function setError(message) {
    if (!elements.count) return;
    elements.count.textContent = message || '!';
    elements.container.classList.add('danger');
  }

  /**
   * Manejar click en el contador
   */
  function handleClick(e) {
    e.preventDefault();
    
    // ✅ Feedback visual durante actualización
    if (elements.container) {
      elements.container.style.opacity = '0.6';
      elements.container.style.transform = 'scale(0.95)';
    }
    
    fetchCredits(true).finally(() => {
      // Restaurar visual
      if (elements.container) {
        elements.container.style.opacity = '1';
        elements.container.style.transform = 'scale(1)';
      }
    });
  }

  /**
   * ✅ v4.65.2: Función eliminada - No más polling automático
   * Los créditos se actualizan:
   * - Al hacer click en el contador
   * - Al refrescar token
   * - Por eventos WebSocket (cuando el backend lo implemente)
   */

  /**
   * Abrir página de recarga
   */
  function openRechargePage() {
    chrome.tabs.create({
      url: `${CONFIG.API_BASE}/dashboard/billing`
    });
  }

  /**
   * Obtener estado actual
   */
  function getState() {
    return { ...state };
  }

  /**
   * Mostrar contador
   */
  function show() {
    if (elements.container) {
      elements.container.style.display = 'flex';
    }
  }

  /**
   * Ocultar contador
   */
  function hide() {
    if (elements.container) {
      elements.container.style.display = 'none';
    }
  }

  /**
   * Limpiar timers
   */
  function destroy() {
    
    // Limpiar timer
    if (updateTimer) {
      clearInterval(updateTimer);
      updateTimer = null;
    }
    
    // ✅ CRÍTICO: Limpiar estado interno completamente
    state = {
      credits: null,
      plan: null,
      publications_used: 0,
      publications_limit: 0,
      is_unlimited: false,
      lastUpdate: null,
      isUpdating: false
    };
    
    // Ocultar UI
    hide();
    
  }

  // API pública
  return {
    init,
    fetchCredits,
    deductCredit,
    hasCredits,
    openRechargePage,
    getState,
    show,
    hide,
    destroy
  };
})();

// Exportar para uso global
window.CreditsManager = CreditsManager;

// NO auto-inicializar - se inicializará desde auth.js después del login exitoso
