/*  Cabecera obligatoria: NO BORRAR NI MODIFICAR este bloque inicial en ningún fichero.
    Archivo: scripts/panel/config.js — Rol: Gestión de configuración de usuario
    Descripcion de todo el proyecto: 
         MitikLive Wallapop (MV3): panel lateral en Wallapop para gestionar anuncios conectado a FastAPI 
            (login JWT, export/import, backups/renovaciones).  
    Aviso: NO BORRAR los comentarios descriptivos situados encima de cada función, solo cambiarlos si se modifica la funcion.
    En Rol (linea mas arriba): si está vacía o el fichero se modifica o reestructura, modificar esa linea de rol 
    
    ✅ v10.5.34: Refactorizado para usar componente centralizado delay-slider.js
    ✅ v10.5.117: Añadido botón para limpiar procesos colgados
*/

import { logger } from './logger.js';
import { 
  initDelaySlider, 
  loadDelayFromDB, 
  getCurrentDelay as getDelayFromSlider 
} from './delay-slider.js';
import { toast } from '../utils.js';
import { getActiveAccountId } from './auth.js';

/** Cleanup function para el slider */
let sliderCleanup = null;

/**
 * Inicializa el panel de configuración
 * ✅ v10.5.34: Usa componente centralizado para el slider de delay
 * ✅ v10.5.117: Añadido botón para limpiar procesos colgados
 */
export async function initConfig() {
  const debugToggle = document.getElementById('debug-mode-toggle');

  // Cargar estado actual del debug mode
  if (debugToggle) {
    const result = await chrome.storage.local.get(['debugMode']);
    debugToggle.checked = result.debugMode || false;
    
    // ✅ El listener está en panel.js (initDebugToggle)
    // No duplicar aquí para evitar toasts dobles
  }

  // ✅ v10.5.34: Inicializar slider con componente centralizado
  // El slider ahora se guarda automáticamente al cambiar (sin botón)
  sliderCleanup = initDelaySlider({
    sliderId: 'publish-delay-slider',
    displayId: 'delay-value-display',
    showToast: true,  // Mostrar toast en pestaña Config
    autoSave: true
  });
  
  // ✅ v10.5.117: Inicializar botón de limpiar procesos colgados
  initResetStuckButton();
  
  // ✅ v3.0.0: Inicializar botón de cancelar backup atascado
  initCancelBackupButton();
  
  // ✅ v4.0.8: Listener para cambio de cuenta activa
  chrome.storage.local.onChanged.addListener((changes) => {
    if (changes.activeAccountId) {
      logger.debug('[Config] Cuenta cambió, actualizando botón cancelar backup');
      // Actualizar botón si existe
      const cancelBtn = document.getElementById('cancel-backup-btn');
      if (cancelBtn) {
        // Llamar función updateButtonText de initCancelBackupButton
        // Como está en scope local, usamos evento custom
        window.dispatchEvent(new CustomEvent('activeAccountChanged'));
      }
    }
  });
}

/**
 * Carga la configuración actual del usuario
 * ✅ v10.5.34: Delega al componente centralizado
 */
export async function loadCurrentConfig() {
  try {
    await loadDelayFromDB();
  } catch (error) {
    logger.error('[Config] Error cargando configuración:', error);
  }
}

/**
 * Obtiene el delay actual de publicación
 * ✅ v10.5.34: Delega al componente centralizado
 */
export function getCurrentDelay() {
  return getDelayFromSlider();
}

/**
 * Limpia recursos del módulo
 */
export function cleanup() {
  if (sliderCleanup) {
    sliderCleanup();
    sliderCleanup = null;
  }
}

/**
 * ✅ v10.5.117: Inicializa el botón de limpiar procesos colgados
 */
function initResetStuckButton() {
  const btn = document.getElementById('reset-stuck-btn');
  const resultEl = document.getElementById('reset-stuck-result');
  
  if (!btn) return;
  
  btn.addEventListener('click', async () => {
    const accountId = getActiveAccountId();
    
    if (!accountId) {
      toast('⚠️ Selecciona una cuenta primero', 'warn');
      return;
    }
    
    btn.disabled = true;
    btn.textContent = '⏳ Limpiando...';
    
    try {
      const res = await chrome.runtime.sendMessage({
        type: 'API.FETCH_JSON',
        url: `/api/walla/publish2/reset-stuck`,
        method: 'POST',
        body: { account_id: accountId }
      });
      
      if (res?.ok && res.data?.ok) {
        const count = res.data.reset || 0;
        if (count > 0) {
          toast(`✅ ${count} anuncio(s) movido(s) a cola`, 'ok', 4000);
          if (resultEl) {
            resultEl.textContent = `Último: ${count} anuncio(s) limpiado(s)`;
            resultEl.style.display = 'block';
          }
        } else {
          toast('ℹ️ No hay procesos colgados', 'info', 3000);
        }
      } else {
        const error = res?.data?.detail || res?.error || 'Error desconocido';
        toast(`❌ Error: ${error}`, 'error');
        logger.error('[Config] Error reset-stuck:', error);
      }
    } catch (e) {
      toast(`❌ Error: ${e.message}`, 'error');
      logger.error('[Config] Error reset-stuck:', e);
    } finally {
      btn.disabled = false;
      btn.textContent = '🔄 Limpiar procesos colgados';
    }
  });
}

/**
 * ✅ v3.0.0: Inicializa el botón de cancelar backup atascado
 * Cancela backups en estado "running" que quedaron colgados
 */
async function initCancelBackupButton() {
  const btn = document.getElementById('cancel-backup-btn');
  const resultEl = document.getElementById('cancel-backup-result');
  
  if (!btn) return;
  
  // ✅ v4.0.5: Función para actualizar texto del botón
  const updateButtonText = async () => {
    const accountId = getActiveAccountId();
    if (!accountId) return;
    
    try {
      // Consultar cuántos backups atascados hay (sin cancelarlos)
      const res = await chrome.runtime.sendMessage({
        type: 'API.FETCH_JSON',
        url: `/api/system/maintenance/count-stuck-backups/${accountId}`,
        method: 'GET'
      });
      
      if (res?.ok && res.data?.count !== undefined) {
        const count = res.data.count;
        btn.textContent = count > 0 
          ? `🛑 Cancelar (${count} backup${count !== 1 ? 's' : ''})`
          : '🛑 Cancelar backup';
      }
    } catch (e) {
      // Si falla, mantener texto genérico
      btn.textContent = '🛑 Cancelar backup';
    }
  };
  
  // Actualizar texto inicial
  await updateButtonText();
  
  // ✅ v4.0.8: Listener para evento de cambio de cuenta
  window.addEventListener('activeAccountChanged', () => {
    updateButtonText();
  });
  
  btn.addEventListener('click', async () => {
    const accountId = getActiveAccountId();
    
    if (!accountId) {
      toast('⚠️ Selecciona una cuenta primero', 'warn');
      return;
    }
    
    // Confirmación antes de cancelar
    if (!confirm('¿Cancelar backup atascado? Se perderá el progreso actual del backup.')) {
      return;
    }
    
    btn.disabled = true;
    btn.textContent = '⏳ Cancelando...';
    
    try {
      const res = await chrome.runtime.sendMessage({
        type: 'API.FETCH_JSON',
        url: `/api/system/maintenance/cancel-stuck-backup/${accountId}`,
        method: 'POST'
      });
      
      if (res?.ok && res.data?.ok) {
        const count = res.data.cancelled || 0;
        if (count > 0) {
          toast(`✅ Se cancelaron ${count} backup(s) atascado(s)`, 'ok', 4000);
          if (resultEl) {
            resultEl.textContent = `Último: ${count} backup(s) cancelado(s)`;
            resultEl.style.display = 'block';
          }
        } else {
          toast('ℹ️ No había backups atascados', 'info', 3000);
          if (resultEl) {
            resultEl.textContent = 'No había backups atascados';
            resultEl.style.display = 'block';
          }
        }
        
        // Actualizar texto del botón después de cancelar
        await updateButtonText();
      } else {
        const error = res?.data?.detail || res?.error || 'Error desconocido';
        toast(`❌ Error: ${error}`, 'error');
        logger.error('[Config] Error cancel-backup:', error);
      }
    } catch (e) {
      toast(`❌ Error: ${e.message}`, 'error');
      logger.error('[Config] Error cancel-backup:', e);
    } finally {
      btn.disabled = false;
      if (btn.textContent === '⏳ Cancelando...') {
        btn.textContent = '🛑 Cancelar backup';
      }
    }
  });
}

