/*  Cabecera obligatoria: NO BORRAR NI MODIFICAR este bloque inicial en ningún fichero.
    Archivo: scripts/panel/base-modal.js – Rol: Componente base reutilizable para modales
    Descripcion de todo el proyecto:
         MitikLive Wallapop (MV3): panel lateral en Wallapop para gestionar anuncios conectado a FastAPI
            (login JWT, export/import, backups/renovaciones).
    Aviso: NO BORRAR los comentarios descriptivos situados encima de cada función.
    
    ✅ v10.5.32: Centralización de lógica común de modales (overlay, ESC, animaciones)
*/

import { logger } from './logger.js';

// Registro de modales activos para limpieza
const activeModals = new Map();

/**
 * Crea y muestra un modal con overlay
 * 
 * @param {Object} options - Opciones del modal
 * @param {string} options.id - ID único del modal (obligatorio)
 * @param {string} options.overlayClass - Clase CSS para el overlay (default: 'base-modal-overlay')
 * @param {string} options.content - HTML del contenido del modal
 * @param {boolean} options.closeOnOverlayClick - Cerrar al click en overlay (default: true)
 * @param {boolean} options.closeOnEsc - Cerrar con tecla ESC (default: true)
 * @param {number} options.autoCloseMs - Auto-cerrar después de X ms (0 = no auto-cerrar)
 * @param {Function} options.onClose - Callback al cerrar el modal
 * @param {Function} options.onOpen - Callback al abrir el modal
 * @returns {Object} - { overlay, close } para control manual
 */
export function createModal({
  id,
  overlayClass = 'base-modal-overlay',
  content = '',
  closeOnOverlayClick = true,
  closeOnEsc = true,
  autoCloseMs = 0,
  onClose = null,
  onOpen = null
}) {
  if (!id) {
    logger.error('[BaseModal] ID es obligatorio');
    return null;
  }
  
  // Cerrar modal existente con mismo ID
  closeModal(id);
  
  // Crear overlay
  const overlay = document.createElement('div');
  overlay.id = id;
  overlay.className = overlayClass;
  overlay.setAttribute('role', 'dialog');
  overlay.setAttribute('aria-modal', 'true');
  
  // Insertar contenido
  if (content) {
    overlay.innerHTML = content;
  }
  
  // Función de cierre
  const close = () => {
    closeModal(id);
  };
  
  // Handler para ESC
  let escHandler = null;
  if (closeOnEsc) {
    escHandler = (e) => {
      if (e.key === 'Escape') {
        close();
      }
    };
    document.addEventListener('keydown', escHandler);
  }
  
  // Click en overlay para cerrar
  if (closeOnOverlayClick) {
    overlay.addEventListener('click', (e) => {
      if (e.target === overlay) {
        close();
      }
    });
  }
  
  // Registrar modal activo
  activeModals.set(id, {
    overlay,
    escHandler,
    onClose,
    autoCloseTimer: null
  });
  
  // Añadir al DOM
  document.body.appendChild(overlay);
  
  // Auto-cerrar si está configurado
  if (autoCloseMs > 0) {
    const timer = setTimeout(close, autoCloseMs);
    activeModals.get(id).autoCloseTimer = timer;
  }
  
  // Callback onOpen
  if (typeof onOpen === 'function') {
    try {
      onOpen(overlay);
    } catch (err) {
      logger.error('[BaseModal] Error en onOpen:', err);
    }
  }
  
  return { overlay, close };
}

/**
 * Cierra un modal por ID
 * 
 * @param {string} id - ID del modal a cerrar
 * @param {boolean} animate - Si aplicar animación de salida (default: true)
 */
export function closeModal(id, animate = true) {
  const modalData = activeModals.get(id);
  if (!modalData) return;
  
  const { overlay, escHandler, onClose, autoCloseTimer } = modalData;
  
  // Limpiar timer de auto-cierre
  if (autoCloseTimer) {
    clearTimeout(autoCloseTimer);
  }
  
  // Remover listener de ESC
  if (escHandler) {
    document.removeEventListener('keydown', escHandler);
  }
  
  // Animación de salida
  if (animate && overlay) {
    overlay.style.opacity = '0';
    overlay.style.transition = 'opacity 0.2s ease';
    
    setTimeout(() => {
      overlay.remove();
      activeModals.delete(id);
      
      // Callback onClose
      if (typeof onClose === 'function') {
        try {
          onClose();
        } catch (err) {
          logger.error('[BaseModal] Error en onClose:', err);
        }
      }
    }, 200);
  } else {
    // Sin animación
    if (overlay) overlay.remove();
    activeModals.delete(id);
    
    if (typeof onClose === 'function') {
      try {
        onClose();
      } catch (err) {
        logger.error('[BaseModal] Error en onClose:', err);
      }
    }
  }
}

/**
 * Cierra todos los modales activos
 */
export function closeAllModals() {
  for (const id of activeModals.keys()) {
    closeModal(id, false);
  }
}

/**
 * Verifica si un modal está abierto
 * 
 * @param {string} id - ID del modal
 * @returns {boolean}
 */
export function isModalOpen(id) {
  return activeModals.has(id);
}

/**
 * Obtiene el overlay de un modal activo
 * 
 * @param {string} id - ID del modal
 * @returns {HTMLElement|null}
 */
export function getModalOverlay(id) {
  const modalData = activeModals.get(id);
  return modalData ? modalData.overlay : null;
}
