/*  Cabecera obligatoria: NO BORRAR NI MODIFICAR este bloque inicial en ningún fichero.
    Archivo: scripts/panel/auth.js – Rol: Login/remember/logout y utilidades
    Descripcion de todo el proyecto:
         MitikLive Wallapop (MV3): panel lateral en Wallapop para gestionar anuncios conectado a FastAPI
            (login JWT, export/import, backups/renovaciones).
    Aviso: NO BORRAR los comentarios descriptivos situados encima de cada función, solo cambiarlos si se modifica la funcion.
    En Rol (linea mas arriba): si está vacía o el fichero se modifica o se reestructura, modificar esa linea de rol
*/

import { stGet, stSet, stRemove } from './storage.js';
import { logger } from './logger.js';
import { checkMaintenance } from './maintenance.js';
import { sleep } from '../utils.js';  // ✅ v6.4.4: Centralizar delays
import { setCurrentDelay } from './delay-slider.js'; // ✅ v10.5.34: Componente centralizado

// ✅ SUPPRESS_KEY eliminado - backend maneja viewed_at
import { 
  listEmails, inPass, inEmail, remember, btnLogin, btnGoogle, 
  selAccount, AUTH,
  set_AUTH
} from './state.js';
import { toast, showModal, showLoader, hideLoader } from '../utils.js';
import { sw } from './runs.js';
import { setAuthUI, setRunControls, renderAccounts } from './ui.js'; // ✅ v5.1.0: hideRunUI eliminado
import { detectLoggedWalla } from './detect.js';
import { switchTab } from './tabs.js';

/* ==========================================
   ✅ v2.2.3: FLUJO POST-LOGIN CENTRALIZADO
   ==========================================
   Esta función se llama después de CUALQUIER login exitoso:
   - Login con email/password
   - Login con Google OAuth
   - (Futuro) Login con Facebook, Apple, etc.
   
   Garantiza que TODOS los proveedores de auth ejecuten
   los mismos pasos post-login en el mismo orden.
   ========================================== */

/**
 * Ejecuta todos los pasos necesarios después de un login exitoso.
 * CENTRALIZADO para todos los métodos de autenticación.
 * 
 * @param {Object} options - Opciones del login
 * @param {boolean} options.isNewUser - Si es un usuario nuevo (mostrar bienvenida)
 * @param {number|null} options.lastSelectedAccountId - Última cuenta seleccionada (para Google OAuth)
 * @param {string} options.provider - Proveedor de auth ('email', 'google', 'facebook', 'apple')
 * @returns {Promise<boolean>} true si completó exitosamente
 */
async function onLoginSuccess({ isNewUser = false, lastSelectedAccountId = null, provider = 'unknown' } = {}) {
  const logPrefix = `[AUTH-${provider.toUpperCase()}]`;
  
  try {
    // ✅ v2.2.3: Versión ya verificada al abrir panel (checkVersionWithoutAuth en panel.js)
    // No es necesario verificar aquí de nuevo
    
    // ═══════════════════════════════════════════════════════════════
    // 1. CARGAR CUENTAS DE WALLAPOP
    // ═══════════════════════════════════════════════════════════════
    await loadAccounts();
    
    // ═══════════════════════════════════════════════════════════════
    // 2. SELECCIONAR ÚLTIMA CUENTA USADA (si aplica)
    // ═══════════════════════════════════════════════════════════════
    if (lastSelectedAccountId && lastSelectedAccountId > 0) {
      const selAccountEl = document.getElementById('sel-account');
      if (selAccountEl) {
        const option = selAccountEl.querySelector(`option[data-account-id="${lastSelectedAccountId}"]`);
        if (option) {
          selAccountEl.value = option.value;
          selAccountEl.dispatchEvent(new Event('change'));
          logger.debug(`${logPrefix} Cuenta ${lastSelectedAccountId} restaurada`);
        }
      }
    }
    
    // ═══════════════════════════════════════════════════════════════
    // 3. ACTIVAR FILTRO "ACTIVE" POR DEFECTO
    // ═══════════════════════════════════════════════════════════════
    try {
      const { highlightActiveFilter } = await import('./listings/filters.js');
      const { FILTERS } = await import('./listings/config.js');
      highlightActiveFilter(FILTERS.ACTIVE);
    } catch (err) {
      // Silenciar - no crítico
    }
    
    // ═══════════════════════════════════════════════════════════════
    // 4. CARGAR Y APLICAR PREFERENCIAS DE USUARIO
    // ═══════════════════════════════════════════════════════════════
    await loadAndApplyUserPreferences();
    
    // ═══════════════════════════════════════════════════════════════
    // 5. INICIALIZAR SISTEMA DE CRÉDITOS
    // ═══════════════════════════════════════════════════════════════
    if (window.CreditsManager && typeof window.CreditsManager.init === 'function') {
      await window.CreditsManager.init();
    }
    
    // ═══════════════════════════════════════════════════════════════
    // 6. REFRESCAR BOTONES DE PUBLICACIÓN (con delay por race conditions)
    // ═══════════════════════════════════════════════════════════════
    setTimeout(async () => {
      try {
        const { refreshAllButtons } = await import('./publish-button-manager.js');
        if (refreshAllButtons) {
          refreshAllButtons();
        }
      } catch (err) {
        // Silenciar - no crítico
      }
    }, 1500);
    
    // ═══════════════════════════════════════════════════════════════
    // 7. MOSTRAR MODAL DE BIENVENIDA (si es usuario nuevo)
    // ═══════════════════════════════════════════════════════════════
    if (isNewUser) {
      try {
        const { showWelcomeModalForNewUser } = await import('./ui.js');
        await showWelcomeModalForNewUser();
      } catch (err) {
        logger.error(`${logPrefix} Error mostrando bienvenida:`, err);
      }
    }
    
    // ═══════════════════════════════════════════════════════════════
    // 8. DETECTAR SESIÓN DE WALLAPOP
    // ═══════════════════════════════════════════════════════════════
    await detectLoggedWalla();
    
    logger.info(`${logPrefix} ✅ Login completado exitosamente`);
    return true;
    
  } catch (err) {
    logger.error(`${logPrefix} Error en onLoginSuccess:`, err);
    return false;
  }
}

/* ==========================================
   CARGA Y APLICACIÓN DE PREFERENCIAS UI
   ========================================== */

/**
 * Carga y aplica TODAS las preferencias UI del usuario desde BD
 * Se llama después de login exitoso
 * 
 * Preferencias soportadas:
 * - zoom: Nivel de zoom del panel (50-150%)
 * - active_tab: Última pestaña activa (listings/logs/stats/config)
 * - publish_delay_seconds: Delay entre publicaciones (5-240s)
 * 
 * @returns {Promise<boolean>} true si se cargaron correctamente
 */
export async function loadAndApplyUserPreferences() {
  try {
    
    // ✅ v10.5.81: Usar PATCH en lugar de GET para también actualizar extension_version
    // El PATCH devuelve las mismas preferencias que el GET
    const version = chrome.runtime.getManifest().version;
    const response = await sw({ 
      type: 'API.FETCH_JSON',
      url: '/api/user/preferences',
      method: 'PATCH',
      body: { extension_version: version }
    });
    
    if (!response?.ok) {
      logger.warn('[AUTH] No se pudieron cargar/actualizar preferencias');
      return true; // No es error crítico
    }
    
    // ✅ v10.5.82: Log de versión eliminado (no necesario)
    
    if (!response?.data?.ui_preferences) {
      return true;
    }
    
    const prefs = response.data.ui_preferences;
    
    // 1. APLICAR ZOOM
    if (prefs.zoom && typeof prefs.zoom === 'number') {
      const zoomValue = Math.max(50, Math.min(150, prefs.zoom)); // Clamp 50-150
      document.documentElement.style.fontSize = `${zoomValue}%`;
    }
    
    // 2. APLICAR PESTAÑA ACTIVA
    if (prefs.active_tab && ['listings', 'logs', 'stats', 'config'].includes(prefs.active_tab)) {
      // Usar setTimeout para asegurar que el DOM está listo
      setTimeout(() => {
        switchTab(prefs.active_tab);
      }, 100);
    }
    
    // 3. APLICAR DELAY DE PUBLICACIÓN
    // ✅ v10.5.34: Usar componente centralizado para sincronizar todos los sliders
    if (prefs.publish_delay_seconds && typeof prefs.publish_delay_seconds === 'number') {
      const delayValue = Math.max(5, Math.min(240, prefs.publish_delay_seconds)); // Clamp 5-240
      setCurrentDelay(delayValue); // Actualiza estado interno + sincroniza sliders via Bus
    }
    
    // 4. AQUÍ AÑADIR FUTURAS PREFERENCIAS
    // Ejemplo:
    // if (prefs.theme) applyTheme(prefs.theme);
    // if (prefs.language) applyLanguage(prefs.language);
    // if (prefs.notifications) applyNotifications(prefs.notifications);
    
    return true;
    
  } catch (error) {
    logger.error('[AUTH] ❌ Error cargando preferencias:', error);
    return false;
  }
}

/* =========================
   Historial de credenciales
   ========================= */
export async function loadLoginHistory() {
  const { login_history } = await stGet(['login_history']);
  return Array.isArray(login_history) ? login_history : [];
}

export async function saveLoginToHistory(email, pass, keepPass) {
  let hist = await loadLoginHistory();
  hist = hist.filter(x => x.email !== email);
  hist.unshift({ email, pass: keepPass ? pass : '' });
  if (hist.length > 8) hist = hist.slice(0, 8);
  await stSet({ 
    login_history: hist, 
    last_login: { email, pass: keepPass ? pass : '' } 
  });
}

export async function fillEmailDatalist() {
  const hist = await loadLoginHistory();
  if (!listEmails) return;
  listEmails.innerHTML = '';
  for (const h of hist) {
    const opt = document.createElement('option');
    opt.value = h.email;
    listEmails.appendChild(opt);
  }
}

export async function autofillPassForEmail(email) {
  const hist = await loadLoginHistory();
  const item = hist.find(x => x.email === email);
  if (item && item.pass && inPass) inPass.value = item.pass;
}

/* ============================
   Utilidades de pestaña Walla
   ============================ */
function isWallaHost(u) {
  try { 
    return /wallapop\.(com|es)$/i.test(new URL(u).hostname); 
  } catch { 
    return false; 
  }
}

export async function requireWallaActive() {
  const [active] = await chrome.tabs.query({ active: true, currentWindow: true });
  
  // ✅ Si ya estamos en Wallapop y tiene login, retornar inmediatamente
  if (isWallaHost(active?.url)) {
    const loginInfo = await checkWallapopLogin(active);
    if (loginInfo.hasLogin) {
      // ✅ v6.2.0: Validar que la cuenta coincida con la seleccionada
      const validationResult = await validateAccountMatch(loginInfo);
      if (!validationResult.ok) {
        return { ok: false, reason: 'account_mismatch' };
      }
      return { ok: true, tab: active, wallapopAccount: loginInfo };
    }
  }

  const wallaTabs = await chrome.tabs.query({ 
    url: ['https://*.wallapop.com/*', 'https://*.wallapop.es/*'] 
  });
  
  // 🔄 v4.47.0: Ocultar loader SOLO cuando vamos a mostrar modal
  const { hideLoader } = await import('../utils.js');
  hideLoader();
  
  const choice = await showModal({
    title: 'Se necesita una pestaña de Wallapop',
    html: wallaTabs.length
      ? 'No estás en Wallapop. ¿Qué hacemos?'
      : 'No estás en Wallapop. ¿Abrimos una pestaña?',
    buttons: [
      ...(wallaTabs.length ? [{ text: 'Cambiar a Wallapop', value: 'switch', primary: true }] : []),
      { text: 'Abrir aquí', value: 'here' },
      { text: 'Abrir nueva', value: 'new' },
      { text: 'Cancelar', value: 'cancel' }
    ]
  });

  let targetTab = null;

  if (choice === 'switch' && wallaTabs[0]) {
    await chrome.tabs.update(wallaTabs[0].id, { active: true });
    await chrome.windows.update(wallaTabs[0].windowId, { focused: true });
    targetTab = wallaTabs[0];
  } else if (choice === 'here' && active?.id) {
    await chrome.tabs.update(active.id, { 
      url: 'https://es.wallapop.com/app/login', 
      active: true 
    });
    targetTab = active;
  } else if (choice === 'new') {
    const t = await chrome.tabs.create({ 
      url: 'https://es.wallapop.com/app/login', 
      active: true, 
      index: (active?.index ?? 0) + 1 
    });
    try { 
      await chrome.windows.update(t.windowId, { focused: true }); 
    } catch {}
    targetTab = t;
  } else {
    return { ok: false };
  }

  // 🔄 Mostrar loader de nuevo mientras espera
  const { showLoader, setLoaderText } = await import('../utils.js');
  showLoader('Esperando login en Wallapop...', { timeout: 20000 });
  
  // 🆕 ESPERAR a que la pestaña cargue Y tenga login
  if (targetTab) {
    const result = await waitForWallapopReady(targetTab);
    
    // 🔄 v4.47.1: Si NO detecta → Refrescar AUTOMÁTICAMENTE (sin preguntar)
    if (!result.ready) {
      
      setLoaderText('Refrescando Wallapop...');
      
      // Recargar Wallapop
      await chrome.tabs.reload(targetTab.id);
      
      // Esperar de nuevo (con timeout más corto)
      const result2 = await waitForWallapopReady(targetTab, 15000);
      
      hideLoader();
      
      if (!result2.ready) {
        // Ahora SÍ mostrar modal pidiendo login
        await showModal({
          title: '⚠️ Inicia sesión en Wallapop',
          html: `
            <p style="margin-bottom: 12px; line-height: 1.6;">
              No se detectó ninguna sesión activa en Wallapop.
            </p>
            <p class="modal-text-warning">
              Por favor, inicia sesión en Wallapop manualmente.
            </p>
            <p class="modal-text-muted">
              Después, vuelve aquí e intenta de nuevo.
            </p>
          `,
          buttons: [
            { text: 'Entendido', value: 'ok', primary: true }
          ]
        });
        
        toast('⚠️ Inicia sesión en Wallapop', 'warn', 4000);
        return { ok: false };
      }
      
      
      // ✅ v6.2.0: Validar que la cuenta coincida
      const validationResult2 = await validateAccountMatch(result2.accountInfo);
      if (!validationResult2.ok) {
        return { ok: false, reason: 'account_mismatch' };
      }
      
      return { ok: true, tab: targetTab, wallapopAccount: result2.accountInfo };
    }
    
    hideLoader();
    
    // ✅ v6.2.0: Validar que la cuenta coincida
    const validationResult = await validateAccountMatch(result.accountInfo);
    if (!validationResult.ok) {
      return { ok: false, reason: 'account_mismatch' };
    }
    
    return { ok: true, tab: targetTab, wallapopAccount: result.accountInfo };
  }

  return { ok: false };
}

/**
 * ✅ v6.2.0: Valida que la cuenta de Wallapop coincida con la seleccionada en el panel
 * @param {object} wallapopInfo - Información del login de Wallapop (alias, avatar)
 * @returns {Promise<{ok: boolean}>}
 */
async function validateAccountMatch(wallapopInfo) {
  const { hideLoader, showLoader } = await import('../utils.js');
  
  // Obtener cuenta seleccionada del dropdown
  const selAccount = document.getElementById('sel-account');
  const selectedOption = selAccount?.selectedOptions?.[0];
  
  const wallapopAlias = wallapopInfo?.alias || '';
  
  // Si no hay cuenta seleccionada en el panel
  if (!selectedOption || !selectedOption.value || selectedOption.value === '') {
    // ✅ v6.4.1: Si tenemos info de Wallapop, intentar crear/seleccionar automáticamente
    if (wallapopAlias) {
      
      // Verificar si la cuenta de Wallapop ya existe en el panel
      const allOptions = Array.from(selAccount?.options || []);
      const existingAccount = allOptions.find(opt => 
        opt.dataset?.alias && opt.dataset.alias.toLowerCase() === wallapopAlias.toLowerCase()
      );
      
      if (existingAccount) {
        // Cuenta existe, solo seleccionarla
        selAccount.value = existingAccount.value;
        selAccount.dispatchEvent(new Event('change', { bubbles: true }));
        toast(`✅ Cuenta ${wallapopAlias} seleccionada automáticamente`, 'ok', 3000);
        return { ok: true };
      } else {
        // Cuenta NO existe, preguntar si crear
        hideLoader();
        const result = await showModal({
          title: '🆕 Nueva cuenta detectada',
          html: `
            <p style="margin-bottom: 12px; line-height: 1.6;">
              Has iniciado sesión en Wallapop con <strong>${wallapopAlias}</strong>.
            </p>
            <p style="font-size: 14px; color: var(--brand-primary); margin-bottom: 12px;">
              ¿Deseas añadir esta cuenta y hacer backup de sus anuncios?
            </p>
            <p class="modal-text-muted">
              La cuenta se añadirá automáticamente y podrás gestionarla desde el panel.
            </p>
          `,
          buttons: [
            { text: 'Cancelar', value: 'cancel' },
            { text: '✅ Añadir y hacer Backup', value: 'add', primary: true }
          ]
        });
        
        if (result === 'add') {
          const newAccount = await createNewWallapopAccount(wallapopInfo);
          if (newAccount) {
            toast(`✅ Cuenta ${wallapopAlias} añadida correctamente`, 'ok', 3000);
            return { ok: true, newAccount };
          } else {
            toast('❌ Error al añadir la cuenta', 'error', 4000);
            return { ok: false };
          }
        }
        
        return { ok: false };
      }
    }
    
    // No tenemos info de Wallapop, mostrar mensaje original
    hideLoader();
    await showModal({
      title: '⚠️ Selecciona una cuenta',
      html: `
        <p style="margin-bottom: 12px; line-height: 1.6;">
          Debes seleccionar una cuenta de Wallapop en el panel antes de realizar esta acción.
        </p>
      `,
      buttons: [
        { text: 'Entendido', value: 'ok', primary: true }
      ]
    });
    toast('⚠️ Selecciona una cuenta primero', 'warn', 4000);
    return { ok: false };
  }
  
  const selectedAlias = selectedOption.dataset.alias || selectedOption.textContent?.split(' ')[0] || '';
  
  // Si no tenemos alias de Wallapop para comparar, permitir (por compatibilidad)
  if (!wallapopAlias) {
    return { ok: true };
  }
  
  // Si no tenemos alias seleccionado para comparar
  if (!selectedAlias) {
    return { ok: true };
  }
  
  // Comparar aliases (case-insensitive)
  if (selectedAlias.toLowerCase() !== wallapopAlias.toLowerCase()) {
    hideLoader();
    
    // ✅ v6.4.0: Verificar si la cuenta de Wallapop existe en alguna cuenta del usuario
    const allOptions = Array.from(selAccount.options);
    const existingAccount = allOptions.find(opt => 
      opt.dataset.alias && opt.dataset.alias.toLowerCase() === wallapopAlias.toLowerCase()
    );
    
    if (existingAccount) {
      // Cuenta existe pero no es la seleccionada
      await showModal({
        title: '⚠️ Cuenta incorrecta',
        html: `
          <p style="margin-bottom: 12px; line-height: 1.6;">
            <strong>Panel:</strong> ${selectedAlias}<br>
            <strong>Wallapop:</strong> ${wallapopAlias}
          </p>
          <p class="modal-text-warning">
            Abre Wallapop con la cuenta <strong>"${selectedAlias}"</strong> para continuar.
          </p>
          <p class="modal-text-muted">
            O selecciona la cuenta "${wallapopAlias}" en el panel si es la que deseas usar.
          </p>
        `,
        buttons: [
          { text: 'Entendido', value: 'ok', primary: true }
        ]
      });
      
      toast(`⚠️ Cuenta incorrecta: ${wallapopAlias} ≠ ${selectedAlias}`, 'warn', 5000);
      return { ok: false };
    } else {
      // ✅ v6.4.0: Cuenta NUEVA - no existe en el panel
      
      const result = await showModal({
        title: '🆕 Nueva cuenta detectada',
        html: `
          <p style="margin-bottom: 12px; line-height: 1.6;">
            Has iniciado sesión en Wallapop con <strong>${wallapopAlias}</strong>, 
            que no está en tu lista de cuentas.
          </p>
          <p style="font-size: 14px; color: var(--brand-primary); margin-bottom: 12px;">
            ¿Deseas añadir esta cuenta y hacer backup de sus anuncios?
          </p>
          <p class="modal-text-muted">
            La cuenta se añadirá automáticamente y podrás gestionarla desde el panel.
          </p>
        `,
        buttons: [
          { text: 'Cancelar', value: 'cancel' },
          { text: '✅ Añadir y hacer Backup', value: 'add', primary: true }
        ]
      });
      
      if (result === 'add') {
        // Crear cuenta nueva
        const newAccount = await createNewWallapopAccount(wallapopInfo);
        if (newAccount) {
          toast(`✅ Cuenta ${wallapopAlias} añadida correctamente`, 'ok', 3000);
          return { ok: true, newAccount };
        } else {
          toast('❌ Error al añadir la cuenta', 'error', 4000);
          return { ok: false };
        }
      }
      
      return { ok: false };
    }
  }
  
  return { ok: true };
}

/**
 * ✅ v6.4.0: Crea una nueva cuenta de Wallapop en el backend
 */
async function createNewWallapopAccount(wallapopInfo) {
  const { showLoader, hideLoader } = await import('../utils.js');
  
  try {
    showLoader('Añadiendo cuenta...');
    
    const alias = wallapopInfo.alias || '';
    // ✅ v6.4.1: Usar la URL real del perfil (con ID numérico) en lugar de construirla
    const profileUrl = wallapopInfo.profileUrl || `https://es.wallapop.com/app/user/${alias}/published`;
    
    // ✅ v10.5.76: Obtener ID alfanumérico del __NEXT_DATA__ (más confiable que el numérico de la URL)
    let wallaUserId = null;
    try {
      const userIdResponse = await sw({ type: 'WALLA.GET_USER_ID', profileUrl });
      if (userIdResponse?.ok && userIdResponse?.walla_user_id) {
        wallaUserId = userIdResponse.walla_user_id;
        logger.debug('[AUTH] ✅ ID alfanumérico obtenido:', wallaUserId);
      }
    } catch (e) {
      logger.warn('[AUTH] ⚠️ No se pudo obtener ID alfanumérico, usando numérico de URL');
    }
    
    // Fallback: extraer ID numérico de la URL si no se pudo obtener el alfanumérico
    if (!wallaUserId && profileUrl) {
      const match = profileUrl.match(/\/user\/[^\/]+-(\d+)\/?$/);
      if (match) {
        wallaUserId = match[1];
        logger.debug('[AUTH] ⚠️ Usando ID numérico de URL:', wallaUserId);
      }
    }
    
    // Llamar al backend para crear la cuenta (usa query params, no body)
    const params = { alias, profile_url: profileUrl };
    if (wallaUserId) params.walla_user_id = wallaUserId;
    
    const queryParams = new URLSearchParams(params).toString();
    
    const response = await sw({
      type: 'API.CALL',
      method: 'POST',
      endpoint: `/api/walla/accounts?${queryParams}`,
      body: null
    });
    
    if (response?.ok && response?.data) {
      
      // Refrescar el combo de cuentas
      await refreshAccountsDropdown(response.data);
      
      hideLoader();
      return response.data;
    } else {
      // ✅ v10.5.82: Ocultar loader ANTES de mostrar modal
      hideLoader();
      
      // ✅ v10.5.81: Manejar error 409 - cuenta ya vinculada a otro usuario
      const errorDetail = response?.data?.detail || response?.error || '';
      const isConflict = response?.status === 409 || errorDetail.includes('ya está vinculada');
      
      if (isConflict) {
        logger.warn('[AUTH] ⚠️ Cuenta duplicada detectada:', errorDetail);
        // ✅ v10.5.124: Usar 'html' en vez de 'message' para showModal
        await showModal({
          title: '⚠️ Cuenta ya vinculada',
          html: 'Esta cuenta de Wallapop ya está vinculada a otro usuario.<br><br>Si eres el propietario legítimo y perdiste acceso a tu cuenta anterior, contacta con soporte.',
          buttons: [{ text: 'Entendido', value: 'ok', primary: true }]
        });
      } else {
        logger.error('[AUTH] ❌ Error creando cuenta:', response);
      }
      
      return null;
    }
  } catch (err) {
    logger.error('[AUTH] ❌ Error al crear cuenta:', err);
    hideLoader();
    return null;
  }
}

/**
 * ✅ v6.4.0: Refresca el dropdown de cuentas y selecciona la nueva
 */
async function refreshAccountsDropdown(newAccountData) {
  try {
    // ✅ v6.4.2: La función correcta es renderAccounts, no renderAccountOptions
    const { renderAccounts } = await import('./ui.js');
    const { sleep } = await import('../utils.js');
    
    // Obtener lista actualizada de cuentas
    const response = await sw({
      type: 'API.CALL',
      method: 'GET',
      endpoint: '/api/walla/accounts?with_counts=1'
    });
    
    if (response?.ok && response?.data) {
      // ✅ v6.4.2: Guardar la nueva cuenta ANTES de renderizar 
      // para que renderAccounts la seleccione automáticamente
      await stSet({ selected_account_id: newAccountData.id });
      
      // Renderizar opciones - esto dispara handleAccountChange() via evento change
      await renderAccounts(response.data);
      
      // Verificar que se seleccionó correctamente
      const selAccount = document.getElementById('sel-account');
      if (selAccount && selAccount.value != newAccountData.id) {
        // Si no se seleccionó, forzar selección
        selAccount.value = String(newAccountData.id);
        selAccount.dispatchEvent(new Event('change', { bubbles: true }));
      }
      
      // ✅ v6.4.2: Esperar a que handleAccountChange() termine de cargar la UI
      // Esto es necesario porque dispatchEvent es síncrono pero el handler es async
      await sleep(800);
      
    }
  } catch (err) {
    logger.error('[AUTH] ❌ Error refrescando dropdown:', err);
  }
}

/**
 * Comprueba si hay login en Wallapop mirando el avatar en el DOM
 * @returns {Promise<{hasLogin: boolean, alias?: string, avatar?: string}>}
 */
async function checkWallapopLogin(tab) {
  if (!tab?.id) return { hasLogin: false };
  
  try {
    // ✅ v10.5.82: Envolver para silenciar lastError
    const response = await new Promise((resolve) => {
      chrome.tabs.sendMessage(tab.id, { type: 'DOM.CHECK_LOGIN' }, (r) => {
        void chrome.runtime.lastError;
        resolve(r);
      });
    });
    
    return {
      hasLogin: response?.hasLogin || false,
      alias: response?.alias || null,
      avatar: response?.avatar || null,
      profileUrl: response?.profileUrl || null  // ✅ v6.4.1: Añadir URL del perfil
    };
  } catch {
    return { hasLogin: false };
  }
}

/**
 * Espera a que Wallapop cargue completamente Y tenga login
 * Timeout: 15 segundos
 * @returns {Promise<{ready: boolean, accountInfo?: object}>}
 */
async function waitForWallapopReady(tab, maxWaitMs = 15000) {
  const startTime = Date.now();
  let lastCheckTime = 0;
  
  while (Date.now() - startTime < maxWaitMs) {
    try {
      // Esperar a que el tab complete la carga
      const currentTab = await chrome.tabs.get(tab.id);
      
      // ⚠️ Si el usuario cambió de pestaña, esperamos a que vuelva
      if (!currentTab.active) {
        await sleep(1000);
        continue;
      }
      
      if (currentTab.status !== 'complete') {
        await sleep(500);
        continue;
      }
      
      // Tab cargado, verificar login
      const now = Date.now();
      
      // 🔄 Reintentar detección cada 2 segundos para dar tiempo al DOM
      if (now - lastCheckTime < 2000 && lastCheckTime > 0) {
        await sleep(500);
        continue;
      }
      
      lastCheckTime = now;
      
      const loginInfo = await checkWallapopLogin(currentTab);
      
      if (loginInfo.hasLogin) {
        return { ready: true, accountInfo: loginInfo }; // ✅ Devolver info completa
      }
      
      // No tiene login todavía, esperar un poco más
      await sleep(1000);
      
    } catch (error) {
      await sleep(500);
    }
  }
  
  // Timeout alcanzado
  return { ready: false };
}

/* ============================
   Cuenta activa (selector UI)
   ============================ */
export function getActiveAccountId() {
  const opt = selAccount?.selectedOptions?.[0];
  const id = opt?.dataset?.accountId;
  return id ? Number(id) : null;
}

/* ========================
   Cargar cuentas desde SW
   ======================== */
export async function loadAccounts() {
  const r = await sw({ type: 'ACCOUNTS.LIST' });
  if (!r?.ok) return;
  renderAccounts(r.data);
}

/* ================
   Login principal
   ================ */
async function doLogin() {
  const email = inEmail?.value?.trim() || '';
  const pass = inPass?.value || '';


  if (!email || !pass) {
    toast('Falta email o contraseña', 'warn');
    return;
  }

  if (btnLogin) btnLogin.disabled = true;
  showLoader('Validando…', {
    timeout: 30000,
    onTimeout: () => toast('⏱️ Tiempo agotado validando credenciales', 'error', 5000)
  });

  try {
    
    const r = await sw({ 
      type: 'SESSION.LOGIN_EMAIL', 
      email, 
      password: pass 
    });


    if (!r?.ok) {
      logger.error('[AUTH] Login falló:', r);
      
      // Manejar bloqueo por brute force
      if (r.error === 'too_many_attempts') {
        const retryAfter = r.retry_after || 120;
        toast(`🔒 Bloqueado por intentos fallidos. Intenta en ${retryAfter}s`, 'err');
        
        // Deshabilitar botón temporalmente
        if (btnLogin) {
          btnLogin.disabled = true;
          btnLogin.textContent = `Bloqueado (${retryAfter}s)`;
          
          // Contador regresivo
          let remaining = retryAfter;
          const interval = setInterval(() => {
            remaining--;
            if (remaining <= 0) {
              clearInterval(interval);
              btnLogin.disabled = false;
              btnLogin.textContent = 'Iniciar Sesión';
            } else {
              const mins = Math.floor(remaining / 60);
              const secs = remaining % 60;
              const timeStr = mins > 0 ? `${mins}:${String(secs).padStart(2, '0')}` : `${secs}s`;
              btnLogin.textContent = `Bloqueado (${timeStr})`;
            }
          }, 1000);
        }
        return;
      }
      
      // Manejar credenciales incorrectas
      if (r.error === 'invalid_credentials') {
        const attemptsLeft = r.attempts_left || 0;
        if (attemptsLeft > 0) {
          toast(`❌ Credenciales incorrectas. Intentos restantes: ${attemptsLeft}`, 'warn');
        } else {
          toast(`⚠️ Último intento. El siguiente te bloqueará`, 'warn');
        }
        return;
      }
      
      // 🔒 Manejar modo mantenimiento (HTTP 503)
      if (r.error === 'maintenance_mode' || r.status === 503) {
        
        // Mostrar modal de mantenimiento (sin token = sin acceso)
        const { showMaintenanceModal } = await import('./maintenance.js');
        showMaintenanceModal(r.message || 'Sistema en mantenimiento. Intenta más tarde.');
        
        return; // No continuar con login
      }
      
      // ✅ v10.5.127: Manejar cuenta suspendida/baneada (HTTP 403)
      if (r.status === 403 || (r.error && r.error.includes('account_suspended'))) {
        const reason = r.error?.replace('account_suspended:', '').trim() || 
                       r.message?.replace('account_suspended:', '').trim() || 
                       'Actividad sospechosa detectada';
        await showModal({
          title: '⛔ Cuenta suspendida',
          html: `Tu cuenta ha sido suspendida.<br><br><strong>Motivo:</strong> ${reason}<br><br>Si crees que es un error, contacta con soporte.`,
          buttons: [{ text: 'Entendido', value: 'ok', primary: true }]
        });
        return;
      }
      
      // Error genérico
      toast('Login error: ' + (r?.message || r?.error || 'desconocido'), 'err');
      return;
    }

    // Guardar en historial
    const keepPass = remember?.checked || false;
    await saveLoginToHistory(email, pass, keepPass);

    // 🔥 LIMPIAR storage del usuario anterior para evitar mostrar datos incorrectos
    await stRemove([
      'selected_account_id',
      'account_id',
      'accounts',
      'listings_cache',
      'stats_cache'
      // ✅ ml.run_id eliminado - backend es fuente de verdad
    ]).catch(() => {});

    
    // Actualizar estado AUTH
    const s = await sw({ type: 'SESSION.GET' });
    

    if (!s || !s.data) {
      logger.error('[AUTH] SESSION.GET devolvió datos inválidos:', s);
      toast('Error consolidando el login (sin datos)', 'err');
      return;
    }

    const newAuth = {
      state: s.data.state || 'UNAUTHENTICATED',
      user: s.data.user || null,
      config: s.data.config || { API_BASE: '' },
      on: false
    };
    
    
    set_AUTH(newAuth);

    // ✅ run_id ya no se guarda en localStorage
    setRunControls(false);
    // ✅ v5.1.0: hideRunUI eliminado (era no-op)


    // Pintar UI autenticada
    setAuthUI(
      newAuth.state === 'AUTHENTICATED', 
      newAuth.user?.email || email
    );

    if (newAuth.state === 'AUTHENTICATED') {
      // ✅ v2.2.3: Usar flujo centralizado post-login
      const isNewUser = r.data?.is_new_user || false;
      const success = await onLoginSuccess({
        isNewUser,
        lastSelectedAccountId: null,  // Login normal no tiene esto
        provider: 'email'
      });
      
      if (!success) return;  // Versión desactualizada, modal ya mostrado
      
    } else {
      toast('Estado inesperado: ' + newAuth.state, 'warn');
    }

    // Detectar sesión Wallapop
    await detectLoggedWalla();

  } catch (e) {
    logger.error('[AUTH] Excepción en doLogin:', e);
    
    // Detectar error de conexión
    const errorMsg = String(e?.message || e).toLowerCase();
    
    if (errorMsg.includes('fetch') || errorMsg.includes('network') || 
        errorMsg.includes('failed to fetch') || errorMsg.includes('networkerror')) {
      toast('❌ Sin conexión al servidor. Verifica tu internet.', 'err');
    } else if (errorMsg.includes('timeout')) {
      toast('⏱️ Tiempo de espera agotado. Intenta nuevamente.', 'err');
    } else {
      toast('Error inesperado: ' + String(e?.message || e), 'err');
    }
  } finally {
    hideLoader();
    if (btnLogin) btnLogin.disabled = false;
  }
}


/* ====================
   Login Google (popup)
   ==================== */
async function doGoogleLogin() {
  try {
    showLoader('Verificando conexión…', {
      timeout: 30000,
      onTimeout: () => toast('⏱️ Tiempo agotado verificando conexión', 'error', 5000)
    });
    
    // 🔍 Verificar conexión ANTES de abrir popup
    const s = await sw({ type: 'SESSION.GET' });
    const API_BASE = s?.data?.config?.API_BASE;
    
    if (!API_BASE) {
      hideLoader();
      toast('❌ Config inválida (API_BASE)', 'err');
      return;
    }
    
    // Ping al servidor para verificar conexión
    try {
      const pingRes = await fetch(`${API_BASE}/api/version`, { 
        method: 'GET',
        signal: AbortSignal.timeout(5000) // 5s timeout
      });
      
      if (!pingRes.ok) {
        throw new Error('Servidor no disponible');
      }
    } catch (pingErr) {
      hideLoader();
      const errMsg = String(pingErr?.message || pingErr).toLowerCase();
      
      if (errMsg.includes('timeout')) {
        toast('⏱️ Servidor no responde. Intenta más tarde.', 'err');
      } else if (errMsg.includes('fetch') || errMsg.includes('network')) {
        toast('❌ Sin conexión al servidor. Verifica tu internet.', 'err');
      } else {
        toast('❌ No se puede conectar al servidor', 'err');
      }
      return;
    }
    
    showLoader('Abriendo Google…', {
      timeout: 30000,
      onTimeout: () => toast('⏱️ Tiempo agotado abriendo Google', 'error', 5000)
    });
    
    const stored = await stGet(['session.device_id']);
    const dev = stored['session.device_id'];
    const did = dev || crypto.randomUUID();
    
    if (!dev) {
      await stSet({ 'session.device_id': did });
    }

    const params = new URLSearchParams({
      origin: location.origin,
      device_id: did
    });
    
    const res = await fetch(`${API_BASE}/api/auth/google/login?${params}`);
    const url = await res.text();
    
    window.open(url, 'ml_google', 'width=500,height=650');
    toast('Popup Google abierto', 'ok');
    
  } catch (err) {
    logger.error('[AUTH] Error en Google login:', err);
    const errorMsg = String(err?.message || err).toLowerCase();
    
    if (errorMsg.includes('fetch') || errorMsg.includes('network')) {
      toast('❌ Sin conexión al servidor. Verifica tu internet.', 'err');
    } else if (errorMsg.includes('timeout')) {
      toast('⏱️ Tiempo de espera agotado. Intenta nuevamente.', 'err');
    } else {
      toast('❌ Error al abrir Google login', 'err');
    }
  } finally {
    hideLoader();
  }
}

/* =============================================
   Listener de mensaje de popup Google → tokens
   ============================================= */
function setupGooglePopupListener() {
  window.addEventListener('message', async (ev) => {
    try {
      const base = AUTH?.config?.API_BASE || 
                   (await sw({ type: 'SESSION.GET' }))?.data?.config?.API_BASE;
      const allowed = base ? new URL(base).origin : '';
      
      if (!allowed || ev.origin !== allowed) return;
      
      const d = ev.data || {};
      
      // ✅ v10.5.126: Manejar error de cuenta suspendida
      if (d.error === 'account_suspended') {
        hideLoader();
        await showModal({
          title: '⛔ Cuenta suspendida',
          html: `Tu cuenta ha sido suspendida.<br><br><strong>Motivo:</strong> ${d.reason || 'Actividad sospechosa detectada'}<br><br>Si crees que es un error, contacta con soporte.`,
          buttons: [{ text: 'Entendido', value: 'ok', primary: true }]
        });
        return;
      }
      
      if (!(d && (d.access || d.token))) return;
      
      const access = d.access || d.token;
      const refresh = d.refresh ?? null;
      const exp = d.exp ?? 0;
      const isNewUser = d.is_new_user || false;  // 🆕 Detectar si es usuario nuevo
      const lastSelectedAccountId = d.last_selected_account_id || null;  // ✅ v6.5.1: Última cuenta
      
      
      // 🔄 Mostrar loader mientras se procesa el login
      showLoader('Iniciando sesión...', {
        timeout: 30000,
        onTimeout: () => toast('⏱️ Tiempo agotado iniciando sesión', 'error', 5000)
      });
      
      const r = await sw({ 
        type: 'SESSION.LOGIN_GOOGLE_TOKENS', 
        access, 
        refresh, 
        exp 
      });
      
      if (r?.ok) {
        showLoader('Configurando cuenta...', {
          timeout: 30000,
          onTimeout: () => toast('⏱️ Tiempo agotado configurando cuenta', 'error', 5000)
        });
        
        // ✅ Limpiar storage del usuario anterior
        await stRemove([
          'selected_account_id',
          'account_id',
          'accounts',
          'listings_cache',
          'stats_cache'
        ]).catch(() => {});
        
        // ✅ Obtener usuario actual y actualizar AUTH
        const meResp = await sw({ type: 'API.FETCH_JSON', url: '/api/me', method: 'GET' });
        if (meResp?.ok && meResp.data) {
          set_AUTH({
            state: 'AUTHENTICATED',
            user: meResp.data,
            config: AUTH?.config || {},
            on: true
          });
          
          setAuthUI(true, meResp.data.email || '');
          
          // ✅ v2.2.3: Usar flujo centralizado post-login
          const success = await onLoginSuccess({
            isNewUser,
            lastSelectedAccountId,
            provider: 'google'
          });
          
          if (!success) {
            hideLoader();
            return;  // Versión desactualizada, modal ya mostrado
          }
        }
        
        hideLoader();  // ✅ Ocultar loader después de completar
      } else {
        hideLoader();  // ✅ Ocultar loader en error
        toast('Error consolidando el login', 'err');
      }
      
      // ❌ NO intentar cerrar popup - causa error COOP en consola
      // El popup se cierra solo después del redirect o el usuario lo cierra manualmente
      
    } catch (err) {
      hideLoader();  // ✅ Ocultar loader en excepción
    }
  });
}

/* =========
   init()
   ========= */
let _inited = false;

export async function init() {
  if (_inited) return;
  _inited = true;

  // ✅ Checkbox "recordar" eliminado - backend maneja persistencia de sesión JWT

  // Rellenar historial de emails
  await fillEmailDatalist();

  // Cargar último login
  const { last_login, remember_checked } = await stGet(['last_login', 'remember_checked']);
  if (last_login?.email && inEmail) inEmail.value = last_login.email;
  if (last_login?.pass && inPass) inPass.value = last_login.pass;
  
  // ✅ v6.3.0: Restaurar estado del checkbox "Recordarme"
  if (remember && remember_checked !== undefined) {
    remember.checked = remember_checked;
  }
  
  // ✅ v6.3.0: Guardar estado del checkbox cuando cambia
  if (remember) {
    remember.addEventListener('change', () => {
      stSet({ remember_checked: remember.checked });
    });
  }

  // Autocompletar password al cambiar email
  if (inEmail) {
    inEmail.addEventListener('change', () => {
      autofillPassForEmail(inEmail.value.trim());
    });
  }

  // Listener botón login
  if (btnLogin) {
    btnLogin.addEventListener('click', doLogin);
  }

  // Listener botón Google
  if (btnGoogle) {
    btnGoogle.addEventListener('click', doGoogleLogin);
  }

  // Listener de popup Google
  setupGooglePopupListener();

  // Cargar estado AUTH inicial desde SW
  const s = await sw({ type: 'SESSION.GET' });
  const newAuth = {
    state: s?.data?.state || 'UNAUTHENTICATED',
    user: s?.data?.user || null,
    config: s?.data?.config || { API_BASE: '' },
    on: false
  };
  set_AUTH(newAuth);

  // Pintar UI
  setAuthUI(
    newAuth.state === 'AUTHENTICATED', 
    newAuth.user?.email || ''
  );

  // Si está autenticado, cargar cuentas
  if (newAuth.state === 'AUTHENTICATED') {
    await loadAccounts();
    
    // ✅ Activar filtro ACTIVE por defecto
    try {
      const { highlightActiveFilter } = await import('./listings/filters.js');
      const { FILTERS } = await import('./listings/config.js');
      highlightActiveFilter(FILTERS.ACTIVE);
    } catch (err) {
    }
    
    // ✅ Cargar y aplicar preferencias UI si ya está autenticado
    await loadAndApplyUserPreferences();
  }
}