/**
 * Configuración centralizada de la API
 * @module panel/api-config
 * @version 7.4.2
 * 
 * Fuente única de verdad para API_BASE.
 * Se inicializa una vez desde session.config en boot.js
 * Incluye fallback seguro para evitar errores en init temprano
 */

// ============================================
// Estado privado
// ============================================

let _apiBase = '';
let _initialized = false;

// ============================================
// API pública
// ============================================

/**
 * Inicializa la configuración desde session
 * @param {Object} sessionConfig - session.config desde storage
 * @throws {Error} Si ya está inicializado
 */
export function initApiConfig(sessionConfig) {
  if (_initialized) {
    console.warn('[API-CONFIG] Ya está inicializado, ignorando re-inicialización');
    return;
  }
  
  if (!sessionConfig?.API_BASE) {
    throw new Error('[API-CONFIG] session.config.API_BASE es requerido');
  }
  
  _apiBase = sessionConfig.API_BASE.replace(/\/$/, ''); // Quitar / final
  _initialized = true;
  
  console.log('[API-CONFIG] ✅ Inicializado:', { apiBase: _apiBase });
}

/**
 * Obtiene API_BASE
 * @returns {string} Base URL de la API
 * @throws {Error} Si no está inicializado (solo en modo estricto)
 */
export function getApiBase() {
  if (!_initialized) {
    // ✅ FALLBACK SEGURO: usar URL por defecto si no está inicializado
    // Esto evita errores cuando credits.js o render.js cargan antes de auth.js
    console.warn('[API-CONFIG] ⚠️ No inicializado, usando fallback https://api.mitiklive.com');
    return 'https://api.mitiklive.com';
  }
  return _apiBase;
}

/**
 * Verifica si está inicializado
 * @returns {boolean}
 */
export function isInitialized() {
  return _initialized;
}

/**
 * Reset (solo para testing)
 * @private
 */
export function _reset() {
  _apiBase = '';
  _initialized = false;
}
