/*
  Archivo: account-auto-selector.js
  Rol: Auto-selección inteligente de cuentas basada en Wallapop detectado
  
  Sistema híbrido:
  - Detecta cuenta activa en Wallapop
  - Auto-selecciona en combo si existe en backend
  - Pregunta para crear si no existe
  - Sincroniza automáticamente al cambiar de cuenta
*/

import { logger } from './logger.js';
import { toast } from '../utils.js';
import { showModal } from './auth.js';

// Estado
let isInitialized = false;
let lastDetectedProfile = null;
let checkInterval = null;
let userManuallySelected = false;  // ✅ v4.80.3: Flag para respetar selección manual

// ============================================
// INICIALIZACIÓN
// ============================================

/**
 * Iniciar sistema de auto-selección
 */
export function initAccountAutoSelector() {
  if (isInitialized) {
    return;
  }
  
  isInitialized = true;
  
  // ✅ v4.80.3: Listener para detectar selección manual
  const selAccount = document.getElementById('sel-account');
  if (selAccount) {
    selAccount.addEventListener('change', (e) => {
      // Marcar que el usuario seleccionó manualmente
      if (!e.isTrusted) {
        // Evento programático (auto-selección), no marcar
        return;
      }
      
      // Si el usuario selecciona manualmente, marcar flag
      userManuallySelected = true;
    });
  }
  
  // Check inicial (solo si no hay selección manual)
  checkAndAutoSelect();
  
  // Check cada 10 segundos
  checkInterval = setInterval(() => {
    checkAndAutoSelect();
  }, 10000);
  
}

/**
 * Detener sistema
 */
export function stopAccountAutoSelector() {
  if (checkInterval) {
    clearInterval(checkInterval);
    checkInterval = null;
    isInitialized = false;
  }
}

// ============================================
// DETECCIÓN Y AUTO-SELECCIÓN
// ============================================

/**
 * Verificar y auto-seleccionar cuenta
 */
async function checkAndAutoSelect() {
  try {
    // 1. Detectar perfil de Wallapop
    const wallapopProfile = await detectWallapopProfile();
    
    if (!wallapopProfile) {
      lastDetectedProfile = null;
      return;
    }
    
    // 2. Verificar si cambió de cuenta
    const hasChanged = hasProfileChanged(wallapopProfile);
    if (!hasChanged) {
      return;
    }
    
    lastDetectedProfile = wallapopProfile;
    
    // 3. Buscar cuenta en backend
    const backendAccounts = await fetchUserAccounts();
    
    if (!backendAccounts || backendAccounts.length === 0) {
      // Preguntar si quiere crear esta cuenta
      await promptCreateAccount(wallapopProfile);
      return;
    }
    
    // 4. Buscar match por walla_user_id o profile_url
    const match = findMatchingAccount(backendAccounts, wallapopProfile);
    
    if (match) {
      // ✅ Cuenta existe: Auto-seleccionar
      await autoSelectAccount(match.id, match.alias);
    } else {
      // ⚠️ Cuenta NO existe: Preguntar si crear
      await promptCreateAccount(wallapopProfile);
    }
    
  } catch (error) {
    logger.error('[AUTO-SELECT] Error en checkAndAutoSelect:', error);
  }
}

/**
 * Detectar perfil activo en Wallapop
 * @returns {Promise<{walla_user_id: string, profile_url: string, alias: string, avatar: string}|null>}
 */
async function detectWallapopProfile() {
  try {
    // Buscar pestaña de Wallapop
    const tabs = await chrome.tabs.query({
      url: ['https://*.wallapop.com/*', 'https://*.wallapop.es/*']
    });
    
    if (!tabs || tabs.length === 0) {
      return null;
    }
    
    // Priorizar pestaña activa
    const tab = tabs.find(t => t.active) || tabs[0];
    
    // ✅ v10.5.82: Pedir perfil al content script (silenciar lastError)
    const response = await new Promise((resolve) => {
      chrome.tabs.sendMessage(tab.id, { type: 'DOM.CURRENT_PROFILE' }, (r) => {
        void chrome.runtime.lastError;
        resolve(r);
      });
    });
    
    if (!response || !response.ok) {
      return null;
    }
    
    // ✅ v10.5.76: Obtener ID alfanumérico del __NEXT_DATA__ (más confiable)
    let wallaUserId = null;
    if (response.profileUrl) {
      try {
        const userIdResponse = await new Promise((resolve) => {
          chrome.runtime.sendMessage(
            { type: 'WALLA.GET_USER_ID', profileUrl: response.profileUrl },
            (r) => {
              void chrome.runtime.lastError;
              resolve(r);
            }
          );
        });
        if (userIdResponse?.ok && userIdResponse?.walla_user_id) {
          wallaUserId = userIdResponse.walla_user_id;
        }
      } catch {
        // Fallback al ID numérico de la URL
      }
    }
    
    // Fallback: extraer ID numérico de la URL
    if (!wallaUserId) {
      wallaUserId = extractWallaUserId(response.profileUrl);
    }
    
    if (!wallaUserId && !response.alias) {
      return null;
    }
    
    return {
      walla_user_id: wallaUserId || null,
      profile_url: response.profileUrl || null,
      alias: response.alias || 'Usuario',
      avatar: response.avatar || null
    };
    
  } catch (error) {
    return null;
  }
}

/**
 * Extraer walla_user_id de la URL del perfil
 * Ej: https://es.wallapop.com/user/juan-123456 → 123456
 */
function extractWallaUserId(profileUrl) {
  if (!profileUrl) return null;
  
  try {
    const url = new URL(profileUrl);
    const segments = url.pathname.split('/').filter(Boolean);
    const lastSegment = segments[segments.length - 1];
    
    // Formato: nombre-123456 o solo 123456
    const match = lastSegment.match(/-(\d+)$/);
    if (match) {
      return match[1];
    }
    
    // Si es solo números
    if (/^\d+$/.test(lastSegment)) {
      return lastSegment;
    }
    
    return null;
  } catch {
    return null;
  }
}

/**
 * Verificar si el perfil cambió
 */
function hasProfileChanged(newProfile) {
  if (!lastDetectedProfile) return true;
  
  // Comparar por walla_user_id (más confiable)
  if (newProfile.walla_user_id && lastDetectedProfile.walla_user_id) {
    return newProfile.walla_user_id !== lastDetectedProfile.walla_user_id;
  }
  
  // Fallback: comparar por profile_url
  if (newProfile.profile_url && lastDetectedProfile.profile_url) {
    return newProfile.profile_url !== lastDetectedProfile.profile_url;
  }
  
  // Fallback: comparar por alias
  return newProfile.alias !== lastDetectedProfile.alias;
}

/**
 * Obtener cuentas del backend
 */
async function fetchUserAccounts() {
  try {
    const response = await new Promise((resolve) => {
      chrome.runtime.sendMessage(
        { type: 'ACCOUNTS.LIST' },
        (result) => resolve(result)
      );
    });
    
    if (!response || !response.ok) {
      return [];
    }
    
    return response.data || [];
  } catch (error) {
    logger.error('[AUTO-SELECT] Error obteniendo cuentas:', error);
    return [];
  }
}

/**
 * Buscar cuenta que coincida con el perfil de Wallapop
 */
function findMatchingAccount(backendAccounts, wallapopProfile) {
  // 1. Match por walla_user_id (más confiable)
  if (wallapopProfile.walla_user_id) {
    const match = backendAccounts.find(
      acc => acc.walla_user_id === wallapopProfile.walla_user_id
    );
    if (match) return match;
  }
  
  // 2. Match por profile_url
  if (wallapopProfile.profile_url) {
    const match = backendAccounts.find(
      acc => acc.profile_url === wallapopProfile.profile_url
    );
    if (match) return match;
  }
  
  // 3. Match por alias (menos confiable, puede haber duplicados)
  if (wallapopProfile.alias) {
    const match = backendAccounts.find(
      acc => acc.alias === wallapopProfile.alias
    );
    if (match) return match;
  }
  
  return null;
}

/**
 * Auto-seleccionar cuenta en el combo
 */
async function autoSelectAccount(accountId, alias) {
  try {
    // ✅ v4.80.3: Respetar selección manual del usuario
    if (userManuallySelected) {
      return;
    }
    
    const selAccount = document.getElementById('sel-account');
    if (!selAccount) {
      return;
    }
    
    // Buscar la opción que coincida
    const options = Array.from(selAccount.options);
    const matchingOption = options.find(opt => 
      opt.value == accountId || 
      opt.dataset.accountId == accountId
    );
    
    if (!matchingOption) {
      return;
    }
    
    // Seleccionar
    selAccount.value = matchingOption.value;
    
    // Guardar en storage
    await chrome.storage.local.set({ selected_account_id: accountId });
    
    // Disparar evento change
    selAccount.dispatchEvent(new Event('change', { bubbles: true }));
    
    // Notificar al usuario (toast discreto)
    toast(`✅ Cuenta "${alias}" seleccionada`, 'success', 2000);
    
    
  } catch (error) {
    logger.error('[AUTO-SELECT] Error auto-seleccionando:', error);
  }
}

/**
 * Preguntar al usuario si quiere crear la cuenta detectada
 */
async function promptCreateAccount(wallapopProfile) {
  try {
    
    const choice = await showModal({
      title: '🆕 Nueva cuenta detectada en Wallapop',
      html: `
        <p>Detectamos la cuenta <strong>"${wallapopProfile.alias}"</strong> en Wallapop.</p>
        <p class="modal-text-secondary">
          ¿Quieres añadirla a tu backend para gestionar sus anuncios?
        </p>
      `,
      buttons: [
        { text: 'Ahora no', value: 'no' },
        { text: 'Sí, añadir', value: 'yes', primary: true }
      ]
    });
    
    if (choice !== 'yes') {
      toast('ℹ️ Puedes añadir la cuenta después manualmente', 'info', 3000);
      return;
    }
    
    // Crear cuenta en backend
    const newAccount = await createWallapopAccount(wallapopProfile);
    
    if (!newAccount) {
      toast('❌ Error al crear la cuenta', 'error');
      return;
    }
    
    
    // Recargar lista de cuentas
    await reloadAccounts();
    
    // Auto-seleccionar la nueva cuenta
    setTimeout(async () => {
      await autoSelectAccount(newAccount.id, newAccount.alias);
    }, 500);
    
  } catch (error) {
    logger.error('[AUTO-SELECT] Error en promptCreateAccount:', error);
    toast('❌ Error al crear la cuenta', 'error');
  }
}

/**
 * Crear cuenta de Wallapop en el backend
 */
async function createWallapopAccount(profileData) {
  try {
    const response = await new Promise((resolve) => {
      chrome.runtime.sendMessage(
        {
          type: 'ACCOUNT.CREATE',
          data: {
            walla_user_id: profileData.walla_user_id,
            profile_url: profileData.profile_url,
            alias: profileData.alias
          }
        },
        (result) => resolve(result)
      );
    });
    
    if (!response || !response.ok) {
      logger.error('[AUTO-SELECT] Error creando cuenta:', response);
      return null;
    }
    
    return response.data;
    
  } catch (error) {
    logger.error('[AUTO-SELECT] Error en createWallapopAccount:', error);
    return null;
  }
}

/**
 * Recargar lista de cuentas
 */
async function reloadAccounts() {
  try {
    const { loadAccounts } = await import('./auth.js');
    await loadAccounts();
  } catch (error) {
    logger.error('[AUTO-SELECT] Error recargando cuentas:', error);
  }
}

// ============================================
// API PÚBLICA
// ============================================

/**
 * Forzar check manual (útil para testing)
 */
export async function forceCheck() {
  await checkAndAutoSelect();
}

/**
 * ✅ v4.80.3: Marcar que el usuario seleccionó manualmente
 * Esto evita que el auto-selector cambie la cuenta
 */
export function setManualSelection(isManual = true) {
  userManuallySelected = isManual;
}

/**
 * ✅ v4.80.3: Resetear flag de selección manual
 * Útil para login inicial
 */
export function resetManualSelection() {
  userManuallySelected = false;
}

/**
 * ✅ v4.80.3: Obtener estado actual del flag
 */
export function isManuallySelected() {
  return userManuallySelected;
}

/**
 * Obtener último perfil detectado
 */
export function getLastDetectedProfile() {
  return lastDetectedProfile;
}
