/**
 * Sistema de Logging Centralizado con Control de Usuario
 * ======================================================
 * 
 * Permite al usuario controlar el nivel de logs desde el panel.
 * Por defecto: solo errores (production)
 * Con debug activado: logs completos
 * 
 * Sanitiza automáticamente datos sensibles (tokens, passwords, emails parciales)
 */

class Logger {
  constructor() {
    this.level = 'error';  // Por defecto: solo errores
    this.init();
  }

  async init() {
    await this.loadPreferences();
    this.watchChanges();
  }

  async loadPreferences() {
    try {
      const result = await chrome.storage.local.get(['debugMode']);
      this.level = result.debugMode ? 'debug' : 'error';
    } catch (err) {
      this.level = 'error';
    }
  }

  watchChanges() {
    if (chrome.storage?.onChanged) {
      chrome.storage.onChanged.addListener((changes) => {
        if (changes.debugMode) {
          this.level = changes.debugMode.newValue ? 'debug' : 'error';
        }
      });
    }
  }

  /**
   * Sanitiza datos sensibles antes de loggear
   */
  sanitize(data) {
    if (typeof data === 'string') {
      return this.sanitizeString(data);
    }
    
    if (Array.isArray(data)) {
      return data.map(item => this.sanitize(item));
    }
    
    if (data && typeof data === 'object') {
      const sanitized = {};
      for (const [key, value] of Object.entries(data)) {
        const lowerKey = key.toLowerCase();
        
        // Campos sensibles a ocultar completamente
        if (lowerKey.includes('password') || 
            lowerKey.includes('secret') || 
            lowerKey.includes('apikey') ||
            lowerKey.includes('api_key')) {
          sanitized[key] = '***HIDDEN***';
          continue;
        }
        
        // Tokens: mostrar solo primeros/últimos caracteres
        if (lowerKey.includes('token') || 
            lowerKey.includes('jwt') || 
            lowerKey.includes('bearer') ||
            lowerKey.includes('authorization')) {
          sanitized[key] = this.maskToken(value);
          continue;
        }
        
        // Email: ocultar parte del dominio
        if (lowerKey.includes('email')) {
          sanitized[key] = this.maskEmail(value);
          continue;
        }
        
        // Recursivo para objetos anidados
        sanitized[key] = this.sanitize(value);
      }
      return sanitized;
    }
    
    return data;
  }

  sanitizeString(str) {
    if (!str) return str;
    
    // Buscar tokens JWT en strings
    const jwtPattern = /eyJ[A-Za-z0-9_-]+\.eyJ[A-Za-z0-9_-]+\.[A-Za-z0-9_-]+/g;
    str = str.replace(jwtPattern, (match) => this.maskToken(match));
    
    // Buscar Bearer tokens
    const bearerPattern = /Bearer\s+[A-Za-z0-9_-]+/gi;
    str = str.replace(bearerPattern, (match) => {
      const token = match.split(' ')[1];
      return `Bearer ${this.maskToken(token)}`;
    });
    
    return str;
  }

  maskToken(token) {
    if (!token || typeof token !== 'string') return token;
    if (token.length < 20) return '***';
    
    // Mostrar primeros 8 y últimos 8 caracteres
    return `${token.substring(0, 8)}...${token.substring(token.length - 8)}`;
  }

  maskEmail(email) {
    if (!email || typeof email !== 'string') return email;
    
    const parts = email.split('@');
    if (parts.length !== 2) return email;
    
    const [user, domain] = parts;
    const maskedUser = user.length > 2 
      ? `${user[0]}***${user[user.length - 1]}`
      : '***';
    
    return `${maskedUser}@${domain}`;
  }

  /**
   * Formatea argumentos para logging
   */
  formatArgs(args) {
    return args.map(arg => {
      if (arg instanceof Error) {
        return {
          message: arg.message,
          stack: arg.stack
        };
      }
      return this.sanitize(arg);
    });
  }

  debug(...args) {
    if (this.level === 'debug') {
    }
  }

  info(...args) {
    if (['debug', 'info'].includes(this.level)) {
    }
  }

  warn(...args) {
    if (['debug', 'info', 'warn'].includes(this.level)) {
    }
  }

  error(...args) {
    // Errores siempre se muestran
    console.error('[❌ ERROR]', ...this.formatArgs(args));
  }

  /**
   * Log de grupo (colapsable en consola)
   */
  group(title, collapsed = true) {
    if (this.level === 'debug') {
      if (collapsed) {
        console.groupCollapsed(`[📦] ${title}`);
      } else {
        console.group(`[📦] ${title}`);
      }
    }
  }

  groupEnd() {
    if (this.level === 'debug') {
      console.groupEnd();
    }
  }

  /**
   * Log condicional (solo cada N veces)
   */
  every(n, key, ...args) {
    if (!this._counters) this._counters = {};
    if (!this._counters[key]) this._counters[key] = 0;
    
    this._counters[key]++;
    
    if (this._counters[key] % n === 0) {
      this.debug(`[${key} #${this._counters[key]}]`, ...args);
    }
  }
}

// Singleton
export const log = new Logger();

// Alias para compatibilidad
export default log;
