/*  Cabecera obligatoria: NO BORRAR NI MODIFICAR este bloque inicial en ningún fichero.
    Archivo: dom.js – Rol: Helpers DOM reutilizables (perfil/sesión, waits, clicks, dropdowns)
    Descripcion de todo el proyecto:
         MitikLive Wallapop (MV3): panel lateral en Wallapop para gestionar anuncios conectado a FastAPI
            (login JWT, export/import, backups/renovaciones).
    Aviso: NO BORRAR los comentarios descriptivos situados encima de cada función, solo cambiarlos si se modifica la funcion.
    En Rol (linea mas arriba): si está vacía o el fichero se modifica o reestructura, modificar esa linea de rol
    
    ✅ v10.5.31: Fusionado con dom-optimized.js - waitFor con MutationObserver integrado
*/

import { sleep, normalize, cssEscape, aliasFromProfileUrl, nearError } from './utils.js';

/* ============================================================
   0) Helpers locales
   ============================================================ */
const isVisible = (el) => {
  if (!el) return false;
  const s = window.getComputedStyle(el);
  const r = el.getBoundingClientRect();
  return s.display !== 'none' && s.visibility !== 'hidden' && r.width > 0 && r.height > 0;
};

/* ============================================================
   1) waitFor - Espera optimizada con MutationObserver
   
   ✅ v10.5.31: Fusionado desde dom-optimized.js
   
   Ventajas vs polling tradicional:
   - 0% CPU cuando no hay cambios en DOM
   - Reacción instantánea (<5ms vs 100-200ms)
   - No sobrecarga en PCs lentos
   - Fallback automático a polling si falla
   ============================================================ */

/**
 * Espera a que una función retorne un valor truthy usando MutationObserver
 * 
 * @param {Function} checkFn - Función que retorna el elemento/valor esperado
 * @param {Object} options - Opciones de configuración
 * @param {number} options.timeout - Timeout máximo en ms (default: 10000)
 * @param {number} options.interval - Intervalo de polling fallback (default: 200)
 * @param {boolean} options.useMutationObserver - Usar MutationObserver (default: true)
 * @returns {Promise} Valor retornado por checkFn o null si timeout
 */
export async function waitFor(checkFn, { 
  timeout = 10000, 
  interval = 200,
  useMutationObserver = true 
} = {}) {
  
  // 1. Check inmediato - si ya existe, retornar
  const immediate = checkFn();
  if (immediate) return immediate;
  
  // 2. Si MutationObserver está disponible y habilitado, usarlo
  if (useMutationObserver && typeof MutationObserver !== 'undefined') {
    try {
      return await waitForWithObserver(checkFn, timeout, interval);
    } catch (err) {
      // Si falla MutationObserver, caer a polling
    }
  }
  
  // 3. Fallback: Polling tradicional
  return await waitForWithPolling(checkFn, timeout, interval);
}

/**
 * Espera usando MutationObserver (eficiente, 0% CPU idle)
 * @private
 */
async function waitForWithObserver(checkFn, timeout, pollInterval) {
  return new Promise((resolve, reject) => {
    let timeoutId;
    let observer;
    let pollIntervalId;
    
    const cleanup = () => {
      if (timeoutId) clearTimeout(timeoutId);
      if (observer) observer.disconnect();
      if (pollIntervalId) clearInterval(pollIntervalId);
    };
    
    const check = () => {
      try {
        const result = checkFn();
        if (result) {
          cleanup();
          resolve(result);
          return true;
        }
      } catch (err) {
        // Si checkFn falla, ignorar y continuar
      }
      return false;
    };
    
    // Observer para cambios en el DOM
    observer = new MutationObserver((mutations) => {
      // Solo chequear si hubo cambios relevantes
      const hasRelevantChanges = mutations.some(m => 
        m.type === 'childList' || 
        m.type === 'attributes'
      );
      
      if (hasRelevantChanges) {
        check();
      }
    });
    
    // Observar desde document.body (todo el DOM visible)
    try {
      observer.observe(document.body || document.documentElement, {
        childList: true,
        subtree: true,
        attributes: true,
        attributeFilter: ['class', 'style', 'hidden', 'disabled', 'aria-expanded', 'aria-selected']
      });
    } catch (err) {
      cleanup();
      reject(new Error(`Observer setup failed: ${err.message}`));
      return;
    }
    
    // Polling lento como backup (por si observer pierde algo)
    // Cada 500ms en lugar de cada 100-200ms = 60-75% menos CPU
    pollIntervalId = setInterval(() => {
      check();
    }, Math.max(pollInterval * 2.5, 500));
    
    // Timeout
    timeoutId = setTimeout(() => {
      cleanup();
      resolve(null); // No rechazar, retornar null
    }, timeout);
  });
}

/**
 * Espera usando polling tradicional (fallback)
 * @private
 */
async function waitForWithPolling(checkFn, timeout, interval) {
  const startTime = Date.now();
  
  while (Date.now() - startTime < timeout) {
    try {
      const result = checkFn();
      if (result) return result;
    } catch (err) {
      // Si checkFn falla, ignorar y continuar
    }
    
    await new Promise(resolve => setTimeout(resolve, interval));
  }
  
  return null; // Timeout alcanzado
}

/**
 * Espera a que aparezca un elemento (simplificado)
 * ✅ v10.5.31: Fusionado desde dom-optimized.js
 * 
 * @param {string} selector - Selector CSS del elemento
 * @param {Object} options - Opciones (timeout, interval, useMutationObserver)
 * @returns {Promise<Element|null>} Elemento encontrado o null
 */
export async function waitForElement(selector, options = {}) {
  return waitFor(() => document.querySelector(selector), options);
}

/**
 * Espera a que elemento sea visible
 * ✅ v10.5.31: Fusionado desde dom-optimized.js
 * 
 * @param {string|Element} selectorOrElement - Selector CSS o elemento
 * @param {Object} options - Opciones
 * @returns {Promise<Element|null>} Elemento visible o null
 */
export async function waitForVisible(selectorOrElement, options = {}) {
  const element = typeof selectorOrElement === 'string'
    ? await waitForElement(selectorOrElement, options)
    : selectorOrElement;
  
  if (!element) return null;
  
  return waitFor(() => {
    const rect = element.getBoundingClientRect();
    const style = getComputedStyle(element);
    
    const isVisibleCheck = 
      style.display !== 'none' &&
      style.visibility !== 'hidden' &&
      style.opacity !== '0' &&
      rect.width > 0 &&
      rect.height > 0;
    
    return isVisibleCheck ? element : null;
  }, options);
}

/**
 * Espera a que elemento tenga atributo específico
 * ✅ v10.5.31: Fusionado desde dom-optimized.js
 * 
 * @param {Element} element - Elemento a monitorear
 * @param {string} attrName - Nombre del atributo
 * @param {string} expectedValue - Valor esperado
 * @param {Object} options - Opciones
 * @returns {Promise<Element|null>}
 */
export async function waitForAttribute(element, attrName, expectedValue, options = {}) {
  if (!element) return null;
  
  return waitFor(() => {
    const value = element.getAttribute(attrName);
    return value === expectedValue ? element : null;
  }, options);
}

/**
 * Espera a que desaparezca un loader
 * ✅ v10.5.31: Fusionado desde dom-optimized.js
 * 
 * @param {string} containerSelector - Selector del contenedor (default: 'body')
 * @param {Object} options - Opciones
 * @returns {Promise<boolean>} true cuando no hay loaders visibles
 */
export async function waitForNoLoader(containerSelector = 'body', options = {}) {
  return waitFor(() => {
    const container = document.querySelector(containerSelector);
    if (!container) return null;
    
    const loaders = container.querySelectorAll('[class*="loader"], [class*="loading"], [class*="spinner"]');
    const hasVisibleLoader = Array.from(loaders).some(l => {
      const style = getComputedStyle(l);
      return style.display !== 'none' && 
             style.visibility !== 'hidden' && 
             style.opacity !== '0';
    });
    
    return !hasVisibleLoader ? true : null;
  }, options);
}

/**
 * Espera a que una condición se cumpla con retry
 * ✅ v10.5.31: Fusionado desde dom-optimized.js
 * 
 * @param {Function} checkFn - Función que retorna true/false
 * @param {Object} options - Opciones
 * @returns {Promise<boolean>}
 */
export async function waitForCondition(checkFn, options = {}) {
  const result = await waitFor(() => {
    return checkFn() ? true : null;
  }, options);
  
  return result === true;
}

/* ============================================================
   2) Utilidades básicas de sesión/perfil (no navegan)
   ============================================================ */

/** Heurística de sesión iniciada (no navega). */
export async function isLoggedInWallapop() {
  try {
    if (document.querySelector('img[data-testid="user-avatar"], img[class*="Header__avatar"], img[class*="avatar"]')) {
      return true;
    }
    if ([...document.querySelectorAll('a[href*="/user/"], a[href*="/profile"]')]
        .some(a => /perfil|profile/i.test(a.textContent || ''))) {
      return true;
    }
  } catch {}
  return false;
}

/** Espera a que haya listado de items visible (no navega). */
export async function waitItemsListed({ timeoutMs = 20000 } = {}) {
  const hasList = () =>
    document.querySelector(
      '[data-testid="item-card"], [data-testid*="product"], a[href*="/item/"], .ItemCard a[href*="/item/"], [class*="Card"] a[href*="/item/"]'
    );
  if (hasList()) return hasList();
  return await waitFor(() => hasList(), { timeout: timeoutMs, interval: 150 });
}

/** Intenta leer el avatar en un documento dado. */
function pickAvatarFromDoc(doc = document) {
  try {
    const sel = [
      'img[data-testid="user-avatar"]',
      'header img.avatar-rounded',
      'header img[class*="avatar"]',
      'img.avatar-rounded',
      'img[class*="avatar"][src*="cdn.wallapop.com"]',
      'img[class*="avatar"][src*="cloudfront"]'
    ].join(', ');
    const img = doc.querySelector(sel);
    const src = img?.currentSrc || img?.src || null;
    return (src && !src.startsWith('data:')) ? src : null;
  } catch { return null; }
}

/**
 * Resuelve la URL del perfil público sin navegar (SPA) creando un iframe a /app/profile/info
 * y leyendo el enlace al perfil (mismo origen ⇒ sin CORS/CSP).
 */
export async function resolvePublicProfileUrlViaIframe() {
  const toAbs = (h) => h?.startsWith('http') ? h : new URL(h, location.origin).toString();
  
  // Selectores para encontrar el enlace al perfil público
  const SELECTORS = [
    'a[data-testid="public-profile-link"]',
    'a.header-link[href*="/user/"]',
    'a[href^="/user/"]',
    'a[href*="/user/"]'
  ];
  
  // Validar que sea URL de perfil válida: /user/alias-ID
  const isValidProfileUrl = (href) => href && /\/user\/[^\/]+-\d+/.test(href);
  
  // Buscar en documento
  const pick = (doc) => {
    for (const sel of SELECTORS) {
      const a = doc.querySelector(sel);
      if (a) {
        const href = a.getAttribute('href') || a.href || '';
        if (isValidProfileUrl(href)) return toAbs(href);
      }
    }
    return null;
  };

  return new Promise((resolve) => {
    const iframe = document.createElement('iframe');
    iframe.style.cssText = 'display:none;width:0;height:0;border:0;position:absolute;pointer-events:none';
    iframe.src = '/app/profile/info';
    
    let done = false;
    let obs = null;
    let safetyTimer = null;
    
    const finish = (val) => {
      if (done) return;
      done = true;
      if (safetyTimer) clearTimeout(safetyTimer);
      if (obs) try { obs.disconnect(); } catch {}
      try { iframe.remove(); } catch {}
      resolve(val || null);
    };
    
    // Safety timeout: 5 segundos DESPUÉS del load (no desde el inicio)
    const startSafetyTimer = () => {
      safetyTimer = setTimeout(() => finish(null), 5000);
    };
    
    iframe.addEventListener('load', () => {
      try {
        const doc = iframe.contentDocument;
        if (!doc) return finish(null);
        
        // Intento inmediato tras load
        const url = pick(doc);
        if (url) return finish(url);
        
        // Si no está, iniciar safety timer y observar mutaciones
        startSafetyTimer();
        
        obs = new MutationObserver(() => {
          const url = pick(doc);
          if (url) finish(url);
        });
        obs.observe(doc.documentElement, { childList: true, subtree: true });
        
      } catch {
        finish(null);
      }
    });
    
    // Error de carga del iframe
    iframe.addEventListener('error', () => finish(null));
    
    document.body.appendChild(iframe);
  });
}

/**
 * Obtiene { href, alias, avatar } de forma robusta sin navegación dura.
 * ✅ v10.3.8: Simplificado - sin intervalos, solo eventos nativos
 */
export async function getProfileInfoSimple() {
  const href = await resolvePublicProfileUrlViaIframe();
  const alias = href ? aliasFromProfileUrl(href) : null;

  // Avatar del documento actual (header de Wallapop)
  let avatar = pickAvatarFromDoc(document);

  // Si no hay avatar en el header y tenemos href, intentar cargarlo del perfil público
  if (!avatar && href) {
    avatar = await new Promise((resolve) => {
      const ifr = document.createElement('iframe');
      ifr.style.cssText = 'display:none;width:0;height:0;border:0;position:absolute;pointer-events:none';
      ifr.src = href;
      
      let done = false;
      let obs = null;
      let safetyTimer = null;
      
      const finish = (v) => {
        if (done) return;
        done = true;
        if (safetyTimer) clearTimeout(safetyTimer);
        if (obs) try { obs.disconnect(); } catch {}
        try { ifr.remove(); } catch {}
        resolve(v || null);
      };

      ifr.addEventListener('load', () => {
        try {
          const doc = ifr.contentDocument;
          if (!doc) return finish(null);
          
          // Intento inmediato
          const v = pickAvatarFromDoc(doc);
          if (v) return finish(v);
          
          // Safety timer: 3 segundos después del load
          safetyTimer = setTimeout(() => finish(null), 3000);
          
          // Observar mutaciones para cuando aparezca el avatar
          obs = new MutationObserver(() => {
            const v = pickAvatarFromDoc(doc);
            if (v) finish(v);
          });
          obs.observe(doc.documentElement, { childList: true, subtree: true, attributes: true, attributeFilter: ['src'] });
          
        } catch {
          finish(null);
        }
      });
      
      ifr.addEventListener('error', () => finish(null));
      
      document.body.appendChild(ifr);
    });
  }

  return { href, alias, avatar };
}

/**
 * Click “Ver mi perfil público” si existe y devuelve su href.
 * No es obligatorio que navegue: por defecto hace click y retorna el href; si no lo encuentra, intenta vía iframe.
 */
export async function clickPublicProfileLink({ click = true, timeoutMs = 6000 } = {}) {
  const MATCH_TXT = /ver mi perfil p[úu]blico|ver mi perfil|view my public profile/i;
  const SELS = [
    'a[data-testid="public-profile-link"]',
    'a.header-link[href*="/user/"]',
    'header a[href*="/user/"]',
    'a[href*="/user/"]'
  ];

  const pickLocal = () => {
    const all = Array.from(document.querySelectorAll(SELS.join(', ')));
    let a = all.find(x => MATCH_TXT.test(x.textContent || ''));
    if (!a) a = all.find(x => x.href || x.getAttribute('href'));
    return a || null;
  };

  let a = pickLocal();
  if (a) {
    const href = a.getAttribute('href')?.startsWith('http') ? a.getAttribute('href') : (a.href || a.getAttribute('href'));
    if (click) safeClick(a);
    return href || null;
  }

  // Fallback: intenta resolver por iframe y (opcional) navegar
  const href = await resolvePublicProfileUrlViaIframe({ timeoutMs });
  if (href && click) {
    try { location.assign(href); } catch {}
  }
  return href || null;
}

/* ============================================================
   3) Helpers de texto (inputs/textarea/contenteditable)
   ============================================================ */

export function getTextControl(el) {
  if (!el) return null;
  if (el.matches?.('input,textarea,[contenteditable="true"]')) return el;
  return el.querySelector?.('input,textarea,[contenteditable="true"]') || null;
}

/**
 * Obtiene el valor de un elemento del formulario
 * @param {HTMLElement} el - Elemento input, textarea o contenteditable
 * @returns {string} Valor del elemento
 */
export function getVal(el) {
  return el?.isContentEditable ? (el.textContent ?? '') : (el?.value ?? '');
}

/** Espera a que el valor deje de cambiar (útil contra autocompletado/IA). */
export async function waitStableValue(el, { grace = 350, timeout = 4000, interval = 150 } = {}) {
  const ctl = getTextControl(el);
  if (!ctl) return false;
  let last = getVal(ctl);
  const t0 = Date.now();
  while (Date.now() - t0 < timeout) {
    await sleep(interval);
    const now = getVal(ctl);
    if (now === last) {
      await sleep(grace);
      if (getVal(ctl) === now) return true;
    }
    last = now;
  }
  return false;
}

/** Set robusto: dispara input/change y blur. Soporta contenteditable. */
export async function setText(elt, value) {
  const el = getTextControl(elt);
  if (!el) return false;
  try { el.focus?.(); } catch {}
  if (el.isContentEditable) {
    el.textContent = String(value ?? '');
    el.dispatchEvent(new InputEvent('input', { bubbles: true, composed: true }));
    el.dispatchEvent(new Event('change', { bubbles: true, composed: true }));
  } else {
    // ✅ v7.2.29: Escritura rápida 5ms entre letras (90% más rápido que 50ms)
    await setInputValueHuman(el, String(value ?? ''), { typeCharByChar: true, baseDelay: 5 });
  }
  el.dispatchEvent(new Event('blur', { bubbles: true }));
  await sleep(50);
  return true;
}

/** Limpia el campo de texto con eventos «reales». */
export async function clearText(elt) { 
  return setText(elt, ''); 
}

/** Secuencia completa: esperar-estable, limpiar, fijar y revalidar ante re-pegado. */
export async function clearThenSetText(elt, value, { ensure = true } = {}) {
  const el = getTextControl(elt);
  if (!el) return { ok: false, code: 'no_control' };
  
  await waitStableValue(el);
  await clearText(el);
  
  if (value != null && String(value).length) {
    await setText(el, value);
    if (ensure) {
      await sleep(220);
      if (getVal(el) !== String(value)) await setText(el, value);
    }
  }
  return { ok: true, value: getVal(el) };
}

/* ============================================================
   4) Utilidades genéricas: selectores / click / setValue
   ============================================================ */

/** Espera a que seleccione el primer selector que exista (y opcionalmente sea visible). */
export async function waitForSelector(selectors, { timeout = 15000, visible = false, interval = 100 } = {}) {
  const list = Array.isArray(selectors) ? selectors : [selectors];
  const t0 = Date.now();
  while (Date.now() - t0 < timeout) {
    for (const sel of list) {
      const el = document.querySelector(sel);
      if (el && (!visible || isVisible(el))) return el;
    }
    await sleep(interval);
  }
  return null;
}

/** Click seguro: centra en viewport, click nativo + MouseEvent. */
export function safeClick(el) {
  if (!el) return false;
  try { el.scrollIntoView({ block: 'center' }); } catch {}
  try { el.click(); } catch {}
  try { el.dispatchEvent(new MouseEvent('click', { bubbles: true, composed: true })); } catch {}
  return true;
}

/** Set de input/textarea compatible con campos controlados por React. */
/**
 * ✅ v7.2.29: Escribe texto letra a letra (comportamiento humano rápido)
 * @param {HTMLInputElement|HTMLTextAreaElement} el - Input element
 * @param {string} value - Texto a escribir
 * @param {Object} options - Opciones
 * @param {boolean} options.typeCharByChar - Si true, escribe letra a letra (default: true)
 * @param {number} options.baseDelay - Delay base entre letras en ms (default: 5)
 */
export async function setInputValueHuman(el, value, { typeCharByChar = true, baseDelay = 5 } = {}) {
  if (!typeCharByChar || !value || value.length === 0) {
    // Fallback a método rápido
    return setInputValue(el, value);
  }
  
  // Obtener setter nativo
  const proto = el instanceof HTMLTextAreaElement
    ? window.HTMLTextAreaElement.prototype
    : window.HTMLInputElement.prototype;
  const valueSetter = Object.getOwnPropertyDescriptor(proto, 'value')?.set;
  const ownSetter = Object.getOwnPropertyDescriptor(el, 'value')?.set;
  const setter = ownSetter || valueSetter;
  
  // Limpiar campo primero
  if (!setter) el.value = ''; else setter.call(el, '');
  el.dispatchEvent(new Event('input', { bubbles: true, composed: true }));
  
  // Escribir letra a letra
  const text = String(value);
  let currentValue = '';
  
  for (let i = 0; i < text.length; i++) {
    currentValue += text[i];
    
    // Set valor acumulado
    if (!setter) el.value = currentValue; else setter.call(el, currentValue);
    
    // Disparar evento input por cada letra
    el.dispatchEvent(new InputEvent('input', { 
      bubbles: true, 
      composed: true,
      data: text[i],
      inputType: 'insertText'
    }));
    
    // Delay humano variable entre letras (±30%)
    if (i < text.length - 1) { // No delay después de última letra
      const delay = Math.floor(baseDelay * (0.7 + Math.random() * 0.6)); // 35-80ms si base=50
      await sleep(delay);
    }
  }
  
  // Evento change final
  el.dispatchEvent(new Event('change', { bubbles: true, composed: true }));
}

export async function setInputValue(el, value) {
  const proto = el instanceof HTMLTextAreaElement
    ? window.HTMLTextAreaElement.prototype
    : window.HTMLInputElement.prototype;
  const valueSetter = Object.getOwnPropertyDescriptor(proto, 'value')?.set;
  const ownSetter = Object.getOwnPropertyDescriptor(el, 'value')?.set;
  const setter = ownSetter || valueSetter;
  if (!setter) el.value = value; else setter.call(el, value);
  el.dispatchEvent(new Event('input', { bubbles: true, composed: true }));
  el.dispatchEvent(new Event('change', { bubbles: true, composed: true }));
}

/* ============================================================
   5) Retry / Online / Ready
   ============================================================ */

/** Retry con backoff exponencial y jitter. */
export async function retry(fn, {
  retries = 3,
  baseDelay = 250,
  factor = 2,
  jitter = true
} = {}) {
  let lastErr;
  for (let i = 0; i <= retries; i++) {
    try { return await fn(i); } catch (e) { lastErr = e; }
    const ms = baseDelay * (factor ** i);
    const wait = jitter ? ms * (0.7 + Math.random() * 0.6) : ms;
    await sleep(wait);
  }
  throw lastErr;
}

/** ¿Conexión online? (navigator + ping ligero al mismo origen) */
export async function isOnline({ pingPath = '/', timeout = 3000 } = {}) {
  if ('onLine' in navigator && navigator.onLine === false) return false;
  try {
    const ctrl = new AbortController();
    const t = setTimeout(() => ctrl.abort(), timeout);
    const r = await fetch(pingPath, { method: 'HEAD', cache: 'no-store', signal: ctrl.signal });
    clearTimeout(t);
    return !!r.ok || r.type === 'opaque';
  } catch { 
    return false; 
  }
}

/** Garantiza que estamos en /app/catalog/upload y que el step raíz está montado. */
export async function ensureUploadPageReady({ timeout = 20000 } = {}) {
  // ✅ v10.5.109: Logger silenciado para no ensuciar consola de Wallapop
  const log = {
    info: () => {},
    debug: () => {},
    log: () => {},
    warn: console.warn.bind(console),
    error: console.error.bind(console)
  };
  
  // 1) Verificar ruta correcta
  const onUpload = /\/app\/catalog\/upload(?:\/|$)/.test(location.pathname);
  if (!onUpload) {
    log.warn('[DOM] No estamos en /upload, ruta:', location.pathname);
    return { ok: false, code: 'wrong_url' };
  }

  log.info('[DOM] ✓ Ruta correcta: /upload');

  // 2) Esperar a que el DOM esté completamente cargado
  if (document.readyState !== 'complete') {
    log.info('[DOM] Esperando a que document.readyState sea complete...');
    const ok = await waitFor(() => document.readyState === 'complete', { timeout: 10000 });
    if (!ok) {
      log.error('[DOM] Timeout esperando document.readyState complete');
      return { ok: false, code: 'dom_not_ready' };
    }
  }
  
  log.info('[DOM] ✓ Document ready');

  // 3) ✅ NUEVO: Esperar un momento adicional para que React/Vue se monte
  await sleep(800);

  // 4) ✅ CORREGIDO: Más selectores y mejor lógica de búsqueda
  const selectors = [
    '.UploadStepVertical__container',
    'tsl-upload-step-vertical',
    '[data-testid="UploadStepVertical"]',
    'tsl-upload-step-form',
    'form [name="title"]',
    'input[name="title"]',
    '[data-testid="UploadStepForm"]',
    '.UploadStepForm',
    'tsl-upload',
    '[class*="Upload"]',
    '[data-testid*="upload"]'
  ];

  log.info('[DOM] Buscando componente de upload...');

  let host = null;
  const startTime = Date.now();
  
  while (Date.now() - startTime < timeout) {
    for (const selector of selectors) {
      host = document.querySelector(selector);
      if (host) {
        log.info(`[DOM] ✓ Encontrado: ${selector}`);
        break;
      }
    }
    
    if (host) break;
    
    // ✅ NUEVO: Scroll al inicio si tarda más de 5 segundos
    if (Date.now() - startTime > 5000) {
      try {
        window.scrollTo({ top: 0, behavior: 'smooth' });
        await sleep(300);
      } catch (e) {}
    }
    
    await sleep(200);
  }

  if (!host) {
    const bodyHasContent = document.body.children.length > 0;
    const hasReactRoot = document.querySelector('#root, [id*="app"]');
    
    log.error('[DOM] No se encontró componente de upload', {
      bodyHasContent,
      hasReactRoot,
      pathname: location.pathname,
      readyState: document.readyState,
      bodyHTML: document.body.innerHTML.substring(0, 500)
    });
    
    return { ok: false, code: 'upload_step_not_found' };
  }

  log.info('[DOM] ✓ Página de upload lista');
  return { ok: true };
}

/* ============================================================
   6) Búsqueda profunda (shadow-root en cascada)
   ============================================================ */
function deepQueryInRoot(root, selector) {
  // 1) probar en el root actual
  const direct = root.querySelector(selector);
  if (direct) return direct;

  // 2) recorrer hosts con shadowRoot y buscar dentro recursivamente
  const all = root.querySelectorAll('*');
  for (const el of all) {
    const sr = el.shadowRoot;
    if (!sr) continue;
    const found = deepQueryInRoot(sr, selector);
    if (found) return found;
  }
  return null;
}

/** Busca un selector dentro de todo el árbol, incluyendo *shadow-roots* anidados. */
export function queryDeep(root, sel) {
  const host = typeof root === 'string' ? document.querySelector(root) : (root || document);
  if (!host) return null;
  const base = host.shadowRoot || (host.querySelector ? host : document);
  return deepQueryInRoot(base === document ? document : base, sel) || (host.matches?.(sel) ? host : null);
}

/** Espera a que aparezca cualquiera de los selectores (con soporte de shadow en cascada). */
export async function waitForDeep(selectors, { timeout = 15000, interval = 120, visible = false } = {}) {
  const list = Array.isArray(selectors) ? selectors : [selectors];
  const t0 = Date.now();
  while (Date.now() - t0 < timeout) {
    for (const s of list) {
      const el = queryDeep(document, s) || document.querySelector(s);
      if (el && (!visible || isVisible(el))) return el;
    }
    await sleep(interval);
  }
  return null;
}

/* ============================================================
   7) Estado/habilitación + clicks sintéticos
   ============================================================ */

/** Espera a que TODOS los targets estén habilitados (disabled/aria-disabled OFF). */
export async function waitEnabledAttr(targets, { timeout = 2000 } = {}) {
  const list = Array.isArray(targets) ? targets : [targets];
  const enabled = () =>
    list.every(el => el &&
      !(el.disabled === true) &&
      !(el.hasAttribute && el.hasAttribute('disabled')) &&
      !(el.getAttribute && el.getAttribute('aria-disabled') === 'true'));

  if (enabled()) return true;
  return await new Promise(res => {
    let done = false;
    const mo = new MutationObserver(() => {
      if (!done && enabled()) { done = true; mo.disconnect(); res(true); }
    });
    for (const el of list) {
      try { el && mo.observe(el, { attributes: true }); } catch {}
    }
    setTimeout(() => { if (!done) { mo.disconnect(); res(enabled()); } }, timeout);
  });
}

/** Dispara una secuencia realista de pointer/mouse + click sobre un elemento (y opcionalmente su host). */
export function dispatchPointerClick(el, { host = null } = {}) {
  if (!el) return false;
  try { el.scrollIntoView({ block: 'center', inline: 'center' }); } catch {}
  try { el.focus({ preventScroll: true }); } catch {}

  const r = el.getBoundingClientRect();
  const cx = Math.floor(r.left + r.width / 2);
  const cy = Math.floor(r.top + r.height / 2);
  const base = { 
    bubbles: true, 
    composed: true, 
    cancelable: true, 
    clientX: cx, 
    clientY: cy, 
    button: 0, 
    buttons: 1, 
    view: window 
  };

  // pointer + mouse secuencial
  if (window.PointerEvent) {
    el.dispatchEvent(new PointerEvent('pointerdown', { ...base, pointerId: 1, pointerType: 'mouse' }));
  }
  el.dispatchEvent(new MouseEvent('mousedown', base));
  el.dispatchEvent(new MouseEvent('mouseup', { ...base, buttons: 0 }));
  if (window.PointerEvent) {
    el.dispatchEvent(new PointerEvent('pointerup', { ...base, pointerId: 1, pointerType: 'mouse' }));
  }
  el.dispatchEvent(new MouseEvent('click', { ...base, detail: 1 }));
  try { el.click(); } catch {}
  try { host?.click(); } catch {}
  return true;
}

/** Espera (con shadow-root) a {host, btn} dados hostSelector e innerSelector. */
export async function waitShadowButton(hostSelector, innerSelector = 'button', { timeout = 12000, interval = 120, visible = true } = {}) {
  const end = Date.now() + timeout;
  while (Date.now() < end) {
    const host = document.querySelector(hostSelector);
    const btn = host?.shadowRoot?.querySelector(innerSelector) || null;
    if (host && btn && (!visible || isVisible(host))) return { host, btn };
    await sleep(interval);
  }
  return { host: null, btn: null };
}

/* ============================================================
   8) Selección en walla-dropdown (label u oculto)
   ============================================================ */

/**
 * Selecciona una opción en un <walla-dropdown> por label visible o por nombre del input hidden.
 * Abre visualmente el combo, hace click en la opción y verifica el hidden.value.
 */
export async function selectWallaDropdown({
  value,
  hiddenName = null,
  label = null,
  testId = null,  // ✅ v6.5.5: Soporte para data-testid
  timeout = 10000,
  contains = false,
  log = false
} = {}) {
  const startTime = Date.now();
  const tEnd = Date.now() + timeout;
  const norm = (s) => String(s || '').toLowerCase().normalize('NFD').replace(/\p{Diacritic}+/gu, '').trim();
  const esc = (s) => (window.CSS?.escape ? CSS.escape(String(s)) : String(s).replace(/"/g, '\\"'));

  // 🔍 DIAGNÓSTICO: Log inicial
  if (log) {
  }

  if (!value) {
    if (log) console.error('[DOM] ❌ No value provided');
    return { ok: false, code: 'no_value' };
  }
  const want = norm(value);

  // 1) localizar trigger + hidden (soporta shadow-root)
  let trigger = null, hidden = null, host = null, sr = null;
  let attempts = 0;
  let lastError = null;
  
  while (Date.now() < tEnd && !trigger) {
    attempts++;
    try {
      // 🔍 DIAGNÓSTICO: Log cada 2 segundos
      if (log && attempts % 16 === 0) { // Cada ~2s (16 * 120ms)
        const elapsed = Math.round((Date.now() - startTime) / 1000);
      }
      
      // ✅ v6.5.5: Primero intentar por data-testid (nuevo Wallapop)
      if (testId && !trigger) {
        host = queryDeep(document, `[data-testid="${esc(testId)}"]`)
          || queryDeep(document, `walla-dropdown[data-testid="${esc(testId)}"]`)
          || queryDeep(document, `walla-floating-area[data-testid="${esc(testId)}"]`);
        if (host) {
          // 🔍 Buscar trigger en shadowRoot Y en el DOM directo
          trigger = host?.shadowRoot?.querySelector('.walla-dropdown__inner-input[role="button"]')
            || host?.querySelector?.('.walla-dropdown__inner-input[role="button"]')
            || host?.querySelector?.('div[role="button"].walla-dropdown__inner-input')
            || null;
          
          // 🆕 v6.8.6: Si no encuentra trigger interno, usar el host mismo como trigger
          if (!trigger && host.getAttribute('data-testid') === testId) {
            trigger = host;
          }
          
          hidden = host?.querySelector?.('input.walla-dropdown__inner-input__hidden-input')
            || host?.shadowRoot?.querySelector('input.walla-dropdown__inner-input__hidden-input')
            || host?.querySelector?.('input[type="hidden"][name]')
            || null;
          
          // 🔍 DIAGNÓSTICO: Log si encontró host pero no trigger
          if (log && host && !trigger) {
            // Log removido para limpiar consola
          }
        }
      }
      
      if (!trigger && hiddenName) {
        hidden = queryDeep(document, `input.walla-dropdown__inner-input__hidden-input[name="${esc(hiddenName)}"]`)
          || queryDeep(document, `input[type="hidden"][name="${esc(hiddenName)}"]`);
        host = hidden?.closest?.('walla-floating-area, walla-dropdown') || null;
        trigger = host?.shadowRoot?.querySelector('.walla-dropdown__inner-input[role="button"]')
          || host?.querySelector?.('.walla-dropdown__inner-input[role="button"]')
          || null;
      }
      if (!trigger && label) {
        // 🆕 FIX: Wallapop eliminó aria-haspopup="listbox" del trigger
        // Intentar tanto en español como en inglés, case-insensitive
        trigger = queryDeep(document, `.walla-dropdown__inner-input[role="button"][aria-label*="${esc(label)}" i]`)
          || queryDeep(document, `.walla-dropdown__inner-input[role="button"][aria-label*="Estado" i]`)
          || queryDeep(document, `.walla-dropdown__inner-input[role="button"][aria-label*="Condition" i]`);
        host = trigger?.closest?.('walla-floating-area, walla-dropdown') || null;
        hidden = host?.querySelector?.('input.walla-dropdown__inner-input__hidden-input')
          || (trigger?.getRootNode() instanceof ShadowRoot && trigger.getRootNode().querySelector('input.walla-dropdown__inner-input__hidden-input'))
          || null;
      }
      sr = trigger ? trigger.getRootNode() : null;
      if (!trigger) await sleep(120);
    } catch (e) { 
      lastError = e.message;
      await sleep(120); 
    }
  }
  
  if (!trigger) {
    // 🔍 DIAGNÓSTICO: Log detallado del fallo
    if (log) {
      console.error('%c[DOM] ❌ TRIGGER NOT FOUND', 'color: white; background: #dc3545; font-weight: bold; padding: 4px;');
      console.error('[DOM] Tiempo transcurrido:', Math.round((Date.now() - startTime) / 1000), 's');
      console.error('[DOM] Intentos realizados:', attempts);
      console.error('[DOM] Parámetros:', { value, hiddenName, label, testId });
      
      if (lastError) {
        console.error('[DOM] Último error:', lastError);
      }
      
      // Mostrar qué elementos SÍ existen en el DOM
      if (testId) {
        const el = document.querySelector(`[data-testid="${testId}"]`);
      }
      if (hiddenName) {
        const el = document.querySelector(`input[name="${hiddenName}"]`);
      }
      if (label) {
        const el = document.querySelector(`[aria-label*="${label}"]`);
      }
    }
    return { ok: false, code: 'trigger_not_found' };
  }
  
  if (log) {
  }

  // 2) abrir (click realista) y resolver listbox por aria-controls global
  
  const openOnce = () => dispatchPointerClick(trigger);
  openOnce();

  const controlsId = trigger.getAttribute('aria-controls') || '';
  
  const findListbox = () => {
    if (controlsId) {
      const wrap = document.getElementById(controlsId);
      const lb = wrap?.querySelector?.('[role="listbox"]');
      if (lb) return lb;
    }
    if (sr instanceof ShadowRoot) return sr.querySelector('[role="listbox"]');
    return host?.querySelector?.('[role="listbox"]') || queryDeep(host || document, '[role="listbox"]');
  };

  let listbox = null;
  let listboxAttempts = 0;
  const tOpen = Date.now() + 4000;
  while (Date.now() < tOpen && !(listbox = findListbox())) {
    listboxAttempts++;
    if (trigger.getAttribute('aria-expanded') !== 'true') openOnce();
    await sleep(120);
    
    // Log cada segundo
    if (log && listboxAttempts % 8 === 0) {
    }
  }
  
  if (!listbox) {
    if (log) {
      console.error('[DOM] ❌ LISTBOX NOT FOUND después de 4s');
      console.error('[DOM] aria-expanded:', trigger.getAttribute('aria-expanded'));
      console.error('[DOM] Intentos de apertura:', listboxAttempts);
    }
    return { ok: false, code: 'listbox_not_found' };
  }
  

  // 3) seleccionar opción por label
  
  const opts = Array.from(listbox.querySelectorAll('[role="option"], walla-dropdown-item[role="option"]'));
  
  if (log) {
  }
  
  const pick = opts.find(o => {
    const a = norm(o.getAttribute('aria-label') || o.textContent || '');
    return contains ? a.includes(want) : a === want;
  }) || opts.find(o => norm(o.getAttribute('aria-label') || '').includes(want));
  
  if (!pick) {
    if (log) {
      console.error('[DOM] ❌ OPCIÓN NO ENCONTRADA');
      console.error('[DOM] Buscando:', want);
      console.error('[DOM] contains:', contains);
      console.error('[DOM] Todas las opciones:', opts.map(o => ({
        aria: o.getAttribute('aria-label'),
        text: o.textContent,
        normalized: norm(o.getAttribute('aria-label') || o.textContent || '')
      })));
    }
    return { ok: false, code: 'option_not_found' };
  }

  
  dispatchPointerClick(pick, { host: listbox });
  await sleep(180);

  // 4) verificar hidden.value
  if (!hidden) {
    hidden = (sr instanceof ShadowRoot
      ? sr.querySelector('input.walla-dropdown__inner-input__hidden-input')
      : host?.querySelector?.('input.walla-dropdown__inner-input__hidden-input')) || null;
  }
  let finalVal = hidden?.value || '';
  const tCheck = Date.now() + 3000;
  while (Date.now() < tCheck && !finalVal) { 
    await sleep(120); 
    finalVal = hidden?.value || ''; 
  }

  if (!finalVal) return { ok: false, code: 'no_hidden_value_after_click' };
  if (log) console.debug('[dom.selectWallaDropdown]', { name: hidden?.name, value: finalVal });
  return { ok: true, name: hidden?.name || null, value: finalVal };
}

/**
 * 🆕 Selecciona el tier de peso apropiado para envío estándar
 * @param {number} kg - Peso en kilogramos
 * @returns {Promise<{ok: boolean, tier?: number, changed?: boolean, error?: string}>}
 */
export async function selectWeightTier(kg) {
  
  // Mapeo kg → tier (índice del radio)
  const kgToTier = (k) => {
    if (k <= 1) return 0;
    if (k <= 2) return 1;
    if (k <= 5) return 2;
    if (k <= 10) return 3;
    if (k <= 20) return 4;
    if (k <= 30) return 5;
    return 5; // Max tier
  };

  const tier = kgToTier(kg);
  
  // Buscar por id (cada radio tiene id="0", id="1", etc.)
  const targetRadio = document.querySelector(`tsl-delivery-radio-selector input[type="radio"][id="${tier}"]`);

  if (!targetRadio) {
    console.error('[DOM] ❌ Radio de peso no encontrado:', { tier, kg });
    return { ok: false, error: `Radio de peso tier ${tier} (${kg}kg) no encontrado` };
  }


  // ✅ v6.6.8: Verificar si ya está seleccionado
  if (targetRadio.checked) {
    return { ok: true, tier, changed: false };
  }

  // ✅ v6.6.8: Verificar que el radio es interactivo antes de hacer click
  if (targetRadio.disabled) {
    console.error('[DOM] ❌ Radio está deshabilitado');
    return { ok: false, error: `Radio tier ${tier} está deshabilitado` };
  }

  // ✅ v6.6.8: Reintentar hasta 3 veces con verificación
  let attempts = 0;
  const maxAttempts = 3;
  
  while (attempts < maxAttempts) {
    attempts++;
    
    // Hacer scroll al elemento para asegurar visibilidad
    targetRadio.scrollIntoView({ block: 'center', behavior: 'smooth' });
    await sleep(200);
    
    // Click con dispatchPointerClick (más robusto que .click())
    dispatchPointerClick(targetRadio);
    
    // ✅ v6.6.8: Esperar y verificar múltiples veces
    await sleep(300);
    
    // Verificar si se marcó
    if (targetRadio.checked) {
      return { ok: true, tier, changed: true };
    }
    
    
    // Esperar más antes de reintentar
    if (attempts < maxAttempts) {
      await sleep(400);
    }
  }

  // ✅ v6.6.8: Si falló todos los intentos, intentar con .click() nativo
  targetRadio.click();
  await sleep(400);
  
  if (targetRadio.checked) {
    return { ok: true, tier, changed: true };
  }

  // ✅ v6.6.8: Último intento - forzar con eventos programáticos
  const label = targetRadio.closest('label');
  if (label) {
    label.click();
    await sleep(400);
    
    if (targetRadio.checked) {
      return { ok: true, tier, changed: true };
    }
  }

  console.error('[DOM] ❌ El radio no se marcó después de todos los intentos:', { tier, kg, attempts: maxAttempts });
  return { ok: false, error: `No se pudo seleccionar tier ${tier} después de ${maxAttempts} intentos` };
}

/**
 * 🆕 Rellena los campos de medidas para envío voluminoso (ancho, fondo, alto)
 * @param {Object} measures - { width, length, height } en cm
 * @returns {Promise<{ok: boolean, filled: string[], errors?: string[]}>}
 */
export async function fillBulkyMeasures({ width, length, height }) {
  const filled = [];
  const errors = [];

  for (const [name, value] of Object.entries({ width, length, height })) {
    if (!value) continue;

    const input = document.querySelector(`input[id="${name}"][name="${name}"]`);
    
    if (!input) {
      errors.push(`Campo ${name} no encontrado`);
      continue;
    }

    try {
      await setInputValue(input, String(value));
      filled.push(name);
    } catch (err) {
      errors.push(`Error rellenando ${name}: ${err.message}`);
    }
  }

  return {
    ok: errors.length === 0,
    filled,
    errors: errors.length > 0 ? errors : undefined
  };
}

/* ============================================================
   Utilidades exportadas (reutilizables)
   ============================================================ */

/**
 * Normaliza texto: minúsculas, sin acentos, trim
 */
export function norm(s) {
  return String(s || '').toLowerCase()
    .normalize('NFD')
    .replace(/\p{Diacritic}+/gu, '')
    .trim();
}

/**
 * Convierte kg a tier de peso de Wallapop
 * 0: ≤1kg, 1: ≤2kg, 2: ≤5kg, 3: ≤10kg, 4: ≤20kg, 5: ≤30kg, 6: >30kg
 */
export function kgToTier(kg) {
  if (kg <= 1) return 0;
  if (kg <= 2) return 1;
  if (kg <= 5) return 2;
  if (kg <= 10) return 3;
  if (kg <= 20) return 4;
  if (kg <= 30) return 5;
  return 6;
}

/**
 * Selecciona elemento del DOM
 */
export function pick(selector, doc = document) {
  const el = doc.querySelector(selector);
  return el || null;
}