/*  Cabecera obligatoria: NO BORRAR NI MODIFICAR este bloque inicial en ningún fichero.
  Archivo: dom-optimized.js — Rol: Utilidades DOM optimizadas con MutationObserver
  Descripcion de todo el proyecto: 
       MitikLive Wallapop (MV3): panel lateral en Wallapop para gestionar anuncios conectado a FastAPI 
          (login JWT, export/import, backups/renovaciones).  
  Aviso: NO BORRAR los comentarios descriptivos situados encima de cada función, solo cambiarlos si se modifica la funcion.
  En Rol: si esta vacio o el fichero se modifica o reestructura si hacer cambio en Rol
  
  ✅ v6.10.0: Módulo optimizado con MutationObserver para reducir CPU en esperas activas
*/

/* ============================================================
   waitFor - Espera optimizada con MutationObserver
   
   Ventajas vs polling tradicional:
   - 0% CPU cuando no hay cambios en DOM
   - Reacción instantánea (<5ms vs 100-200ms)
   - No sobrecarga en PCs lentos
   - Fallback automático a polling si falla
   ============================================================ */

/**
 * Espera a que una función retorne un valor truthy usando MutationObserver
 * 
 * @param {Function} checkFn - Función que retorna el elemento/valor esperado
 * @param {Object} options - Opciones de configuración
 * @param {number} options.timeout - Timeout máximo en ms (default: 10000)
 * @param {number} options.interval - Intervalo de polling fallback (default: 200)
 * @param {boolean} options.useMutationObserver - Usar MutationObserver (default: true)
 * @returns {Promise} Valor retornado por checkFn o null si timeout
 */
export async function waitFor(checkFn, { 
  timeout = 10000, 
  interval = 200,
  useMutationObserver = true 
} = {}) {
  
  // 1. Check inmediato - si ya existe, retornar
  const immediate = checkFn();
  if (immediate) return immediate;
  
  // 2. Si MutationObserver está disponible y habilitado, usarlo
  if (useMutationObserver && typeof MutationObserver !== 'undefined') {
    try {
      return await waitForWithObserver(checkFn, timeout, interval);
    } catch (err) {
      // Si falla MutationObserver, caer a polling
    }
  }
  
  // 3. Fallback: Polling tradicional
  return await waitForWithPolling(checkFn, timeout, interval);
}

/**
 * Espera usando MutationObserver (eficiente, 0% CPU idle)
 * 
 * @private
 */
async function waitForWithObserver(checkFn, timeout, pollInterval) {
  return new Promise((resolve, reject) => {
    let timeoutId;
    let observer;
    let pollIntervalId;
    
    const cleanup = () => {
      if (timeoutId) clearTimeout(timeoutId);
      if (observer) observer.disconnect();
      if (pollIntervalId) clearInterval(pollIntervalId);
    };
    
    const check = () => {
      try {
        const result = checkFn();
        if (result) {
          cleanup();
          resolve(result);
          return true;
        }
      } catch (err) {
        // Si checkFn falla, ignorar y continuar
      }
      return false;
    };
    
    // Observer para cambios en el DOM
    observer = new MutationObserver((mutations) => {
      // Solo chequear si hubo cambios relevantes
      const hasRelevantChanges = mutations.some(m => 
        m.type === 'childList' || 
        m.type === 'attributes'
      );
      
      if (hasRelevantChanges) {
        check();
      }
    });
    
    // Observar desde document.body (todo el DOM visible)
    try {
      observer.observe(document.body || document.documentElement, {
        childList: true,
        subtree: true,
        attributes: true,
        attributeFilter: ['class', 'style', 'hidden', 'disabled', 'aria-expanded', 'aria-selected']
      });
    } catch (err) {
      cleanup();
      reject(new Error(`Observer setup failed: ${err.message}`));
      return;
    }
    
    // Polling lento como backup (por si observer pierde algo)
    // Cada 500ms en lugar de cada 100-200ms = 60-75% menos CPU
    pollIntervalId = setInterval(() => {
      check();
    }, Math.max(pollInterval * 2.5, 500));
    
    // Timeout
    timeoutId = setTimeout(() => {
      cleanup();
      resolve(null); // No rechazar, retornar null
    }, timeout);
  });
}

/**
 * Espera usando polling tradicional (fallback)
 * 
 * @private
 */
async function waitForWithPolling(checkFn, timeout, interval) {
  const startTime = Date.now();
  
  while (Date.now() - startTime < timeout) {
    try {
      const result = checkFn();
      if (result) return result;
    } catch (err) {
      // Si checkFn falla, ignorar y continuar
    }
    
    await new Promise(resolve => setTimeout(resolve, interval));
  }
  
  return null; // Timeout alcanzado
}

/**
 * Espera a que aparezca un elemento (simplificado)
 * 
 * @param {string} selector - Selector CSS del elemento
 * @param {Object} options - Opciones (timeout, interval, useMutationObserver)
 * @returns {Promise<Element|null>} Elemento encontrado o null
 */
export async function waitForElement(selector, options = {}) {
  return waitFor(() => document.querySelector(selector), options);
}

/**
 * Espera a que elemento sea visible
 * 
 * @param {string|Element} selectorOrElement - Selector CSS o elemento
 * @param {Object} options - Opciones
 * @returns {Promise<Element|null>} Elemento visible o null
 */
export async function waitForVisible(selectorOrElement, options = {}) {
  const element = typeof selectorOrElement === 'string'
    ? await waitForElement(selectorOrElement, options)
    : selectorOrElement;
  
  if (!element) return null;
  
  return waitFor(() => {
    const rect = element.getBoundingClientRect();
    const style = getComputedStyle(element);
    
    const isVisible = 
      style.display !== 'none' &&
      style.visibility !== 'hidden' &&
      style.opacity !== '0' &&
      rect.width > 0 &&
      rect.height > 0;
    
    return isVisible ? element : null;
  }, options);
}

/**
 * Espera a que elemento tenga atributo específico
 * 
 * @param {Element} element - Elemento a monitorear
 * @param {string} attrName - Nombre del atributo
 * @param {string} expectedValue - Valor esperado
 * @param {Object} options - Opciones
 * @returns {Promise<Element|null>}
 */
export async function waitForAttribute(element, attrName, expectedValue, options = {}) {
  if (!element) return null;
  
  return waitFor(() => {
    const value = element.getAttribute(attrName);
    return value === expectedValue ? element : null;
  }, options);
}

/**
 * Espera a que desaparezca un loader
 * 
 * @param {string} containerSelector - Selector del contenedor (default: 'body')
 * @param {Object} options - Opciones
 * @returns {Promise<boolean>} true cuando no hay loaders visibles
 */
export async function waitForNoLoader(containerSelector = 'body', options = {}) {
  return waitFor(() => {
    const container = document.querySelector(containerSelector);
    if (!container) return null;
    
    const loaders = container.querySelectorAll('[class*="loader"], [class*="loading"], [class*="spinner"]');
    const hasVisible = Array.from(loaders).some(l => {
      const style = getComputedStyle(l);
      return style.display !== 'none' && 
             style.visibility !== 'hidden' && 
             style.opacity !== '0';
    });
    
    return !hasVisible ? true : null;
  }, options);
}

/**
 * Espera a que una condición se cumpla con retry
 * Útil para verificar cambios después de acciones
 * 
 * @param {Function} checkFn - Función que retorna true/false
 * @param {Object} options - Opciones
 * @returns {Promise<boolean>}
 */
export async function waitForCondition(checkFn, options = {}) {
  const result = await waitFor(() => {
    return checkFn() ? true : null;
  }, options);
  
  return result === true;
}

// Re-exportar desde dom.js para compatibilidad
export { norm, kgToTier } from './dom.js';
