/*
  Archivo: delete.js — Borrado de anuncios (API-only + verificación; sin DOM fallback)
  Proyecto: MitikLive Wallapop (MV3)
*/

import { sleep, retryWithBackoff } from './utils.js';

// ========= utilidades locales =========

// -- pedir credenciales frescas al SW (Bearer, deviceId, etc.)
async function getCredsFromSW() {
  return new Promise((resolve) => {
    try {
      chrome.runtime.sendMessage({ type: 'WALLA.CREDS.GET' }, (res) => {
        resolve(res || { ok: false });
      });
    } catch (e) {
      resolve({ ok: false, error: String(e) });
    }
  });
}

// -- invalidar credenciales en SW (forzar recaptura)
async function invalidateWallaCreds() {
  return new Promise((resolve) => {
    try {
      chrome.runtime.sendMessage({ type: 'WALLA.CREDS.INVALIDATE' }, (res) => {
        resolve(res || { ok: false });
      });
    } catch (e) {
      resolve({ ok: false });
    }
  });
}

// -- construir headers mimetizando la UI
function buildHeaders(creds) {
  const h = new Headers();
  h.set('authorization', `Bearer ${creds.bearer}`);
  h.set('X-Device-Id', creds.deviceId || '');
  if (creds.appVersion) h.set('x-appversion', String(creds.appVersion));
  if (creds.deviceOs)  h.set('x-deviceos', String(creds.deviceOs));
  h.set('accept', 'application/json');
  return h;
}

// -- fetch desde el propio contexto de página (evita CORS raros)
async function pageFetch(url, init) {
  const opts = Object.assign({ referrer: 'https://es.wallapop.com/', credentials: 'omit' }, init || {});
  return fetch(url, opts);
}

// -- helpers API (API-only)
async function apiDeleteItem(id, headers) {
  const r = await pageFetch(`https://api.wallapop.com/api/v3/items/${id}`, { method: 'DELETE', headers });
  return { ok: r.status === 200 || r.status === 204, status: r.status };
}

async function apiGetItem(id, headers) {
  const r = await pageFetch(`https://api.wallapop.com/api/v3/items/${id}`, { headers });
  if (r.status === 404 || r.status === 410) return { ok: true, status: r.status, gone: true };
  if (!r.ok) return { ok: false, status: r.status };
  return { ok: true, status: r.status, gone: false, json: await r.json() };
}

// ========= helpers de parsing =========
function resolveId(item) {
  // 1) si ya viene id, usarlo
  const direct = String(item?.id || '').trim();
  if (direct) return direct;

  // 2) origen: slug o url
  let src = String(item?.slug || item?.url || '').trim();
  if (!src) return '';

  // 3) normalizar: quedarnos con pathname/último segmento, sin query/hash y sin slash final
  try {
    // soporta pasar solo slug (no URL completa)
    const u = new URL(src, 'https://dummy.invalid');
    src = u.pathname || src;
  } catch { /* slug simple */ }

  src = src.replace(/[?#].*$/, '').replace(/\/+$/, '');
  const lastSeg = (src.split('/').filter(Boolean).pop() || '').trim();
  if (!lastSeg) return '';

  // 4) casos comunes: "titulo-123456789" → "123456789"
  const hyTail = lastSeg.split('-').pop();
  if (/^\d+$/.test(hyTail)) return hyTail;

  // 5) si es todo dígitos, usar tal cual
  if (/^\d+$/.test(lastSeg)) return lastSeg;

  // 6) último bloque de dígitos al final (por si trae prefijo alfanumérico)
  const m = lastSeg.match(/(\d+)(?!.*\d)/);
  if (m) return m[1];

  // 7) fallback: devolver el segmento (puede ser UUID/slug válido para la API)
  return lastSeg;
}

// ========= API-only + verificación (sin DOM) =========

/**
 * Elimina un anuncio por ID/slug/URL.
 * Flujo: DELETE -> GET (verificación). Sin DOM fallback.
 * Usa retryWithBackoff para reintentos automáticos en 401/429/5xx.
 * 
 * 🚨 CRÍTICO: Si falla, el proceso de publicación DEBE detenerse
 * para evitar baneo por duplicados en Wallapop.
 * 
 * @param {{id?: string, slug?: string, url?: string}} item 
 * @returns {Promise<{ok:boolean, method:'api'|'idempotent'|undefined, http?:number, verify?:string, ms:number, error?:string, halt?:boolean, retry?:boolean}>}
 */
export async function deleteOne(item) {
  const t0 = performance.now();

  // 1) Resolver ID (robusto ante / final, ?query, #hash, slugs tipo "nombre-123...")
  const id = resolveId(item);
  if (!id) return { ok: false, ms: performance.now() - t0, http: 0, error: 'no_id', halt: true };

  // 2) Intento con retry (401/429/5xx) y guardado de headers para la verificación
  let lastHeaders = null;
  let retryCount = 0;
  
  const attemptDelete = async () => {
    const creds = await getCredsFromSW();
    if (!creds?.ok || !creds.bearer) {
      throw Object.assign(new Error('no_creds'), { halt: true });
    }
    const headers = buildHeaders(creds);
    lastHeaders = headers;

    const del = await apiDeleteItem(id, headers);

    // 🔒 PROTECCIÓN: Reintentar en errores transitorios
    if (del.status === 401) {
      // 401 Unauthorized: invalidar credenciales para forzar recaptura en retry
      await invalidateWallaCreds();
      retryCount++;
      throw Object.assign(new Error(`transient_401`), { retriable: true, status: 401 });
    }
    
    if (del.status === 429 || (del.status >= 500 && del.status < 600)) {
      throw Object.assign(new Error(`transient_${del.status}`), { retriable: true, status: del.status });
    }
    
    return del;
  };

  // 3) DELETE con retry automático (2 reintentos máximo)
  let del;
  try {
    del = await retryWithBackoff(attemptDelete, { 
      maxRetries: 2,  // 🔒 2 reintentos para TODOS los casos
      baseDelay: 500, // Aumentado a 500ms (más conservador)
      onRetry: (attempt, delay, err) => {
      }
    });
  } catch (e) {
    const ms = performance.now() - t0;
    if (e.halt) return { ok: false, ms, http: 401, error: 'no_walla_creds', halt: true };
    // 🚨 CRÍTICO: Si agotamos reintentos, HALT
    return { ok: false, ms, http: e.status || 0, error: `delete_failed_after_retries: ${e.message}`, halt: true };
  }

  // 4) Clasificación estados
  const ms = performance.now() - t0;
  const http = del.status;

  if (del.ok && (http === 200 || http === 204)) {
    // 🔒 Verificación con RETRY (Wallapop puede tardar en procesar)
    const headers = lastHeaders || buildHeaders(await getCredsFromSW());
    
    // Intentar verificar hasta 3 veces con delays progresivos
    let verified = false;
    const delays = [500, 1500, 3000]; // 500ms, 1.5s, 3s
    
    for (let attempt = 0; attempt < delays.length; attempt++) {
      await sleep(delays[attempt]);
      const chk = await apiGetItem(id, headers);
      
      if (chk.ok && chk.gone) {
        // ✅ Confirmado: anuncio eliminado
        return { ok: true, method: 'api', http, verify: 'get=404', ms: performance.now() - t0 };
      }
      
    }
    
    // Si después de 3 intentos (5s total) aún existe, considerar que falló
    console.error('[DELETE] 🚨 CRITICAL: Anuncio NO se eliminó después de 5s de espera', { id, http });
    return { 
      ok: false, 
      http, 
      verify: 'still_exists_after_retries', 
      ms: performance.now() - t0, 
      error: 'verify_failed_anuncio_aun_existe_despues_5s', 
      halt: true
    };
  }

  if (http === 404 || http === 410) {
    // Ya no existe → operación idempotente satisfactoria
    return { ok: true, method: 'idempotent', http, verify: 'gone', ms };
  }
  
  if (http === 401 || http === 403) {
    // 🚨 no autorizado / not_mine → detener TODO
    console.error('[DELETE] 🚨 No autorizado (401/403) - posible cuenta incorrecta');
    return { ok: false, http, ms, halt: true, error: 'not_authorized' };
  }
  
  // Cualquier otro error → HALT
  console.error('[DELETE] 🚨 Error inesperado en DELETE', { http, id });
  return { ok: false, http, ms, error: 'delete_failed_unknown', halt: true };
}

/**
 * Elimina en serie (helper simple). Mejor usar el orquestador del SW.
 * @param {Array<{id?:string, slug?:string, url?:string}>} items 
 * @returns {Promise<Array<any>>}
 */
export async function deleteBatch(items = []) {
  const out = [];
  for (const it of items) out.push(await deleteOne(it));
  return out;
}