/*
  Cabecera obligatoria: NO BORRAR NI MODIFICAR este bloque inicial en ningún fichero.
  Archivo: backup.js — Rol: Orquestación local de navegación para backup (sin tocar API, sin auto-ejecuciones).
  Descripcion de todo el proyecto: 
       MitikLive Wallapop (MV3): panel lateral en Wallapop para gestionar anuncios conectado a FastAPI 
          (login JWT, export/import, backups/renovaciones).  
  Aviso: NO BORRAR los comentarios descriptivos situados encima de cada función, solo cambiarlos si se modifica la funcion.
  En Rol: si esta vacio o el fichero se modifica o reestructura si hacer cambio en Rol
*/

import {
  isLoggedInWallapop,
  waitItemsListed,
  resolvePublicProfileUrlViaIframe
} from './dom.js';
import { logger } from './panel/logger.js';

/**
 * Clave de estado en storage para el flujo de backup.
 * Se usa únicamente como marcador de navegación (sin side-effects).
 */
export const BK_KEY = 'ml_backup_flow';

const URLS = {
  login: 'https://es.wallapop.com/app/login'
};

/**
 * Navega de forma segura sin provocar bucles. No hace nada si ya estás en la URL.
 * @param {string} href
 */
function safeGoto(href) {
  try {
    if (typeof href !== 'string' || !href) return;
    if (href === location.href) return;
    location.assign(href);
  } catch {}
}

/**
 * Inicia el flujo de backup SOLO a nivel de cliente (navegación).
 * No dispara llamadas a tu API ni tareas de negocio.
 *
 * Reglas:
 *  - Si NO hay sesión en Wallapop: guarda step "need_login" y navega a /login.
 *  - Si hay sesión: intenta resolver la URL pública del perfil; si existe,
 *    guarda step "open_public" y navega allí.
 *  - Si no se puede resolver: limpia estado para evitar bucles.
 *
 * No hay auto-ejecuciones: este módulo NUNCA se ejecuta por sí solo.
 *
 * @param {{
 *   storageSet: (obj:object)=>Promise<void>,
 *   guardAuth?: ()=>Promise<boolean> | boolean,  // opcional: si lo pasas, exige sesión de tu app antes de actuar
 * }} deps
 */
export async function startBackupFlow({ storageSet, guardAuth } = {}) {
  
  // (Opcional) Si se aporta guardAuth y no hay sesión en tu app, no hacemos nada.
  try {
    if (typeof guardAuth !== 'undefined') {
      const ok = typeof guardAuth === 'function' ? await guardAuth() : !!guardAuth;
      if (!ok) {
        return;
      }
    }
  } catch (err) {
    logger.error('❌ Error en guardAuth', { error: err.message });
    /* ignore guard errors */
  }

  const logged = await isLoggedInWallapop();
  if (!logged) {
    await storageSet?.({ [BK_KEY]: { step: 'need_login', ts: Date.now() } });
    safeGoto(URLS.login);
    return;
  }

  const href = await resolvePublicProfileUrlViaIframe().catch((err) => {
    logger.error('❌ Error al resolver URL pública del perfil', { error: err.message });
    return null;
  });
  
  if (!href) {
    await storageSet?.({ [BK_KEY]: null });
    return;
  }

  await storageSet?.({ [BK_KEY]: { step: 'open_public', ts: Date.now(), href } });
  safeGoto(href);
}

/**
 * Reanuda el flujo tras recargas o navegación SPA.
 * Solo realiza navegación y limpieza de estado; NO llama a la API.
 *
 * Casos:
 *  - step=need_login → intenta resolver perfil y navega; si falla, limpia.
 *  - step=open_public y ya estás en /user/... → espera listado y limpia.
 *  - resto → limpia.
 *
 * @param {{
 *   storageGet: (keys:string[])=>Promise<object>,
 *   storageSet: (obj:object)=>Promise<void>,
 *   toast?: (msg:string, type?:'ok'|'warn'|'err'|'info')=>void,
 *   guardAuth?: ()=>Promise<boolean> | boolean, // opcional: exige sesión de tu app antes de reanudar
 * }} deps
 */
export async function maybeResumeBackup({ storageGet, storageSet, toast, guardAuth } = {}) {
  
  // (Opcional) Si se aporta guardAuth y no hay sesión en tu app, no hacemos nada.
  try {
    if (typeof guardAuth !== 'undefined') {
      const ok = typeof guardAuth === 'function' ? await guardAuth() : !!guardAuth;
      if (!ok) {
        return;
      }
    }
  } catch { /* ignore guard errors */ }

  const st = (await storageGet?.([BK_KEY]))?.[BK_KEY];
  if (!st) {
    return;
  }

  const path = location.pathname || '';

  if (st.step === 'need_login') {
    const href = await resolvePublicProfileUrlViaIframe().catch(() => null);
    if (href) {
      await storageSet?.({ [BK_KEY]: { step: 'open_public', ts: Date.now(), href } });
      safeGoto(href);
      return;
    }
    logger.error('❌ No se pudo resolver la URL del perfil');
    toast?.('No se pudo resolver la URL del perfil.', 'warn');
    await storageSet?.({ [BK_KEY]: null });
    return;
  }

  if (st.step === 'open_public' && /\/user\//.test(path)) {
    const first = await waitItemsListed({ timeoutMs: 20000 }).catch((err) => {
      logger.error('❌ Error esperando items', { error: err.message });
      return null;
    });
    
    if (!first) {
      toast?.('No se cargaron anuncios.', 'warn');
    } else {
    }
    
    await storageSet?.({ [BK_KEY]: null });
    return;
  }

  // Estado desconocido → limpiar
  await storageSet?.({ [BK_KEY]: null });
}

/* 
  Notas de diseño:
  - Este fichero queda “inerte”: solo exporta funciones puras de navegación/estado.
  - No hay auto-imports desde content_script ni timers, observers o hooks globales.
  - Si en el futuro lo invoca el panel o el CS, hazlo SIEMPRE después de verificar sesión
    vía SW (o pasando guardAuth en deps), para mantener el gate global.
*/
