/*  Cabecera obligatoria: NO BORRAR NI MODIFICAR este bloque inicial en ningún fichero.
    Archivo: panel.js – Rol: UI del panel lateral con sincronización de estado completa
    Descripcion de todo el proyecto:
         MitikLive Wallapop (MV3): panel lateral en Wallapop para gestionar anuncios conectado a FastAPI
            (login JWT, export/import, backups/renovaciones).
    Aviso: NO BORRAR los comentarios descriptivos situados encima de cada función, solo cambiarlos si se modifica la funcion.
    En Rol (linea mas arriba): si está vacía o el fichero se modifica o reestructura, modificar esa linea de rol
    
    VERSIÓN: 3.2.0 - CONTRATO ÚNICO WebSocket
    
    ═══════════════════════════════════════════════════════════════════
    📡 LISTENERS DE EVENTOS WEBSOCKET (línea ~1230)
    ═══════════════════════════════════════════════════════════════════
    
    El panel escucha eventos WebSocket que vienen TAL CUAL del backend.
    
    EVENTOS ACTUALES:
    - 'connected'          → WebSocket conectado (dot verde)
    - 'disconnected'       → WebSocket desconectado (dot rojo)
    - 'backup_progress'    → Progreso de backup en tiempo real
    - 'backup_complete'    → Backup finalizado
    - 'publish_progress'   → Progreso de publicación
    - 'publish_complete'   → Publicación finalizada
    - 'activity_update'    → Actualización de actividad general
    
    AÑADIR NUEVO EVENTO:
    1. Backend envía: {type: 'nuevo_evento', data: {...}}
    2. Añadir listener aquí: if (msg?.type === 'nuevo_evento') {...}
    3. NO modificar realtime.js (reenvía automáticamente)
    
    Ver: /outputs/CONTRATO_WEBSOCKET.md para detalles completos
    ═══════════════════════════════════════════════════════════════════
*/

import DOMUtils from './scripts/panel/dom-utils.js';
import { toast, showLoader, hideLoader, setLoaderText, showModal, sleep, debounce } from './scripts/utils.js';
import * as STATE from './scripts/panel/state.js';
import * as UI from './scripts/panel/ui.js';
import * as AUTH from './scripts/panel/auth.js';
import * as ButtonManager from './scripts/panel/publish-button-manager.js';
import * as AppState from './scripts/panel/state-manager.js';
// ✅ v2.5.1: Modal educativo de confirmación antes de publicar
import { showPublishConfirmation, isEducationalModalEnabled } from './scripts/panel/publish-confirmation-modal.js';
// ✅ v10.5.82: selectAllByFilter movido a module-state.js
// ✅ v2.0.0: Añadido getExcludedIds para "todos excepto estos"
import { getSelectAllByFilter, clearSelectAllByFilter, getExcludedIds } from './scripts/panel/listings/module-state.js';
// SSE eliminado - dot ahora vía WebSocket
import * as POLLING from './scripts/panel/polling.js';
import * as RUNS from './scripts/panel/runs.js';
import * as RESUME from './scripts/panel/resume.js';
import * as MENU from './scripts/panel/menu.js';
import * as DETECT from './scripts/panel/detect.js';
import * as LISTENERS from './scripts/panel/listeners.js';
import * as LISTINGS from './scripts/panel/listings/index.js';
import * as CONFIG from './scripts/panel/config.js';
// ✅ v10.5.146: Sistema de onboarding completo
import * as ONBOARDING from './scripts/panel/onboarding/index.js';
// ✅ v6.11.0: ResourceManager para debug de memory leaks
import ResourceManager from './scripts/panel/resource-manager.js';

// Exponer ResourceManager globalmente para debug en consola
window.ResourceManager = ResourceManager;
import { ProgressTab } from './scripts/panel/progress-tab.js';  // ✅ v6.12.0: Tab de progreso
import { stGet, stSet, stRemove } from './scripts/panel/storage.js';
import { logger } from './scripts/panel/logger.js';
import { initTabs } from './scripts/panel/tabs.js';
import { validateCreditsBeforePublish, deductCreditAfterPublish } from './scripts/panel/publish-validation.js';
import { USER_STATES } from './scripts/panel/constants.js';
import { showSuccessModal } from './scripts/panel/success-modal.js';

// ❌ NO importar simple_publish (usa repost.js que tiene export)

/* ============================================
   ✅ v2.2.3: VERIFICACIÓN DE VERSIÓN SIN AUTH
   ============================================
   Se ejecuta ANTES de cualquier init, incluso sin login.
   Usa fetch directo porque el endpoint no requiere auth.
   ============================================ */

/**
 * Verificar versión de extensión SIN requerir autenticación.
 * Usa fetch directo al endpoint público /api/system/version-check
 * 
 * @returns {Promise<boolean>} true si necesita actualizar (modal mostrado)
 */
async function checkVersionWithoutAuth() {
  try {
    const currentVersion = chrome.runtime.getManifest().version;
    logger.info(`[VERSION] Verificando versión: ${currentVersion}`);
    
    // Obtener API_BASE del storage o config
    const stored = await stGet(['api_base', 'config_overrides']);
    let apiBase = stored?.api_base || stored?.config_overrides?.API_BASE || 'https://www.mitiklive.com/fa';
    
    // Asegurar que no tenga slash final
    apiBase = apiBase.replace(/\/$/, '');
    
    const url = `${apiBase}/api/system/version-check?current_version=${currentVersion}`;
    
    const response = await fetch(url, {
      method: 'GET',
      headers: {
        'Content-Type': 'application/json',
        'X-Extension-Version': currentVersion
      },
      signal: AbortSignal.timeout(10000)  // 10s timeout
    });
    
    if (!response.ok) {
      logger.warn(`[VERSION] Error HTTP ${response.status} verificando versión`);
      return false;  // No bloquear en caso de error
    }
    
    const data = await response.json();
    logger.info(`[VERSION] Respuesta: needs_update=${data.needs_update}, required=${data.required_version}`);
    
    if (data.needs_update) {
      // Importar dinámicamente para mostrar modal
      const { showUpdateModal } = await import('./scripts/panel/maintenance.js');
      showUpdateModal(data.required_version, data.download_url);
      return true;
    }
    
    return false;
    
  } catch (err) {
    logger.error('[VERSION] Error verificando versión sin auth:', err);
    return false;  // No bloquear en caso de error (fail-open)
  }
}

/* ============================================
   ✅ SISTEMA DE PROCESAMIENTO CONTINUO
   ============================================ */
let processingActive = false;
let shouldStopProcessing = false;

// ============================================================
// FLUJO UNIFICADO DE PUBLICACIÓN (v4.63.8)
// ============================================================

/**
 * Flujo unificado de publicación para anuncios activos y deleted
 * @param {Object} options
 * @param {number} options.accountId - ID de la cuenta
 * @param {Array<number>} options.selectedIds - IDs seleccionados (vacío = todos)
 * @param {boolean} options.isDeletedMode - true si son anuncios deleted a republicar
 */
async function handlePublishFlow({ accountId, selectedIds = [], isDeletedMode = false, skipConfirmation = false, isResumeMode = false, resumePendingCount = 0 }) {
  
  // ✅ v7.2.44: Validar si hay publicación ACTIVA (item en processing)
  try {
    const statusRes = await chrome.runtime.sendMessage({
      type: 'API.FETCH_JSON',
      url: `/api/walla/publish2/status?account_id=${accountId}`,
      method: 'GET'
    });
    
    if (statusRes?.ok && statusRes.data?.processing > 0) {
      // ✅ v2.2.2: Mostrar modal con opciones en vez de toast
      const action = await showModal({
        title: '⚠️ Proceso interrumpido',
        html: `
          <p style="margin-bottom: 12px;">Hay un anuncio que quedó a medias de publicarse.</p>
          <p style="margin-bottom: 12px; font-size: 13px; color: #93c5fd;">Esto puede ocurrir por un corte de luz, recarga de la extensión, o cierre inesperado del navegador.</p>
          <p style="font-size: 13px;">¿Qué quieres hacer?</p>
        `,
        buttons: [
          { text: '🔄 Limpiar y continuar', value: 'clean', primary: true },
          { text: '❌ Cancelar', value: 'cancel', primary: false }
        ]
      });
      
      if (action === 'clean') {
        // Llamar a reset-stuck y luego continuar
        toast('⏳ Limpiando proceso colgado...', 'info', 2000);
        const resetRes = await chrome.runtime.sendMessage({
          type: 'API.FETCH_JSON',
          url: `/api/walla/publish2/reset-stuck`,
          method: 'POST',
          body: { account_id: accountId }
        });
        
        if (resetRes?.ok && resetRes.data?.ok) {
          toast('✅ Proceso limpiado, continuando...', 'ok', 2000);
          // Pequeño delay para que el usuario vea el mensaje
          await new Promise(r => setTimeout(r, 500));
        } else {
          const error = resetRes?.data?.detail || resetRes?.error || 'Error desconocido';
          toast(`❌ Error limpiando: ${error}`, 'error');
          return;
        }
      } else {
        // Usuario canceló
        return;
      }
    }
  } catch (e) {
  }
  
  // 1. Validar cuenta
  if (!accountId) {
    return toast('Selecciona una cuenta', 'warn');
  }
  
  // ✅ v10.5.82: getSelectAllByFilter viene de module-state.js
  // ✅ v2.0.0: Añadido soporte para excludedIds
  const selectAllByFilter = getSelectAllByFilter();
  const excludedIds = getExcludedIds(); // Set de IDs excluidos
  const dbStats = AppState.getDbStats();
  
  const isPublishAll = (selectedIds.length === 0 && !isResumeMode) || selectAllByFilter;
  
  // 2. Determinar cuántos anuncios se van a publicar
  let totalToPublish;
  let idsToPublish = selectedIds;
  let excludeIdsToSend = []; // ✅ v2.0.0: IDs a excluir (para backend)
  
  if (isResumeMode) {
    totalToPublish = resumePendingCount || 0;
    idsToPublish = [];  // Backend sabe cuáles son los pendientes
  } else if (isDeletedMode) {
    // Modo deleted: usar stats de BD si selectAllByFilter está activo
    if (selectAllByFilter) {
      totalToPublish = (dbStats.deleted || 0) - excludedIds.size;
      idsToPublish = []; // Backend tomará todos los deleted
      excludeIdsToSend = Array.from(excludedIds);
    } else {
      totalToPublish = selectedIds.length;
      idsToPublish = selectedIds;
    }
  } else {
    // Modo normal: activos seleccionados o todos
    if (selectAllByFilter) {
      // ✅ v2.0.0: Total de BD menos excluidos
      totalToPublish = (dbStats.active || 0) - excludedIds.size;
      idsToPublish = [];
      excludeIdsToSend = Array.from(excludedIds);
    } else if (isPublishAll) {
      const activeListings = LISTINGS.getAllListings().filter(l => l.status === 'active');
      totalToPublish = activeListings.length;
      idsToPublish = [];
    } else {
      totalToPublish = selectedIds.length;
      idsToPublish = selectedIds;
    }
  }
  
  if (totalToPublish === 0) {
    const msg = isDeletedMode 
      ? 'No hay anuncios eliminados seleccionados'
      : isResumeMode
        ? 'No hay anuncios pendientes'
        : 'No hay anuncios activos para publicar';
    return toast(msg, 'warn');
  }
  
  // 3. Modal educativo de confirmación (solo si no se solicita saltar y NO es reanudar)
  // ✅ v2.5.1: Muestra beneficios de la renovación, con opción "No volver a mostrar"
  let educationalModalWasShown = false;
  if (!skipConfirmation && !isResumeMode) {
    educationalModalWasShown = await isEducationalModalEnabled();
    const confirmed = await showPublishConfirmation({ selectedCount: totalToPublish });
    if (!confirmed) {
      return; // Usuario canceló en el modal educativo
    }
  }
  
  // 4. Modal de confirmación técnico (sincronización)
  // ✅ v2.5.1: Solo mostrar si el modal educativo está deshabilitado (usuario lo ocultó)
  if (!skipConfirmation && !educationalModalWasShown) {
    const confirmTitle = isDeletedMode ? '📤 Confirmar republicación' : '📤 Confirmar publicación';
    const confirmMsg = isDeletedMode
      ? `¿Republicar ${totalToPublish} anuncio${totalToPublish !== 1 ? 's' : ''} eliminado${totalToPublish !== 1 ? 's' : ''}?`
      : isPublishAll 
        ? `¿Publicar TODOS los anuncios activos (${totalToPublish})?`
        : `¿Publicar ${totalToPublish} anuncios seleccionados?`;
    
    // ✅ v2.0.0: Obtener fecha de última sincronización y generar mensaje contextual
    const lastSyncElement = document.getElementById('last-backup-date');
    const lastSyncText = lastSyncElement?.textContent?.trim() || '';
    
    let syncWarningHtml = '';
    if (!isDeletedMode) {
      if (!lastSyncText) {
        // Nunca se ha sincronizado
        syncWarningHtml = `
          <div style="background: rgba(245, 158, 11, 0.15); border: 1px solid rgba(245, 158, 11, 0.3); border-radius: 8px; padding: 14px; margin-top: 14px;">
            <p style="margin: 0 0 8px 0; color: #f59e0b; font-weight: 600; font-size: 15px;">
              ⚠️ No hay sincronización registrada
            </p>
            <p style="margin: 0; font-size: 14px; color: #94a3b8; line-height: 1.4;">
              Es recomendable sincronizar antes de publicar para asegurar que los datos están actualizados.
            </p>
          </div>
        `;
      } else {
        // Hay fecha de sincronización
        syncWarningHtml = `
          <div style="background: rgba(59, 130, 246, 0.1); border: 1px solid rgba(59, 130, 246, 0.2); border-radius: 8px; padding: 14px; margin-top: 14px;">
            <p style="margin: 0 0 6px 0; font-size: 15px; color: #e2e8f0;">
              🔄 Última sincronización: <strong style="color: #3b82f6;">${lastSyncText}</strong>
            </p>
            <p style="margin: 0; font-size: 14px; color: #94a3b8; line-height: 1.4;">
              Si has hecho cambios en Wallapop, sincroniza primero para que se reflejen en las publicaciones.
            </p>
          </div>
        `;
      }
    } else {
      // Modo republicación
      syncWarningHtml = `
        <p style="margin-top: 10px; font-size: 14px; color: #f59e0b;">⚠️ Los anuncios se activarán y añadirán a la cola de publicación</p>
      `;
    }
    
    const result = await showModal({
      title: confirmTitle,
      html: `
        <p style="margin-bottom: 10px; font-size: 16px;">${confirmMsg}</p>
        ${syncWarningHtml}
      `,
      buttons: isDeletedMode ? [
        { text: 'Cancelar', value: 'cancel', primary: false },
        { text: 'Republicar', value: 'confirm', primary: true }
      ] : [
        { text: 'Cancelar', value: 'cancel', primary: false },
        { text: 'Sincronizar primero', value: 'sync', primary: false },
        { text: 'Publicar', value: 'confirm', primary: true }
      ]
    });
    
    if (result === 'cancel') return;
    
    // ✅ v1.10.0: Si elige sincronizar, hacer clic en el botón de backup
    if (result === 'sync') {
      const btnBackup = document.getElementById('btn-backup');
      if (btnBackup) {
        btnBackup.click();
        toast('🔄 Sincronizando... Publica cuando termine', 'info', 3000);
      }
      return;
    }
    
    if (result !== 'confirm') return;
  }
  
  // 5. Validar backup (solo si NO es modo deleted)
  if (!isDeletedMode && !STATE.canPublish()) {
    const status = STATE.BACKUP_STATE.status;
    if (status === 'running') {
      return toast('⏳ Espera a que termine el backup', 'warn');
    } else if (['incomplete', 'cancelled', 'canceled'].includes(status)) {
      return toast('⚠️ Completa o cancela el backup incompleto primero', 'warn');
    } else if (status === 'error') {
      return toast('❌ Resuelve los errores del backup primero', 'warn');
    } else {
      return toast('Completa el backup antes de publicar', 'warn');
    }
  }
  
  // 6. Validar créditos
  const hasCredits = await validateCreditsBeforePublish(totalToPublish);
  if (!hasCredits) {
    return;
  }
  
  // 7. Verificar Wallapop con cuenta correcta
  
  const { requireWallaActive } = await import('./scripts/panel/auth.js');
  const { showLoader, hideLoader } = await import('./scripts/utils.js');
  
  showLoader('Verificando Wallapop...', { 
    timeout: 30000,
    onTimeout: () => {
      toast('⏱️ Tiempo agotado verificando Wallapop', 'error', 5000);
    }
  });
  
  const wallaCheck = await requireWallaActive();
  hideLoader();
  
  if (!wallaCheck?.ok) {
    return;
  }
  
  try {
    showLoader('Preparando publicación...', {
      timeout: 30000,
      onTimeout: () => toast('⏱️ Tiempo agotado preparando publicación', 'error', 5000)
    });
    
    // 8. Si es modo deleted: cambiar deleted → active PRIMERO
    if (isDeletedMode) {
      
      const updatePromises = idsToPublish.map(id => 
        chrome.runtime.sendMessage({
          type: 'API.FETCH_JSON',
          url: `/api/walla/listings/${id}/status`,
          method: 'PATCH',
          body: { status: 'active' }
        })
      );
      
      const results = await Promise.all(updatePromises);
      const successCount = results.filter(r => r?.ok).length;
      
      
      if (successCount === 0) {
        hideLoader();
        return toast('❌ No se pudo activar ningún anuncio', 'error');
      }
    }
    
    // 9. Verificar si hay pendientes (reanudar)
    const statusRes = await chrome.runtime.sendMessage({
      type: 'API.FETCH_JSON',
      url: `/api/walla/publish2/status?account_id=${accountId}`,
      method: 'GET'
    });
    
    const pending = statusRes?.data?.pending || 0;
    const isResume = pending > 0;
    
    if (!isResume) {
      // 10. Encolar en BD
      // ✅ v2.0.0: Pasar exclude_ids si hay exclusiones
      const publishResult = await chrome.runtime.sendMessage({
        type: 'PUBLISH.START',
        account_id: accountId,
        listing_ids: idsToPublish,
        exclude_ids: excludeIdsToSend, // ✅ v2.0.0: IDs a excluir
        is_resume: false  // ✅ v6.7.0: Flag para indicar que NO es resume
      });
      
      if (!publishResult?.ok) {
        hideLoader();
        let errorMsg = publishResult?.error || 'Desconocido';
        if (errorMsg === 'publish_active') {
          errorMsg = 'Ya hay una publicación en curso.';
        }
        return toast(`❌ ${errorMsg}`, 'error');
      }
      
    }
    
    hideLoader();
    toast(`🚀 Procesando ${totalToPublish} anuncios...`);
    
    // ✅ v6.5.5: Activar aviso de publicación en curso
    setPublishingActive(true);
    
    // ✅ v6.12.0: Notificar inicio al tab de progreso
    // ✅ v1.1.0: Pasar isResumeMode para cargar datos de BD
    if (window.ProgressTab?.startPublication) {
      window.ProgressTab.startPublication(totalToPublish, isResumeMode);
    }
    
    // ✅ v10.5.144: Limpiar selección COMPLETA (estado + visual)
    // Usa LISTINGS.clearSelection() que ya hace todo: estado, checkboxes, clase .selected
    LISTINGS.clearSelection();
    
    // 11. Iniciar procesamiento continuo
    await startContinuousProcessing(accountId);
    
  } catch (e) {
    hideLoader();
    // ✅ v6.5.5: Desactivar aviso si hay error
    setPublishingActive(false);
    logger.error('[PUBLISH-FLOW] ❌ Error:', e);
    toast(`❌ Error: ${e.message}`, 'error');
  }
}

// Exponer para uso desde otros módulos
window.handlePublishFlow = handlePublishFlow;

/**
 * Inicia el procesamiento continuo de items pendientes
 */
async function startContinuousProcessing(accountId) {
  if (processingActive) {
    return;
  }

  processingActive = true;
  shouldStopProcessing = false;

  const btnPublish = document.getElementById('btn-publish');
  const btnCancel = document.getElementById('btn-cancel-pending');
  
  if (btnPublish) {
    btnPublish.disabled = true;
    btnPublish.setAttribute('aria-disabled', 'true');
  }
  if (btnCancel) {
    btnCancel.disabled = true;
    btnCancel.setAttribute('aria-disabled', 'true');
  }
  
  let successCount = 0;
  let errorCount = 0;
  let totalToProcess = 0;

  // 🆕 Obtener total de items a procesar
  try {
    const statusRes = await chrome.runtime.sendMessage({
      type: 'API.FETCH_JSON',
      url: `/api/walla/publish2/status?account_id=${accountId}`,
      method: 'GET'
    });
    
    if (statusRes?.ok && statusRes.data) {
      totalToProcess = statusRes.data.pending || 0;
    }
  } catch (e) {
  }

  while (!shouldStopProcessing) {
    try {
      // Procesar siguiente item
      const response = await chrome.runtime.sendMessage({
        type: 'PUBLISH.PROCESS_NEXT',
        account_id: accountId
      });


      if (!response?.ok) {
        errorCount++;
        logger.error('[PANEL] Error procesando item', response?.error); // REGLA #4
        
        // ✅ v10.5.59: Detectar sesión expirada de Wallapop
        if (response?.code === 'wallapop_session_expired') {
          toast('🔒 Wallapop cerró tu sesión. Vuelve a iniciar sesión en Wallapop.', 'error', 10000);
          
          // Mostrar alerta más visible
          setTimeout(() => {
            alert('⚠️ SESIÓN DE WALLAPOP EXPIRADA\n\n' +
              'Wallapop ha cerrado tu sesión después de muchas publicaciones.\n\n' +
              'Para continuar:\n' +
              '1. Cierra sesión en Wallapop\n' +
              '2. Vuelve a iniciar sesión\n' +
              '3. Reintenta publicar los anuncios pendientes');
          }, 500);
          
          // Este error es fatal, romper el bucle
          break;
        }
        
        // ✅ v10.5.63: Errores recuperables - continuar con siguiente anuncio
        // El siguiente anuncio navegará a /upload y obtendrá token nuevo
        const recoverableErrors = [
          'No se pudieron obtener credenciales de Wallapop',
          'token_expired',
          'Token expirado'
        ];
        
        const isRecoverable = recoverableErrors.some(e => 
          response?.error?.includes(e) || response?.code?.includes(e)
        );
        
        if (isRecoverable) {
          console.log('[PANEL] ⚠️ Error recuperable, continuando con siguiente anuncio...');
          toast(`⚠️ Anuncio marcado para reintento`, 'warning', 3000);
          
          // ✅ v10.5.131: Actualizar UI para marcar item como error ANTES de continuar
          // Esto evita que aparezcan dos items en "processing" simultáneamente
          if (response?.item_id && window.ProgressTab?.handleProgressUpdate) {
            window.ProgressTab.handleProgressUpdate({
              item_id: response.item_id,
              status: 'error_retry',
              error: response?.error || 'Token expirado'
            });
          }
          
          // Pequeña espera para que Wallapop genere token nuevo y UI se actualice
          await new Promise(r => setTimeout(r, 2500));
          continue;  // Continuar con el siguiente anuncio
        }
        
        // Para otros errores, mostrar y detener
        const errorMsg = response?.error || 'Error desconocido';
        toast(`❌ Error: ${errorMsg}`, 'error');
        
        // ✅ v7.2.60: Detener pero MOSTRAR MODAL de resumen
        
        // ✅ Obtener stats finales aunque haya error
        try {
          const statsRes = await chrome.runtime.sendMessage({
            type: 'API.FETCH_JSON',
            url: `/api/walla/publish2/stats/${accountId}`,
            method: 'GET'
          });
          
          if (statsRes?.ok && statsRes.data) {
            
            // ✅ v7.2.60: Mostrar modal AUNQUE haya errores
            setTimeout(() => {
              showCompletionModal({
                total: statsRes.data.total || 0,
                published: statsRes.data.published || 0,
                errors: statsRes.data.errors || errorCount
              });
            }, 500);
          }
        } catch (err) {
          logger.error('[PANEL] Error obteniendo stats finales', err);
        }
        
        break;
      }
      
      // ✅ v10.5.43: Si el SW está esperando una alarm (delay largo >60s),
      // esperar a que termine antes de pedir el siguiente item
      if (response.waiting_alarm) {
        console.log(`[PANEL] ⏰ Esperando alarm para delay largo (${response.delay_seconds}s)...`);
        
        // Crear Promise que se resuelve cuando recibimos notificación de publicación completada
        const waitForAlarm = new Promise((resolve) => {
          const alarmHandler = (msg) => {
            if (msg?.type === 'PUBLISH.ALARM_COMPLETED') {
              chrome.runtime.onMessage.removeListener(alarmHandler);
              resolve(msg);
            }
          };
          chrome.runtime.onMessage.addListener(alarmHandler);
          
          // Timeout de seguridad: delay + 60s extra
          const safetyTimeout = (response.delay_seconds + 60) * 1000;
          setTimeout(() => {
            chrome.runtime.onMessage.removeListener(alarmHandler);
            resolve({ timeout: true });
          }, safetyTimeout);
        });
        
        const alarmResult = await waitForAlarm;
        
        if (alarmResult.timeout) {
          console.warn('[PANEL] ⚠️ Timeout esperando alarm');
          // Continuar de todos modos, el SW manejará el estado
        } else if (!alarmResult.ok) {
          console.error('[PANEL] ❌ Error en alarm:', alarmResult.error);
          errorCount++;
        } else {
          console.log('[PANEL] ✅ Alarm completada, continuando...');
          successCount++;
          // ✅ v10.5.49: Toast eliminado - progreso visible en pestaña "En Curso"
        }
        
        // Recargar estado
        await loadAccountState(accountId);
        
        // Continuar con el siguiente item (el while loop pedirá el siguiente)
        continue;
      }

      if (response.done || !response.has_next) {
        try {
          const statsRes = await chrome.runtime.sendMessage({
            type: 'API.FETCH_JSON',
            url: `/api/walla/publish2/stats/${accountId}`,
            method: 'GET'
          });
          
          if (statsRes?.ok && statsRes.data) {
            // Enviar mensaje para mostrar modal
            chrome.runtime.sendMessage({
              type: 'PUBLISH.SHOW_SUCCESS_MODAL',
              stats: {
                total: statsRes.data.total || 0,
                published: statsRes.data.published || 0,
                errors: statsRes.data.errors || 0
              }
            });
          }
        } catch (e) {
        }
        
        // 🆕 Resetear estados de anuncios publicados exitosamente
        try {
          const cleanupRes = await chrome.runtime.sendMessage({
            type: 'API.FETCH_JSON',
            url: `/api/walla/publish2/cleanup/${accountId}`,
            method: 'POST'
          });
          
          if (cleanupRes?.ok) {
          }
        } catch (e) {
        }
        
        // Mensaje final según resultados
        if (errorCount > 0) {
          toast(`✅ Completado: ${successCount} publicados, ${errorCount} con errores`);
        } else {
          toast(`✅ ${successCount} anuncios publicados correctamente`);
        }
        
        // 🆕 Recargar estado del panel para actualizar botón
        await loadAccountState(accountId);
        
        break;
      }

      // Item procesado exitosamente
      successCount++;
      // ✅ v10.5.49: Toast eliminado - progreso visible en pestaña "En Curso"
      
      // Recargar estado para actualizar UI
      await loadAccountState(accountId);

      // ⚠️ El delay ya se aplica en el SW DESPUÉS del DELETE y ANTES de publicar
      // No hace falta delay aquí

    } catch (err) {
      errorCount++;
      console.error('[PANEL] ❌ Error en loop:', err);
      
      // ✅ NO mostrar errores técnicos de Chrome al usuario
      if (!err.message?.includes('listener indicated') && !err.message?.includes('message channel')) {
        toast('❌ Error durante publicación');
      }
      try {
        await loadAccountState(accountId);
      } catch (e) {
      }
      
      // Esperar antes de continuar
      await sleep(5000);
    }
  }

  processingActive = false;
  
  // ✅ v1.5.0: Desactivar sesión de publicación (única fuente de processingCount)
  setPublishingActive(false);
  
  if (btnPublish) {
    btnPublish.disabled = false;
    btnPublish.removeAttribute('aria-disabled');
  }
  if (btnCancel) {
    btnCancel.disabled = false;
    btnCancel.removeAttribute('aria-disabled');
  }
  
  // Recargar estado final
  await loadAccountState(accountId);
  
  if (shouldStopProcessing) {
    toast(`🛑 Cancelado: ${successCount} publicados, ${errorCount} errores`);
  } else {
  }
}

/**
 * Detiene el procesamiento continuo
 */
function stopContinuousProcessing() {
  if (!processingActive) return;
  
  shouldStopProcessing = true;
}

/* ============================================
   ✅ FUNCIÓN MAESTRA: Cargar estado completo de cuenta
   ⚡ OPTIMIZADO: Solo HTTP inicial, luego WebSocket streaming
   ============================================ */
async function loadAccountState(accountId, forceRefresh = false) {
  // ✅ v6.6.0: DEBUG - Log para rastrear llamadas múltiples
  
  if (!accountId || accountId <= 0) {
    // Sin cuenta válida → limpiar todo
    STATE.setUIState('currentAccountId', null);
    STATE.setBackupState({ status: 'idle', pending: 0, errors: 0, seen: 0, total: 0, run_id: null });
    ButtonManager.updateState({
      hasPendingProcess: false,
      pendingProcessId: null,
      pendingType: null
    });
    STATE.setUIState('validation', { duplicates: [], orphans: [], inconsistent_states: [] });
    return;
  }

  // ✅ OPTIMIZACIÓN: Si hay backup activo Y no es force refresh → SKIP HTTP
  // WebSocket ya está actualizando el estado en tiempo real
  if (!forceRefresh && STATE.isBackupActive()) {
    return;
  }

  try {
    // ✅ HTTP solo para carga inicial o refresh explícito
    
    const [backupRes, publishRes] = await Promise.all([
      RUNS.sw({ type: 'RUN.LAST', account_id: accountId }),
      RUNS.sw({ type: 'PUBLISH.PENDING', account_id: accountId })
    ]);

    // 1️⃣ Estado de Backup
    if (backupRes?.ok && backupRes.data?.exists) {
      const { run_id, status, pending, errors, seen, total, viewed_at, validation_warnings } = backupRes.data;
      const statusNormalized = status === 'canceled' ? 'cancelled' : String(status || 'idle').toLowerCase();
      
      STATE.setBackupState({
        status: statusNormalized,
        pending: Number(pending || 0),
        errors: Number(errors || 0),
        seen: Number(seen || 0),
        total: Number(total || 0),
        run_id: run_id || null,
        viewed_at: viewed_at || null
      });
      
      if (validation_warnings) {
        STATE.setUIState('validation', validation_warnings);
      }
    } else {
      STATE.setBackupState({ status: 'idle', pending: 0, errors: 0, seen: 0, total: 0, run_id: null, viewed_at: null });
    }

    // 2️⃣ Estado de Publicación
    const pubData = publishRes?.data ?? publishRes ?? {};
    
    const pending = Number(pubData.pending || 0);
    const processing = Number(pubData.processing || 0); // ✅ v7.2.44
    const errors = Number(pubData.errors || 0);
    const resumable = !!(pubData.resumable || pending > 0 || errors > 0);
    
    // ✅ v1.5.0: Si hay items en processing, activar sesión de publicación
    // Esto puede pasar si se recarga el panel durante una publicación
    if (processing > 0 && !_publishingActive) {
      setPublishingActive(true);
    }
    
    if (resumable) {
      ButtonManager.updateState({
        hasPendingProcess: true,
        pendingProcessId: null,
        pendingType: 'publish',
        pendingCount: pending // ✅ v6.6.0: Número de pendientes
        // ✅ v1.5.0: processingCount se gestiona SOLO desde setPublishingActive()
      });
      
      // ✅ v6.12.0: Actualizar tab de progreso si hay publicación en curso
      if (window.ProgressTab?.switchAccount) {
        window.ProgressTab.switchAccount(accountId).catch(err => {
        });
      }
    } else {
      ButtonManager.updateState({
        hasPendingProcess: false,
        pendingProcessId: null,
        pendingType: null,
        pendingCount: 0 // ✅ v6.6.0: Resetear contador
      });
    }

    // 3️⃣ Validation warnings
    if (pubData.validation_warnings && !backupRes?.data?.validation_warnings) {
      STATE.setUIState('validation', pubData.validation_warnings);
    } else if (!backupRes?.data?.validation_warnings && !pubData.validation_warnings) {
      STATE.setUIState('validation', { duplicates: [], orphans: [], inconsistent_states: [] });
    }

    STATE.setUIState('currentAccountId', accountId);

  } catch (err) {
    console.error('[PANEL] Error cargando estado de cuenta:', err);
    // En caso de error, limpiar
    STATE.setUIState('backup', { status: 'idle', pending: 0, errors: 0 });
    ButtonManager.updateState({
      hasPendingProcess: false,
      pendingProcessId: null,
      pendingType: null
    });
    STATE.setUIState('validation', { duplicates: [], orphans: [], inconsistent_states: [] });
  }
  
  // 5️⃣ Inicializar handlers de publicación DESPUÉS de actualizar estado
}

/* ============================================
   Handlers de botones principales (Backup, Publicar, Resume, etc.)
   ============================================ */

/** Helper: requiere que esté en pestaña de Wallapop activa */
let isBackupRunning = false; // Flag anti-doble click

function runWithWalla(fn) {
  return async (e) => {
    e?.preventDefault?.();
    
    // ✅ Anti-doble click
    if (isBackupRunning) {
      return;
    }
    
    // ✅ Deshabilitar botón y mostrar loader INMEDIATAMENTE
    const btnBackup = STATE.btnBackup;
    if (btnBackup) btnBackup.disabled = true;
    
    showLoader('Verificando Wallapop...', {
      timeout: 30000,
      onTimeout: () => {
        isBackupRunning = false;
        if (btnBackup) btnBackup.disabled = false;
        toast('⏱️ Tiempo agotado verificando Wallapop', 'error', 5000);
      }
    });
    
    isBackupRunning = true;
    
    try {
      const g = await AUTH.requireWallaActive();
      if (!g.ok) {
        hideLoader();
        isBackupRunning = false;
        if (btnBackup) btnBackup.disabled = false;
        return;
      }
      
      await fn(g.tab);
    } catch (err) {
      console.error('[BACKUP] Error:', err);
      hideLoader();
      toast('❌ Error en operación', 'error');
    } finally {
      isBackupRunning = false;
      if (btnBackup) btnBackup.disabled = false;
    }
  };
}

/** 🗑️ ELIMINADO - Helper: Mostrar widget de backup (ya no se usa)
function showBackupWidgetIfNeeded(runId) {
  if (!runId) return;
  
  import('./scripts/panel/backup-widget.js').then(({ showBackupWidget }) => {
    showBackupWidget();
}
*/

/** Notifica que el backup ha iniciado */
/** Botón Backup: lanza export+import */
async function handleBackup(tab) {
  if (!STATE.AUTH.on) {
    toast('Inicia sesión para usar esta función');
    return;
  }

  const btnBackup = STATE.btnBackup;
  const buttonText = btnBackup?.querySelector('.label')?.textContent || '';

  // ✅ Decidir acción según texto del botón
  if (buttonText.startsWith('Reanudar')) {
    // Usuario quiere reanudar
    await handleResumeBackup();
    return;
  }

  // Caso normal: Iniciar nuevo backup
  if (STATE.RUN_ACTIVE) {
    toast('Ya hay un backup en curso', 'warn');
    return;
  }

  // ✅ v6.3.0: runWithWalla ya mostró loader, solo actualizamos texto
  setLoaderText('Preparando backup…');

  // UI.showRunUI(); // DISABLED
  // UI.setBar(0); // DISABLED
  // Eliminado: runStatus ya no existe

  try {
    const hrefNow = tab?.url || '';

    // 1) Ya estás en /user/ → lanzar directo
    if (/^https:\/\/(www\.)?es\.wallapop\.com\/user\//i.test(hrefNow)) {
      // ✅ v6.3.0: Mantener loader visible durante exportación
      setLoaderText('Exportando anuncios de Wallapop…');
      const r = await RUNS.sw({ type: 'EXPORT.PROFILE_AND_IMPORT', profileUrl: hrefNow });

      if (r?.ok) {
        const imp = r?.data?.import || r?.data;
        
        // ✅ Detectar backup ya en curso
        if (imp?.status === 'already_running') {
          hideLoader();
          const elapsed = Math.floor((imp.elapsed_seconds || 0) / 60);
          const wait = Math.floor((imp.wait_seconds || 0) / 60);
          toast(`⏳ Backup en curso desde hace ${elapsed} min. Intenta en ${wait} min.`, 'warn');
          return;
        }
        
        if (imp?.status === 'empty') {
          hideLoader();
          toast('No hay anuncios en Wallapop', 'warn');
          return;
        }

        // ✅ v6.3.0: Ocultar loader y mostrar barra de progreso
        hideLoader();
        const totalItems = imp?.queued || imp?.run?.total || 0;
        UI.updateProgressBar({
          operation: 'backup',
          current: 0,
          total: totalItems,
          active: true
        });
        
        toast(`🚀 Backup iniciado: ${totalItems} anuncios`, 'info', 3000);
      } else {
        // ✅ v10.5.81: Detectar error de cuenta duplicada (409)
        const errorDetail = r?.error || r?.data?.import?.detail || '';
        if (errorDetail.includes('ya está vinculada')) {
          hideLoader();
          await showModal({
            title: '⚠️ Cuenta ya vinculada',
            message: errorDetail,
            buttons: [{ text: 'Entendido', value: 'ok', primary: true }]
          });
        } else {
          toast('Export+Import error', 'err');
        }
      }
      return;
    }

    // 2) FAST: pedir URL de perfil SIN navegar (usando iframe interno)
    setLoaderText('Conectando con Wallapop…');
    let prof = null;
    try {
      const info = await chrome.tabs.sendMessage(tab.id, { type: 'DOM.CURRENT_PROFILE' });
      prof = info?.profileUrl || null;
      if (!prof) {
        const r0 = await chrome.tabs.sendMessage(tab.id, { type: 'DOM.GET_PROFILE_URL' });
        prof = r0?.url || null;
      }
    } catch {}

    if (prof) {
      // ✅ v6.3.0: Mantener loader visible durante exportación
      setLoaderText('Exportando anuncios de Wallapop…');
      const r = await RUNS.sw({ type: 'EXPORT.PROFILE_AND_IMPORT', profileUrl: prof });

      if (r?.ok) {
        const imp = r?.data?.import || r?.data;
        
        // ✅ Detectar backup ya en curso
        if (imp?.status === 'already_running') {
          hideLoader();
          const elapsed = Math.floor((imp.elapsed_seconds || 0) / 60);
          const wait = Math.floor((imp.wait_seconds || 0) / 60);
          toast(`⏳ Backup en curso desde hace ${elapsed} min. Intenta en ${wait} min.`, 'warn');
          return;
        }
        
        if (imp?.status === 'empty') {
          hideLoader();
          toast('No hay anuncios en Wallapop', 'warn');
          return;
        }

        // ✅ v6.3.0: Ocultar loader y mostrar barra de progreso
        hideLoader();
        const totalItems = imp?.queued || imp?.run?.total || 0;
        UI.updateProgressBar({
          operation: 'backup',
          current: 0,
          total: totalItems,
          active: true
        });
        
        toast(`🚀 Backup iniciado: ${totalItems} anuncios`, 'info', 3000);
      } else {
        // ✅ v10.5.81: Detectar error de cuenta duplicada (409)
        const errorDetail = r?.error || r?.data?.import?.detail || '';
        if (errorDetail.includes('ya está vinculada')) {
          hideLoader();
          await showModal({
            title: '⚠️ Cuenta ya vinculada',
            message: errorDetail,
            buttons: [{ text: 'Entendido', value: 'ok', primary: true }]
          });
        } else {
          toast('Export+Import error', 'err');
        }
      }
      return;
    }

    // 3) FALLBACK: intentar BACKUP.START
    setLoaderText('Localizando perfil…');
    const start = await new Promise(res => {
      try {
        chrome.tabs.sendMessage(tab.id, { type: 'BACKUP.START' }, m => res(m || { ok: false }));
      } catch {
        res({ ok: false });
      }
    });

    if (start?.ok) {
      const fresh = await chrome.tabs.get(tab.id);
      const prof2 = fresh?.url || start?.href || '';
      // ✅ v6.3.0: Mantener loader visible durante exportación
      setLoaderText('Exportando anuncios de Wallapop…');
      const r2 = await RUNS.sw({ type: 'EXPORT.PROFILE_AND_IMPORT', profileUrl: prof2 });

      if (r2?.ok) {
        const imp = r2?.data?.import || r2?.data;
        
        // ✅ Detectar backup ya en curso
        if (imp?.status === 'already_running') {
          hideLoader();
          const elapsed = Math.floor((imp.elapsed_seconds || 0) / 60);
          const wait = Math.floor((imp.wait_seconds || 0) / 60);
          toast(`⏳ Backup en curso desde hace ${elapsed} min. Intenta en ${wait} min.`, 'warn');
          return;
        }
        
        // ✅ v6.3.0: Ocultar loader y mostrar barra de progreso
        hideLoader();
        const totalItems = imp?.queued || imp?.run?.total || 0;
        UI.updateProgressBar({
          operation: 'backup',
          current: 0,
          total: totalItems,
          active: true
        });
        
        toast(`🚀 Backup iniciado: ${totalItems} anuncios`, 'info', 3000);
      } else {
        // ✅ v10.5.81: Detectar error de cuenta duplicada (409)
        const errorDetail = r2?.error || r2?.data?.import?.detail || '';
        if (errorDetail.includes('ya está vinculada')) {
          hideLoader();
          await showModal({
            title: '⚠️ Cuenta ya vinculada',
            message: errorDetail,
            buttons: [{ text: 'Entendido', value: 'ok', primary: true }]
          });
        } else {
          toast('Export+Import error', 'err');
        }
      }
      return;
    }

    toast('No se pudo localizar el perfil', 'warn');

  } catch {
    toast('No se pudo iniciar el backup', 'err');
  } finally {
    hideLoader();
    // ✅ runWithWalla gestiona el botón, no lo tocamos aquí
  }
}

/** Botón Publicar: obtiene credenciales de Wallapop */
async function handlePublish(tab) {
  if (!STATE.AUTH.on) {
    toast('Inicia sesión en MitikLive', 'warn');
    return;
  }

  const btnPublish = STATE.btnPublish;
  if (btnPublish) btnPublish.disabled = true;
  showLoader('Preparando publicación…', {
    timeout: 30000,
    onTimeout: () => toast('⏱️ Tiempo agotado preparando publicación', 'error', 5000)
  });

  try {
    // Verificar login en Wallapop
    const logged = await DETECT.isLoggedInWallapopTab(tab.id);
    if (!logged) {
      hideLoader();
      toast('Inicia sesión en Wallapop y vuelve a pulsar "Publicar".', 'warn');
      await chrome.tabs.update(tab.id, { url: 'https://es.wallapop.com/app/login' });
      return;
    }

    // Aviso opcional si selector ≠ alias actual
    const selAlias = UI.getSelectedAlias();
    if (selAlias && STATE.LAST_GOOD.alias && selAlias !== STATE.LAST_GOOD.alias) {
      toast(`Cuenta seleccionada (${selAlias}) no coincide con la activa (${STATE.LAST_GOOD.alias})`, 'warn');
    }

    // Obtener credenciales frescas
    const r = await RUNS.sw({ type: 'WALLA.CREDS' });
    if (!r?.ok) {
      toast('No se pudo obtener token de Wallapop', 'err');
      return;
    }

    hideLoader();
    // ❌ DESHABILITADO: Toast técnico innecesario para usuario
    // toast(`Token listo. DeviceId: ${r.data.deviceId || '—'}`, 'ok');

  } finally {
    hideLoader();
    if (btnPublish) btnPublish.disabled = false;
  }
}

/** Botón Reanudar (dentro de la run box) */
async function handleResume() {
  if (!STATE.AUTH.on) {
    toast('Inicia sesión para usar esta función');
    return;
  }

  const rid = STATE.currentRunId || 
              Number(STATE.btnResume?.dataset?.runId || 0) || 
              STATE.LAST_PENDING_RUN_ID;

  if (!rid) {
    toast('No hay proceso pendiente', 'warn');
    return;
  }

  await RUNS.resumeRun(rid);
}

/** Botón Cancelar (dentro de la run box) */
/** ✅ Reanudar desde botón Backup principal */
async function handleResumeBackup() {
  if (!STATE.AUTH.on) {
    toast('Inicia sesión primero', 'warn');
    return;
  }
  
  const accountId = AUTH.getActiveAccountId();
  if (!accountId) {
    toast('Selecciona una cuenta primero', 'warn');
    return;
  }
  
  // Usar run_id del estado centralizado
  const runId = STATE.BACKUP_STATE.run_id;
  
  
  if (!runId) {
    toast('No hay backup pendiente', 'warn');
    return;
  }
  
  const btnBackup = STATE.btnBackup;
  if (btnBackup) btnBackup.disabled = true;
  
  // Loader eliminado
  // Ya hay progreso en tiempo real en la run box, no hace falta este loader
  
  try {
    await RUNS.resumeRun(runId);
  } catch (err) {
    console.error('[PANEL] Error reanudando:', err);
    toast('Error al reanudar backup', 'err');
  } finally {
    // ❌ hideLoader() ya no necesario
    if (btnBackup) btnBackup.disabled = false;
  }
}

/** Botón Reanudar Publicación */
/** Botón Reanudar Publicación */
/** ✨ NUEVO: Handler para botón "Preparar publicación (X)" */
async function handlePreparePublish() {
  
  // Verificar autenticación
  if (!STATE.AUTH.on) {
    toast('⚠️ Debes iniciar sesión en MitikLive primero', 'warn');
    return;
  }
  
  const btnPrepare = document.getElementById('btnPreparePublish');
  if (btnPrepare) btnPrepare.disabled = true;
  
  try {
    // ✅ v10.5.82: getSelectAllByFilter viene de module-state.js
    const selectAllActive = getSelectAllByFilter();
    const dbStats = AppState.getDbStats();
    
    let activeIds = [];
    let totalToPublish = 0;
    
    if (selectAllActive) {
      // Modo "seleccionar todos": enviar array vacío para que el backend tome todos
      activeIds = []; // Array vacío = backend toma todos los activos
      totalToPublish = dbStats.active || 0;
      
      if (totalToPublish === 0) {
        toast('❌ No hay anuncios activos para publicar', 'warn');
        return;
      }
      
    } else {
      // Modo normal: obtener IDs de checkboxes seleccionados
      const checkboxes = document.querySelectorAll('input[type="checkbox"].listing-checkbox:checked');
      const selectedIds = Array.from(checkboxes)
        .map(cb => cb.dataset.id)
        .filter(id => id && id !== '');
      
      
      if (selectedIds.length === 0) {
        toast('❌ No hay anuncios seleccionados', 'warn');
        return;
      }
      
      // 🔒 VALIDAR: Filtrar solo anuncios activos (doble validación por seguridad)
      activeIds = Array.from(checkboxes)
        .filter(cb => {
          const tr = cb.closest('tr');
          const status = tr?.dataset?.status;
          return status === 'active';
        })
        .map(cb => cb.dataset.id)
        .filter(id => id && id !== '');
      
      if (activeIds.length === 0) {
        toast('❌ No hay anuncios activos seleccionados. Solo se pueden publicar anuncios activos.', 'warn');
        return;
      }
      
      if (activeIds.length < selectedIds.length) {
        const diff = selectedIds.length - activeIds.length;
        toast(`⚠️ Se omitieron ${diff} anuncios no activos. Solo se publicarán ${activeIds.length} activos.`, 'warn');
      }
      
      totalToPublish = activeIds.length;
    }
    
    const account_id = AUTH.getActiveAccountId();
    if (!account_id) {
      toast('❌ No hay cuenta activa seleccionada', 'warn');
      return;
    }
    
    showLoader(`Preparando ${totalToPublish} anuncios...`, {
      timeout: 30000,
      onTimeout: () => toast('⏱️ Tiempo agotado preparando anuncios', 'error', 5000)
    });
    
    // Llamar al backend usando el endpoint existente con listing_ids
    const response = await RUNS.sw({
      type: 'API.FETCH_JSON',
      url: `/api/walla/publish/start`,
      method: 'POST',
      body: {
        account_id: account_id,
        listing_ids: activeIds  // ← 🔒 Enviar IDs o [] para todos
      }
    });
    
    // ✅ v10.5.82: Limpiar flag después de usar
    if (selectAllActive) {
      clearSelectAllByFilter();
    }
    
    hideLoader();
    
    if (response?.ok && response.data?.ok) {
      const { run_id, pending } = response.data;
      toast(`✅ ${pending || activeIds.length} anuncios preparados para publicar`, 'ok');
      
      // Desmarcar checkboxes
      checkboxes.forEach(cb => cb.checked = false);
      
      // ✅ Ya no necesario - ButtonManager se actualiza automáticamente
      
      // Recargar estado para mostrar botones de Publicar/Cancelar
      const accountId = AUTH.getActiveAccountId();
      if (accountId) {
        setTimeout(async () => {
          await RUNS.refreshPublishUI(accountId);
          
          // Recargar tabla de anuncios
          const { loadListingsTable } = await import('./scripts/panel/listings/index.js');
          await loadListingsTable(accountId);
        }, 500);
      }
    } else {
      const errorMsg = response?.data?.detail || response?.data?.message || response?.data?.error || 'Error desconocido';
      toast(`❌ Error al preparar publicación: ${errorMsg}`, 'err');
      console.error('[PANEL] Error preparando publicación:', response);
    }
    
  } catch (err) {
    hideLoader();
    
    // ✅ NO mostrar errores técnicos de Chrome al usuario
    if (!err.message?.includes('listener indicated') && !err.message?.includes('message channel')) {
      toast('❌ Error al publicar', 'err');
    }
    console.error('[PANEL] ❌ Error crítico:', err);
    console.error('[PANEL] Excepción en handlePreparePublish:', err);
  } finally {
    if (btnPrepare) btnPrepare.disabled = false;
  }
}

/** Botón Cancelar Publicación */

/* ============================================
   ✅ ACTUALIZACIÓN DE ESTADO Y BOTONES
   ============================================ */
/**
 * Determina el estado del usuario y actualiza los botones según corresponda
 * @param {Number} accountId - ID de la cuenta seleccionada
 */
async function updateUserStateAndButtons(accountId) {
  if (!accountId) return;
  
  try {
    // Obtener número de listings de la tabla
    const tbody = document.querySelector('#listings-tbody');
    const rowCount = tbody ? tbody.querySelectorAll('tr:not(.loading-row)').length : 0;
    
    // Obtener créditos
    const creditCountEl = document.getElementById('credit-count');
    const credits = parseInt(creditCountEl?.textContent || '0', 10);
    
    // Determinar estado del usuario
    const userState = await STATE.determineUserState([{ id: accountId }], rowCount);
    
    
    // Actualizar botones según estado
    UI.renderActionButtons(userState, credits);
    
    // Si es primera vez con anuncios descargados, marcar como experimentado
    if (userState === USER_STATES.HAS_ACCOUNTS_WITH_LISTINGS && rowCount > 0) {
      await STATE.markUserAsExperienced();
    }
    
  } catch (err) {
    console.error('[PANEL] Error actualizando estado de usuario:', err);
  }
}

/** ✅ Selector de cuenta: sincronización completa */
async function handleAccountChange() {
  const opt = STATE.selAccount?.selectedOptions?.[0];
  const accountId = opt?.dataset?.accountId ? Number(opt.dataset.accountId) : null;
  
  // ✅ Si no hay cuenta válida (placeholder) → limpiar TODO
  if (!accountId || accountId <= 0) {
    
    // Ocultar loader si estaba visible
    UI.hideLoader?.();
    
    // 🔥 Ocultar stats
    const statsWrap = document.querySelector('.ml-listings-stats');
    if (statsWrap) {
      statsWrap.querySelectorAll('.stat-value').forEach(el => el.textContent = '0');
      DOMUtils.hide(statsWrap);
    }
    
    // 🔥 Ocultar controles (buscador, contador)
    const controls = document.querySelector('.ml-listings-controls');
    if (controls) DOMUtils.hide(controls);
    
    // 🔥 Ocultar tabla
    const tableWrapper = document.querySelector('.ml-listings-table-wrapper');
    if (tableWrapper) DOMUtils.hide(tableWrapper);
    
    // 🔥 Limpiar tbody
    const tbody = document.querySelector('#listings-table tbody, #listings-tbody');
    if (tbody) tbody.innerHTML = '';
    
    // 🔥 Ocultar sección completa
    const listingsSection = document.querySelector('.ml-listings-section');
    if (listingsSection) DOMUtils.hide(listingsSection);
    
    // Limpiar estado
    loadAccountState(null);
    
    return;
  }
  
  // ✅ v6.2.0: Guardar en BD para persistencia entre dispositivos
  
  // ✅ v6.8.2: Gestionar visibilidad de barra de progreso según cuenta (centralizado)
  const barRestored = UI.showProgressBarIfSameAccount?.(accountId);
  if (!barRestored) {
    UI.hideProgressBarIfDifferentAccount?.(accountId);
  }
  
  // 💾 Guardar en storage local (caché)
  await stSet({ selected_account_id: accountId }).catch(err => {
  });
  
  // 💾 Guardar en BD via SW (persistencia real)
  try {
    const response = await new Promise((resolve) => {
      chrome.runtime.sendMessage({
        type: 'USER.PREFERENCES.UPDATE',
        payload: { last_selected_account_id: accountId }
      }, resolve);
    });
    
    if (response?.ok) {
    } else {
    }
  } catch (err) {
  }
  
  // 🔥 Mostrar UI completa
  const statsWrap = document.querySelector('.ml-listings-stats');
  if (statsWrap) DOMUtils.show(statsWrap);
  
  const controls = document.querySelector('.ml-listings-controls');
  if (controls) DOMUtils.show(controls);
  
  const tableWrapper = document.querySelector('.ml-listings-table-wrapper');
  if (tableWrapper) DOMUtils.show(tableWrapper);
  
  const listingsSection = document.querySelector('.ml-listings-section');
  if (listingsSection) DOMUtils.show(listingsSection);
  
  UI.uiFromOption(opt);
  
  // ✅ v10.5.82: Actualizar fecha/hora de última sincronización en botón
  const lastBackupDate = document.getElementById('last-backup-date');
  if (lastBackupDate) {
    const timestamp = opt?.dataset?.lastBackupAt;
    if (timestamp) {
      const date = new Date(parseInt(timestamp) * 1000);
      const day = date.getDate();
      const month = date.getMonth() + 1;
      const hours = date.getHours();
      const minutes = String(date.getMinutes()).padStart(2, '0');
      const formatted = `${day}/${month} ${hours}:${minutes}`;
      lastBackupDate.textContent = formatted;
      lastBackupDate.title = `Última sincronización: ${date.toLocaleString('es-ES')}`;
    } else {
      lastBackupDate.textContent = '';
    }
  }
  
  // 🔄 Mostrar loader mientras se cargan los datos
  const accountAlias = opt?.dataset?.alias || opt?.textContent || 'cuenta';
  UI.showLoader?.(`Cargando anuncios de ${accountAlias}...`);
  
  // ✅ Marcar como loading
  STATE.setUIState('loading', true);
  
  try {
    // ✅ Cargar en paralelo: estado completo + tabla
    await Promise.all([
      loadAccountState(accountId),
      LISTINGS?.loadListingsTable ? LISTINGS.loadListingsTable(accountId) : Promise.resolve()
    ]);
    
    // ✅ Determinar estado del usuario y actualizar botones
    await updateUserStateAndButtons(accountId);
    
    // ✅ v10.5.153: Inicializar onboarding si no se ha inicializado
    // (esto maneja el caso cuando el usuario hace login DESPUÉS de abrir el panel)
    if (ONBOARDING?.init && !window._onboardingInitialized) {
      window._onboardingInitialized = true;
      setTimeout(() => {
        ONBOARDING.init();
      }, 1500);
    }
    
  } finally {
    STATE.setUIState('loading', false);
    // ✅ Ocultar loader
    UI.hideLoader?.();
  }
}

/* ==================
   Overlay legal + Banner de cookies
   ================== */
async function initLegal() {
  const { LEGAL } = STATE;
  const COOKIE_KEY = 'COOKIE_CONSENT_V1';

  // Listeners del overlay
  LEGAL.backBtn?.addEventListener('click', UI.closeLegal);
  LEGAL.closeBtn?.addEventListener('click', UI.closeLegal);
  
  LEGAL.overlay?.addEventListener('click', (e) => {
    if (e.target === LEGAL.overlay) UI.closeLegal();
  });
  
  document.addEventListener('keydown', (e) => {
    if (e.key === 'Escape' && !LEGAL.overlay?.hidden) UI.closeLegal();
  });

  // Banner de cookies
  try {
    const stored = await stGet([COOKIE_KEY]);
    const consent = stored?.[COOKIE_KEY];
    
    if (!consent && LEGAL.cookieBanner) {
      LEGAL.cookieBanner.hidden = false;
      
      // dentro de initLegal()
      LEGAL.cookieAccept?.addEventListener('click', async () => {
        try {
          await stSet({ [COOKIE_KEY]: { accepted: true, ts: Date.now() } });
        } catch (e) {
        }
        LEGAL.cookieBanner.hidden = true;          // usa [hidden]
        DOMUtils.hide(cookieBanner); // por si acaso
        toast('Gracias. Preferencias guardadas.', 'ok');
      }, { once: true });

    }
  } catch {}
}

/* ============================================
   ✅ INIT PRINCIPAL CON CARGA DE ESTADO
   ============================================ */
(async function initPanel() {
  try {
    // ═══════════════════════════════════════════════════════════════════
    // ✅ v2.2.3: VERIFICAR VERSIÓN ANTES DE TODO (sin requerir login)
    // Bloquea panel completo si la extensión está desactualizada
    // ═══════════════════════════════════════════════════════════════════
    try {
      const needsUpdate = await checkVersionWithoutAuth();
      if (needsUpdate) {
        logger.warn('[PANEL] Extensión desactualizada - panel bloqueado');
        return;  // No continuar inicialización
      }
    } catch (versionErr) {
      logger.error('[PANEL] Error verificando versión:', versionErr);
      // Continuar en caso de error (fail-open) para no bloquear usuarios
    }
    
    // Versión + marca de preparado
    try {
      if (STATE.verEl) {
        STATE.verEl.textContent = chrome.runtime.getManifest().version || '—';
      }
    } catch {}
    
    document.documentElement.classList.add('ml-ready');

    // Render intro
    const slot = document.getElementById('intro-slot');
    if (slot) {
      fetch(chrome.runtime.getURL('intro.html'))
        .then(r => r.text())
        .then(html => { slot.innerHTML = html; })
        .catch(err => { console.error(`[${filepath.split('/').pop()}] Error:`, err); });
    }

    // Inicializaciones modulares
    if (UI?.init) await UI.init();
    if (AUTH?.init) await AUTH.init();
    if (ButtonManager?.init) ButtonManager.init();  // ✅ Inicializar ButtonManager
    
    // 🔒 SEGURIDAD: Verificar versión al abrir panel
    // (Mantenimiento NO se verifica aquí - middleware backend lo maneja)
    if (AUTH?.state === 'AUTHENTICATED') {
      try {
        // ❌ VALIDACIÓN DE VERSIÓN DESHABILITADA (causaba pool exhaustion)
        // await checkVersion();
        
        // ❌ MANTENIMIENTO: No verificar aquí
        // El middleware backend rechaza peticiones (HTTP 503) si hay mantenimiento
        // Si el usuario tiene token válido, el middleware decide por cada petición
        
      } catch (err) {
        logger.error('[PANEL] Error en check de versión:', err);
        // En caso de error, permitir continuar (fail-open)
      }
    }
    
    // ✅ CORRECCIÓN: Dot actualizado solo por eventos WebSocket
    // Los eventos 'connected'/'disconnected' (líneas 1289-1324) actualizan el dot
    // No necesitamos consulta manual que puede desincronizar
    
    if (MENU?.init) await MENU.init();
    if (DETECT?.init) await DETECT.init();
    if (LISTENERS?.init) await LISTENERS.init();
    if (POLLING?.init) await POLLING.init();
    if (RESUME?.init) await RESUME.init();
    if (LISTINGS?.init) await LISTINGS.init();
    if (CONFIG?.initConfig) await CONFIG.initConfig();
    
    // ✅ Exportar CONFIG globalmente para tabs.js
    window.CONFIG = CONFIG;
    
    // ✅ Inicializar sistema de tabs y logger
    initTabs();
    
    // La verificación se hace SOLO en login (según decisión del cliente)
    // Razón: Sin tráfico extra, modo mantenimiento funciona al hacer logout/login

    // Detección inicial de Wallapop
    await DETECT.detectLoggedWalla();

    // ✅ Si está autenticado → cargar estado completo de cuenta actual
    if (STATE.AUTH.state === 'AUTHENTICATED') {
      const currentAccountId = AUTH.getActiveAccountId();
      
      // 🆕 Inicializar sistema de créditos si ya está logueado
      if (window.CreditsManager && typeof window.CreditsManager.init === 'function') {
        await window.CreditsManager.init();
      }
      
      // ✅ v10.5.146: Sistema de onboarding (SOLO si autenticado)
      if (ONBOARDING?.init) {
        window._onboardingInitialized = true;
        // Esperar a que se carguen los listings para tener stats correctas
        setTimeout(() => {
          ONBOARDING.init();
        }, 1500);  // Aumentado a 1.5s para dar más tiempo
      }
      
      if (currentAccountId && currentAccountId > 0) {
        // ✅ Cargar estado completo Y tabla de listings
        await Promise.all([
          loadAccountState(currentAccountId),
          LISTINGS?.loadListingsTable ? LISTINGS.loadListingsTable(currentAccountId) : Promise.resolve()
        ]);
        
        // ✅ v7.2.14: Triple refresh para asegurar botones (timing de inicialización)
        // 1. Inmediato (después de loadAccountState)
        if (ButtonManager?.refreshAllButtons) {
          ButtonManager.refreshAllButtons();
        }
        
        // 2. Delay corto (después de renderizado inicial)
        setTimeout(() => {
          if (ButtonManager?.refreshAllButtons) {
            ButtonManager.refreshAllButtons();
          }
        }, 300);
        
        // 3. Delay largo (asegurar todo inicializado)
        setTimeout(() => {
          if (ButtonManager?.refreshAllButtons) {
            ButtonManager.refreshAllButtons();
          }
        }, 1000);
      }
      
      // Avisos opcionales (sin modales invasivos)
      await POLLING.checkLastRunAndWarn();
      await RESUME.promptResumeIfPending();
      
      // ✅ NO llamar promptPublishResumeIfPending() aquí
      // Ya se cargó el estado con loadAccountState()
    }

    // Conectar listeners de botones
    if (STATE.btnBackup) {
      STATE.btnBackup.addEventListener('click', runWithWalla(handleBackup));
    }

    if (STATE.btnPublish) {
      STATE.btnPublish.addEventListener('click', runWithWalla(handlePublish));
    }

    if (STATE.btnResume) {
      STATE.btnResume.addEventListener('click', handleResume);
    }

    if (STATE.btnHide) {
      STATE.btnHide.addEventListener('click', handleHideProgress);
    }

    // ✅ NUEVO: Listener para botón "Preparar publicación"
    const btnPreparePublish = document.getElementById('btnPreparePublish');
    if (btnPreparePublish) {
      btnPreparePublish.addEventListener('click', handlePreparePublish);
    }

    if (STATE.selAccount) {
      STATE.selAccount.addEventListener('change', handleAccountChange);
    }

    // Inicializar overlay legal y banner de cookies
    await initLegal();

  } catch (e) {
    try {
      toast('Error iniciando panel', 'err');
    } catch {}
    console.error('[MitikLive] initPanel error:', e);
  }
})();

/* ============================================
   🎨 SISTEMA DE ZOOM GLOBAL
   ============================================ */

// Constantes de zoom (rango ampliado: 60% - 150%)
const ZOOM_LEVELS = [0.60, 0.65, 0.70, 0.75, 0.80, 0.85, 0.90, 0.95, 1.0, 1.05, 1.10, 1.15, 1.20, 1.25, 1.30, 1.35, 1.40, 1.45, 1.50];
const ZOOM_DEFAULT = 1.0;
const ZOOM_STORAGE_KEY = 'mitiklive_ui_scale';

// Estado actual
let currentZoomIndex = ZOOM_LEVELS.indexOf(ZOOM_DEFAULT);

// Elementos
// Botones de zoom (ahora en Config)
const btnZoomOut = document.getElementById('btn-zoom-out-config');
const btnZoomIn = document.getElementById('btn-zoom-in-config');
const btnZoomReset = document.getElementById('btn-zoom-reset-config');

/**
 * Aplicar nivel de zoom al documento
 */
async function applyZoom(scale) {
  document.documentElement.style.setProperty('--ui-scale', scale);
  
  // Actualizar label del botón reset
  const percentage = Math.round(scale * 100);
  if (btnZoomReset) {
    btnZoomReset.textContent = `${percentage}%`;
  }
  
  // ✅ Guardar en BD (con fallback a localStorage)
  try {
    await RUNS.sw({
      type: 'USER.PREFERENCES.UPDATE',
      payload: { 
        ui_preferences: { 
          zoom: Math.round(percentage)  // Guardar como porcentaje entero
        } 
      }
    });
  } catch (e) {
    try {
      localStorage.setItem(ZOOM_STORAGE_KEY, scale.toString());
    } catch (e2) {
    }
  }
  
}

/**
 * Aumentar zoom
 */
function zoomIn() {
  if (currentZoomIndex < ZOOM_LEVELS.length - 1) {
    currentZoomIndex++;
    applyZoom(ZOOM_LEVELS[currentZoomIndex]);
  }
}

/**
 * Reducir zoom
 */
function zoomOut() {
  if (currentZoomIndex > 0) {
    currentZoomIndex--;
    applyZoom(ZOOM_LEVELS[currentZoomIndex]);
  }
}

/**
 * Resetear zoom a valor por defecto
 */
function zoomReset() {
  currentZoomIndex = ZOOM_LEVELS.indexOf(ZOOM_DEFAULT);
  applyZoom(ZOOM_DEFAULT);
}

/**
 * Cargar zoom guardado (primero desde BD, luego localStorage)
 */
async function loadSavedZoom() {
  try {
    // ✅ Intentar cargar desde BD primero
    const prefs = await RUNS.sw({ type: 'USER.PREFERENCES.GET' });
    const zoomPercent = prefs?.ok && prefs?.data?.ui_preferences?.zoom 
      ? Number(prefs.data.ui_preferences.zoom) 
      : null;
    
    if (zoomPercent) {
      const scale = zoomPercent / 100;
      const index = ZOOM_LEVELS.indexOf(scale);
      
      if (index !== -1) {
        currentZoomIndex = index;
        // No usar await aquí porque applyZoom ahora es async
        applyZoom(scale);
        return;
      }
    }
    
    // Fallback a localStorage si no hay en BD
    const saved = localStorage.getItem(ZOOM_STORAGE_KEY);
    if (saved) {
      const scale = parseFloat(saved);
      const index = ZOOM_LEVELS.indexOf(scale);
      
      if (index !== -1) {
        currentZoomIndex = index;
        applyZoom(scale);
        return;
      }
    }
  } catch (e) {
  }
  
  // Si no hay guardado o es inválido, usar default
  applyZoom(ZOOM_DEFAULT);
}

// Inicializar zoom al cargar
loadSavedZoom();

// ✅ v4.0.8: Slider de zoom con debounce y algoritmo mejorado
const zoomSlider = document.getElementById('zoom-slider-config');
const zoomDisplay = document.getElementById('config-zoom-display');

if (zoomSlider && zoomDisplay) {
  // Sincronizar slider con zoom actual
  const updateSliderFromZoom = () => {
    const percentage = Math.round(ZOOM_LEVELS[currentZoomIndex] * 100);
    zoomSlider.value = percentage;
    zoomSlider.setAttribute('aria-valuenow', percentage);
    zoomDisplay.textContent = `${percentage}%`;
  };
  
  // Actualizar al inicio
  updateSliderFromZoom();
  
  // ✅ Función para encontrar el nivel MÁS CERCANO (no el primero >= scale)
  const findClosestZoomLevel = (scale) => {
    return ZOOM_LEVELS.reduce((prev, curr, i) => {
      return Math.abs(curr - scale) < Math.abs(ZOOM_LEVELS[prev] - scale) ? i : prev;
    }, 0);
  };
  
  // ✅ Aplicar zoom con debounce (solo guardar en BD después de 300ms sin cambios)
  const debouncedSaveZoom = debounce(async (scale) => {
    await applyZoom(scale);
  }, 300);
  
  // Listener del slider
  zoomSlider.addEventListener('input', (e) => {
    const percentage = parseInt(e.target.value);
    
    // ✅ Validar entrada
    if (isNaN(percentage) || percentage < 70 || percentage > 130) {
      return;
    }
    
    const scale = percentage / 100;
    
    // ✅ Buscar el nivel más cercano (no el primero >=)
    const index = findClosestZoomLevel(scale);
    currentZoomIndex = index;
    
    // ✅ Actualizar UI inmediatamente (sin esperar BD)
    const actualScale = ZOOM_LEVELS[currentZoomIndex];
    const actualPercentage = Math.round(actualScale * 100);
    document.documentElement.style.setProperty('--ui-scale', actualScale);
    zoomDisplay.textContent = `${actualPercentage}%`;
    zoomSlider.setAttribute('aria-valuenow', actualPercentage);
    
    // ✅ Guardar en BD con debounce (evita muchas escrituras)
    debouncedSaveZoom(actualScale);
  });
}

// Event listeners botones (mantener compatibilidad, aunque no existen en HTML)
if (btnZoomIn) {
  btnZoomIn.addEventListener('click', () => {
    zoomIn();
    if (zoomSlider && zoomDisplay) {
      const percentage = Math.round(ZOOM_LEVELS[currentZoomIndex] * 100);
      zoomSlider.value = percentage;
      zoomSlider.setAttribute('aria-valuenow', percentage);
      zoomDisplay.textContent = `${percentage}%`;
    }
  });
}

if (btnZoomOut) {
  btnZoomOut.addEventListener('click', () => {
    zoomOut();
    if (zoomSlider && zoomDisplay) {
      const percentage = Math.round(ZOOM_LEVELS[currentZoomIndex] * 100);
      zoomSlider.value = percentage;
      zoomSlider.setAttribute('aria-valuenow', percentage);
      zoomDisplay.textContent = `${percentage}%`;
    }
  });
}

if (btnZoomReset) {
  btnZoomReset.addEventListener('click', () => {
    zoomReset();
    if (zoomSlider && zoomDisplay) {
      zoomSlider.value = 100;
      zoomSlider.setAttribute('aria-valuenow', 100);
      zoomDisplay.textContent = '100%';
    }
  });
}

// Atajos de teclado (Ctrl/Cmd + +/-/0)
document.addEventListener('keydown', (e) => {
  const isMac = navigator.platform.toUpperCase().indexOf('MAC') >= 0;
  const modKey = isMac ? e.metaKey : e.ctrlKey;
  
  if (!modKey) return;
  
  // Ctrl/Cmd + +
  if (e.key === '+' || e.key === '=') {
    e.preventDefault();
    zoomIn();
  }
  
  // Ctrl/Cmd + -
  if (e.key === '-' || e.key === '_') {
    e.preventDefault();
    zoomOut();
  }
  
  // Ctrl/Cmd + 0
  if (e.key === '0') {
    e.preventDefault();
    zoomReset();
  }
});


// ===== LISTENER PARA PUBLICACIÓN COMPLETADA =====
chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
  // 🐛 DEBUG: Log mensajes relevantes (filtrar spam)
  const spamTypes = ['SERVER_ALIVE', 'HEARTBEAT', 'WALLA.SESSION_CHANGED', 'ACTIVITY.UPDATE', 'SESSION.STATE'];
  if (msg?.type && !spamTypes.includes(msg.type)) {
  }
  
  // ✅ FORZAR MODAL DE ACTUALIZACIÓN (desde backend HTTP 426)
  if (msg?.type === 'FORCE_UPDATE_MODAL') {
    
    import('./scripts/panel/maintenance.js').then(({ showUpdateModalForced }) => {
      showUpdateModalForced(msg.data.minimum_version, msg.data.download_url);
    }).catch(err => {
      console.error('[PANEL] Error mostrando modal forzada:', err);
    });
    
    sendResponse({ ok: true });
    return;
  }
  
  // ✅ v10.5.53: Mensaje de paso dinámico (desde form-filler)
  if (msg?.type === 'PUBLISH.STEP_MESSAGE') {
    if (window.ProgressTab?.updateStepMessage) {
      window.ProgressTab.updateStepMessage(msg.message, msg.step || 1);
    }
    sendResponse({ ok: true });
    return;
  }
  
  if (msg?.type === 'PUBLISH.SHOW_SUCCESS_MODAL') {
    const statusBar = document.getElementById('status-bar');
    if (statusBar) {
      statusBar.style.display = 'none';
    }
    
    import('./scripts/panel/success-modal.js').then(({ showSuccessModal }) => {
      showSuccessModal(msg.stats);
      const accountId = msg.accountId || getActiveAccountId();
      if (accountId) {
        
        // Esperar 1 segundo para que el usuario vea el modal
        setTimeout(async () => {
          try {
            const cleanupRes = await RUNS.sw({
              type: 'API.FETCH_JSON',
              url: `/api/walla/publish2/cleanup/${accountId}`,
              method: 'POST'
            });
            
            if (cleanupRes?.ok) {
              
              // Recargar estado de la cuenta
              await loadAccountState(accountId, true);
            } else {
            }
          } catch (cleanupErr) {
            console.error('[PANEL] ❌ Error en cleanup:', cleanupErr);
          }
        }, 1000);
      } else {
      }
    }).catch(err => {
      console.error('[PANEL] Error mostrando modal de éxito:', err);
    });
    
    sendResponse({ ok: true });
    return;
  }
  const isNoisy = msg?.type === 'pong' || 
                  msg?.type === 'SERVER_ALIVE' || 
                  msg?.type === 'HEARTBEAT' ||
                  msg?.type === 'WALLA.SESSION_CHANGED' ||  // Repetitivo
                  msg?.type === 'ACTIVITY.UPDATE';          // Repetitivo
  
  if (!isNoisy) {
  }
  
  // 🟢 WebSocket conectado
  if (msg?.type === 'connected') {
    const dotElement = document.querySelector('.ml-dot');
    if (dotElement) {
      dotElement.classList.remove('warn', 'err');
      dotElement.classList.add('ok');
      
      // Solo actualizar clase del dot, SIN texto
      const pillElement = dotElement.parentElement;
      if (pillElement) {
        // No añadir texto, solo asegurar que dot está presente
        const dot = pillElement.querySelector('.ml-dot');
        if (dot) {
          dot.classList.remove('warn', 'err');
          dot.classList.add('ok');
        }
      }
    }
    sendResponse({ ok: true });
    return;
  }
  
  // 🔴 WebSocket desconectado
  if (msg?.type === 'disconnected') {
    const dotElement = document.querySelector('.ml-dot');
    if (dotElement) {
      dotElement.classList.remove('ok');
      dotElement.classList.add('err');
      
      // Solo actualizar clase del dot, SIN texto
      const pillElement = dotElement.parentElement;
      if (pillElement) {
        const dot = pillElement.querySelector('.ml-dot');
        if (dot) {
          dot.classList.remove('ok', 'warn');
          dot.classList.add('err');
        }
      }
    }
    sendResponse({ ok: true });
    return;
  }
  
  // 🚪 FASE 4: Sesión revocada en otro dispositivo
  if (msg?.type === 'SESSION.REVOKED') {
    
    // Mostrar modal
    alert(msg.message || 'Sesión cerrada. Has iniciado sesión en otro dispositivo.');
    
    // Recargar página (vuelve a login)
    window.location.reload();
    
    sendResponse({ ok: true });
    return;
  }
  
  // 🕐 COUNTDOWN: Mostrar temporizador durante delay de publicación
  if (msg?.type === 'PUBLISH.COUNTDOWN_START') {
    const delaySeconds = msg.delaySeconds || 30;
    const isLongBreak = msg.isLongBreak || false;  // ✅ v10.5.34: Flag para pausa anti-detección
    
    // ✅ v7.2.51: Enviar a página de Wallapop (NO mostrar en panel)
    chrome.tabs.query({ url: '*://*.wallapop.com/*' }, (tabs) => {
      if (tabs && tabs.length > 0) {
        chrome.tabs.sendMessage(tabs[0].id, {
          type: 'SHOW_COUNTDOWN_OVERLAY',
          delaySeconds,
          isLongBreak  // ✅ v10.5.34: Pasar flag al content_script
        }).catch(err => {
        });
      }
    });
    
    sendResponse({ ok: true });
    return;
  }
  
  // ✅ v2.6.0: DELAY STARTED - Notificación en tiempo real via WebSocket
  if (msg?.type === 'publish_delay_started') {
    const { account_id, data } = msg;
    
    // Solo actualizar si es la cuenta activa
    if (account_id === AUTH.getActiveAccountId()) {
      ButtonManager.updateState({
        publishInDelay: true,
        delayUntil: data.delay_until,
        delaySeconds: data.delay_seconds
      });
    }
    
    sendResponse({ ok: true });
    return;
  }
  
  // ✅ v2.6.0: DELAY ENDED - Notificación en tiempo real via WebSocket
  if (msg?.type === 'publish_delay_ended') {
    const { account_id, data } = msg;
    
    // Solo actualizar si es la cuenta activa
    if (account_id === AUTH.getActiveAccountId()) {
      ButtonManager.updateState({
        publishInDelay: false,
        delayUntil: null,
        delaySeconds: null
      });
      
      // Re-consultar estado de publicación para actualizar botones
      if (window.promptPublishResumeIfPending) {
        setTimeout(() => {
          window.promptPublishResumeIfPending();
        }, 100);
      }
    }
    
    sendResponse({ ok: true });
    return;
  }
  
  // 📊 Progreso de backup (CONTRATO ÚNICO - sin prefijo WS.)
  if (msg?.type === 'backup_progress') {
    const { account_id, data } = msg;
    
    const seen = data?.items_processed || 0;
    const total = data?.items_total || 0;
    const errors = data?.errors || 0;
    
    STATE.setBackupState({
      status: 'running',
      seen,
      total,
      errors,
      run_id: data?.run_id || null
    });
    
    // ✅ v6.8.3: Solo actualizar UI si es la cuenta activa
    const currentAccountId = window.AUTH?.getActiveAccountId?.();
    if (String(account_id) === String(currentAccountId)) {
      UI.updateProgressBar({
        operation: 'backup',
        current: seen,
        total: total,
        active: true,
        accountId: account_id
      });
    }
    
    sendResponse({ ok: true });
    return;
  }
  
  // ✅ Backup completado (CONTRATO ÚNICO - sin prefijo WS.)
  if (msg?.type === 'backup_complete') {
    const { account_id, data } = msg;
    
    STATE.setBackupState({
      status: data?.status || 'ok',
      seen: data?.items_processed || 0,
      total: data?.items_total || 0,
      errors: data?.errors || 0,
      run_id: data?.run_id || null
    });
    
    // ✅ v10.5.82: Actualizar fecha/hora de sync INMEDIATAMENTE
    if (data?.status === 'ok' && data?.finished_at) {
      const lastBackupDate = document.getElementById('last-backup-date');
      if (lastBackupDate) {
        const timestamp = data.finished_at; // Unix timestamp en segundos
        const date = new Date(timestamp * 1000);
        const day = date.getDate();
        const month = date.getMonth() + 1;
        const hours = date.getHours();
        const minutes = String(date.getMinutes()).padStart(2, '0');
        const formatted = `${day}/${month} ${hours}:${minutes}`;
        lastBackupDate.textContent = formatted;
        lastBackupDate.title = `Última sincronización: ${date.toLocaleString('es-ES')}`;
      }
    }
    
    // ✅ v10.5.89: Mostrar progreso al 100% ANTES de ocultar (para feedback visual)
    const totalItems = data?.items_total || data?.total || 0;
    if (totalItems > 0) {
      UI.updateProgressBar({
        operation: 'backup',
        current: totalItems,
        total: totalItems,
        active: false,  // Ya no activo
        accountId: account_id
      });
    }
    
    // ✅ v10.5.89: Esperar 1.5s para que el usuario vea el 100% antes de ocultar
    setTimeout(() => {
      UI.updateProgressBar({ operation: null });
    }, 1500);
    
    if (data?.status === 'ok') {
      const skippedCars = data.skipped_cars || 0;
      const skippedRealEstate = data.skipped_real_estate || 0;
      
      // ✅ v6.4.1: Recargar tabla de anuncios después de backup exitoso
      if (typeof LISTINGS !== 'undefined' && LISTINGS.loadListingsTable) {
        const currentAccountId = AUTH.getActiveAccountId();
        if (currentAccountId) {
          LISTINGS.loadListingsTable(currentAccountId);
        }
      }
      
      if (skippedCars > 0 || skippedRealEstate > 0) {
        // Modal con detalles
        setTimeout(() => {
          showModal({
            title: '✅ Backup completado',
            html: `
              <p style="margin-bottom: 16px;">El backup se completó correctamente.</p>
              ${skippedCars > 0 ? `<p class="text-info">🚗 <strong>${skippedCars}</strong> anuncio${skippedCars > 1 ? 's' : ''} de vehículos ${skippedCars > 1 ? 'fueron' : 'fue'} ignorado${skippedCars > 1 ? 's' : ''}</p>` : ''}
              ${skippedRealEstate > 0 ? `<p class="text-info">🏠 <strong>${skippedRealEstate}</strong> anuncio${skippedRealEstate > 1 ? 's' : ''} de inmuebles ${skippedRealEstate > 1 ? 'fueron' : 'fue'} ignorado${skippedRealEstate > 1 ? 's' : ''}</p>` : ''}
              <p style="margin-top: 16px; font-size: 0.9em; color: rgba(255,255,255,0.6);">Los vehículos e inmuebles no se descargan para ahorrar espacio.</p>
            `,
            buttons: [{ text: 'Entendido', value: 'ok', primary: true }]
          });
        }, 500);
      } else {
        toast('✅ Backup completado correctamente');
      }
    } else if (data.status === 'error') {
      toast('❌ Backup finalizado con errores', 'error');
    } else if (data.status === 'cancelled') {
      toast('⚠️ Backup cancelado', 'warn');
    }
    
    sendResponse({ ok: true });
    return;
  }
  
  // 📤 Progreso de publicación (CONTRATO ÚNICO)
  if (msg?.type === 'publish_progress') {
    const { account_id, data } = msg;
    
    const processed = data.processed || 0;
    const total = data.total || 0;
    const currentItemStatus = data.current_item?.status; // 'processing', 'success', 'error'
    
    // ✅ v1.5.0: processingCount ahora se gestiona SOLO desde setPublishingActive()
    // No actualizar aquí para evitar inconsistencias con errores temporales
    
    // ✅ v6.8.3: Solo actualizar UI si es la cuenta activa
    const currentAccountId = window.AUTH?.getActiveAccountId?.();
    if (String(account_id) === String(currentAccountId)) {
      UI.updateProgressBar({
        operation: 'publish',
        current: processed,
        total: total,
        active: true,
        accountId: account_id
      });
      
      // ✅ v6.12.0: Actualizar tab de progreso
      if (window.ProgressTab?.handleProgressUpdate) {
        window.ProgressTab.handleProgressUpdate(data);
      }
    } else {
    }
    
    sendResponse({ ok: true });
    return;
  }
  
  // ✅ Publicación completada (CONTRATO ÚNICO)
  if (msg?.type === 'publish_complete') {
    const { account_id, data } = msg;
    
    // ✅ v6.5.5: Desactivar aviso de publicación activa
    // ✅ v1.5.0: Esto también resetea processingCount a 0 (única fuente)
    setPublishingActive(false);
    
    // ✅ v6.12.0: Notificar fin al tab de progreso
    if (window.ProgressTab?.finishPublication) {
      window.ProgressTab.finishPublication();
    }
    
    // ✅ Ocultar barra de progreso
    UI.updateProgressBar({ operation: null });
    
    // Extraer datos del evento
    const { published = 0, errors = 0, total = 0 } = data || {};
    
    
    // ❌ v7.2.62: COMENTADO - La modal se muestra desde PUBLISH.SHOW_SUCCESS_MODAL
    // Este código causaba modales duplicadas
    /*
    setTimeout(() => {
      try {
        showCompletionModal({
          published: published,
          errors: errors,
          total: total
        });
      } catch (err) {
        logger.error('[PANEL] Error mostrando modal', err);
        logger.error('[PANEL] ❌ Error mostrando modal:', err);
        
        // Fallback: Mostrar toast si modal falla
        const msg = errors > 0 
          ? `✅ Publicación completada: ${published}/${total} (${errors} errores)`
          : `🎉 ¡${published} anuncios publicados correctamente!`;
        toast(msg, errors > 0 ? 'warning' : 'success', { ms: 5000 });
      }
    }, 500);
    */
    
    if (account_id) {
      setTimeout(() => {
        loadAccountState(account_id, true).catch(err => {
          logger.error('[PANEL] Error recargando estado', err); // ✅ v7.2.56: REGLA #4
        });
      }, 2000); // Esperar 2s para que el backend termine de procesar
    }
    
    sendResponse({ ok: true });
    return;
  }
  
  // ✅ v6.6.0: Actualización de item en monitor de publicación
  if (msg?.type === 'PUBLISH_ITEM_UPDATE') {
    const { id, title, price, image_url, status, error, missingFields } = msg.data || {};
    
    // ✅ v6.12.0: Actualizar tab de progreso
    if (window.ProgressTab?.handleProgressUpdate) {
      window.ProgressTab.handleProgressUpdate({
        item_id: id,
        title: title || '',
        price: price || 0,
        image_url: image_url || '',
        status: status || 'processing',
        error: error || null,
        missingFields: missingFields || []
      });
    }
    
    sendResponse({ ok: true });
    return;
  }
  
  // ✅ WEBSOCKET: Run Viewed (sincronización multi-dispositivo)
  if (msg?.type === 'RUN_VIEWED') {
    const { run_id, viewed_at } = msg.data || {};
    
    // Actualizar estado para incluir viewed_at
    if (STATE.BACKUP_STATE.run_id === run_id) {
      STATE.setBackupState({ viewed_at });
      
      // Si el botón estaba mostrando "Backup ✅", resetear a "Backup"
      const btnBackup = document.getElementById('btn-backup');
      const label = btnBackup?.querySelector('.label');
      if (label?.textContent === 'Backup ✅') {
        label.textContent = 'Backup';
        btnBackup.classList.remove('btn-success');
      }
      
      // Limpiar badge si está mostrando "Backup completo"
      const pill = document.getElementById('pill-backup');
      if (pill?.textContent?.includes('Backup completo')) {
        pill.textContent = '';
        pill.className = 'ml-pill ml-action-pill';
        pill.innerHTML = '';
      }
    }
    
    sendResponse({ ok: true });
    return;
  }
  
  // 🆕 WEBSOCKET: First Backup Complete (configuración inicial usuario nuevo)
  if (msg?.type === 'first_backup_complete') {
    const { account_id, data } = msg;
    
    // Ya NO recargamos el panel - WebSocket es persistente
    // En su lugar, recargamos solo la tabla de anuncios
    if (data?.is_first_setup === true) {
      
      // Mostrar toast de éxito
      toast('✅ Primer backup completado - ¡Tus anuncios están listos!', 'success');
      
      // Recargar tabla de anuncios (sin reload de página)
      if (typeof LISTINGS !== 'undefined' && LISTINGS.loadListingsTable) {
        const currentAccountId = AUTH.getActiveAccountId();
        if (currentAccountId) {
          LISTINGS.loadListingsTable(currentAccountId);
        }
      }
    } else {
    }
    
    sendResponse({ ok: true });
    return;
  }
  
  // ❌ v7.2.62: DESHABILITADO - Causa modales duplicadas
  // Este handler es ANTIGUO y ya NO se usa (reemplazado por PUBLISH.SHOW_SUCCESS_MODAL)
  /*
  if (msg?.type === 'PUBLISH.COMPLETED') {
    
    // Mostrar modal de éxito
    showCompletionModal(msg.stats);
    
    // Refrescar UI de runs
    try {
      if (typeof RUNS !== 'undefined' && RUNS.refreshUI) {
        RUNS.refreshUI();
      }
    } catch (e) {
    }
    
    // ✅ NUEVO: Verificar si quedan pendientes y mostrar botón Reanudar
    setTimeout(async () => {
      try {
        const { promptPublishResumeIfPending } = await import('./scripts/panel/resume.js');
        await promptPublishResumeIfPending();
      } catch (err) {
        console.error('[PANEL] Error verificando pendientes:', err);
      }
    }, 1000);
    
    sendResponse({ ok: true });
  }
  */
  
  if (msg?.type === 'SHOW_WALLAPOP_REQUIRED_MODAL') {
    showWallapopRequiredModal();
    sendResponse({ ok: true });
  }
});

// Función para mostrar modal "Se necesita Wallapop"
function showWallapopRequiredModal() {
  const modal = showModal(`
    <h3 style="margin:0 0 20px 0; font-size: 20px;">Se necesita una pestaña de Wallapop</h3>
    <p style="margin: 0 0 24px 0; color: #666;">
      No estás en Wallapop. ¿Qué hacemos?
    </p>
    <div style="display: flex; gap: 12px; justify-content: center; flex-wrap: wrap;">
      <button id="ml-modal-switch" style="
        background: #13ce66;
        color: white;
        border: none;
        padding: 12px 24px;
        border-radius: 8px;
        cursor: pointer;
        font-size: 15px;
        font-weight: 600;
      ">Cambiar a Wallapop</button>
      <button id="ml-modal-open-here" style="
        background: #2c3e50;
        color: white;
        border: none;
        padding: 12px 24px;
        border-radius: 8px;
        cursor: pointer;
        font-size: 15px;
      ">Abrir aquí</button>
      <button id="ml-modal-open-new" style="
        background: #2c3e50;
        color: white;
        border: none;
        padding: 12px 24px;
        border-radius: 8px;
        cursor: pointer;
        font-size: 15px;
      ">Abrir nueva</button>
      <button id="ml-modal-cancel" style="
        background: #f56565;
        color: white;
        border: none;
        padding: 12px 24px;
        border-radius: 8px;
        cursor: pointer;
        font-size: 15px;
      ">Cancelar</button>
    </div>
  `);
  
  // Event listeners
  document.getElementById('ml-modal-switch')?.addEventListener('click', async () => {
    const tabs = await chrome.tabs.query({ url: ['https://*.wallapop.com/*', 'https://*.wallapop.es/*'] });
    if (tabs[0]) {
      await chrome.tabs.update(tabs[0].id, { active: true });
      await chrome.windows.update(tabs[0].windowId, { focused: true });
    }
    modal.remove();
  });
  
  document.getElementById('ml-modal-open-here')?.addEventListener('click', async () => {
    const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
    if (tabs[0]) {
      await chrome.tabs.update(tabs[0].id, { url: 'https://es.wallapop.com' });
    }
    modal.remove();
  });
  
  document.getElementById('ml-modal-open-new')?.addEventListener('click', async () => {
    await chrome.tabs.create({ url: 'https://es.wallapop.com', active: true });
    modal.remove();
  });
  
  document.getElementById('ml-modal-cancel')?.addEventListener('click', () => {
    modal.remove();
  });
}

// Función para mostrar modal de finalización
function showCompletionModal(stats) {
  
  // ✅ v7.2.59: Backend YA separa published/errors correctamente
  // NO restar errores - son valores independientes
  const total = Number(stats?.total || 0);
  const published = Number(stats?.published || 0);
  const errors = Number(stats?.errors || 0);
  
  
  // Validación: Si no hay nada que mostrar, no abrir modal
  if (total === 0 && published === 0 && errors === 0) {
    return;
  }
  
  
  showSuccessModal({
    total,
    published,
    errors
  });
}


/* ============================================
   ✅ ACTUALIZAR CONTADOR Y VISIBILIDAD DE BOTONES
   ============================================ */
// Ahora todo se gestiona desde ButtonManager centralizado

/* ============================================
   ✅ AUTO-INICIALIZACIÓN DE EVENTOS
// ============================================
// DEBUG MODE TOGGLE
// ============================================
// 🐛 DEBUG MODE - Banner visual
// ============================================

/**
 * Mostrar/ocultar banner debug en header
 * Se ejecuta al cargar la página y al cambiar el estado
 */
async function updateDebugBanner() {
  try {
    const result = await chrome.storage.local.get(['debugMode']);
    const debugActive = result.debugMode || false;
    
    
    const banner = document.getElementById('debug-warning-banner');
    if (banner) {
      const previousDisplay = banner.style.display;
      banner.style.display = debugActive ? 'inline-flex' : 'none';
      banner.style.visibility = debugActive ? 'visible' : 'hidden';
      
      
      // Verificación adicional
      if (debugActive && banner.style.display !== 'inline-flex') {
        console.error('[DEBUG-BANNER] ⚠️ PROBLEMA: Banner debería estar visible pero display es:', banner.style.display);
      }
    } else {
      console.error('[DEBUG-BANNER] ❌ Elemento #debug-warning-banner NO encontrado en DOM');
    }
  } catch (error) {
    console.error('[DEBUG-BANNER] ❌ Error:', error);
  }
}

/**
 * ✅ v4.84.4: Cargar versión de la extensión desde manifest.json
 * y mostrarla en el footer
 */
async function loadExtensionVersion() {
  const versionEl = document.getElementById('extension-version');
  if (!versionEl) return;
  
  try {
    const manifest = chrome.runtime.getManifest();
    const version = manifest.version;
    versionEl.textContent = `v${version}`;
    versionEl.title = `Versión de la extensión: ${version}`;
  } catch (err) {
    versionEl.textContent = 'v—';
  }
}

async function initDebugToggle() {
  const toggle = document.getElementById("debug-mode-toggle");
  if (!toggle) return;
  
  // Cargar estado actual
  const result = await chrome.storage.local.get(["debugMode"]);
  toggle.checked = result.debugMode || false;
  
  // Actualizar banner (usa la función centralizada)
  await updateDebugBanner();
  
  // Listener de cambios
  toggle.addEventListener("change", async (e) => {
    const enabled = e.target.checked;
    
    // ✅ Sin confirmación - activar directamente
    
    try {
      // Guardar en chrome.storage
      await chrome.storage.local.set({ debugMode: enabled });
      
      // Opcional: Guardar en BD también
      const apiBase = await window.COMMON?.getApiBase?.();
      const token = await window.AUTH?.getToken?.();
      
      if (apiBase && token) {
        await fetch(`${apiBase}/api/user/preferences`, {
          method: "PATCH",
          headers: {
            "Authorization": `Bearer ${token}`,
            "Content-Type": "application/json"
          },
          body: JSON.stringify({ debug_mode: enabled })
        }).catch(() => {}); // Ignorar errores
      }
      
      // Mostrar/ocultar warning visual
      
      // LOG DIRECTO para verificar que funciona
      
      // Actualizar banner usando función centralizada
      await updateDebugBanner();
      
      
      toast(enabled ? "🐛 Modo debug activado" : "✅ Modo debug desactivado", enabled ? "info" : "success");
      
      if (enabled) {
      } else {
      }
    } catch (err) {
      console.error("Error al cambiar debug mode:", err);
      toast("Error al cambiar modo debug", "error");
      e.target.checked = !enabled;  // Revertir toggle
    }
  });
}

// ============================================================
// ✅ v6.5.5: TOAST DEL PANEL (Notificaciones en el header)
// ============================================================

/**
 * Muestra un toast en el panel lateral (debajo del header)
 * @param {string} message - Mensaje a mostrar
 * @param {string} icon - Emoji o icono
 * @param {string} type - Tipo: 'warning' | 'error' | 'success' | 'info'
 * @param {number|null} autoHideMs - Auto-ocultar después de X ms (null = no auto-ocultar)
 */
function showPanelToast(message, icon = '⚠️', type = 'warning', autoHideMs = null) {
  const toastContainer = document.getElementById('ml-panel-toasts');
  if (!toastContainer) {
    return;
  }
  
  const iconEl = toastContainer.querySelector('.toast-icon');
  const messageEl = toastContainer.querySelector('.toast-message');
  const closeBtn = toastContainer.querySelector('.toast-close');
  
  if (iconEl) iconEl.textContent = icon;
  if (messageEl) messageEl.textContent = message;
  
  // Limpiar clases previas
  toastContainer.classList.remove('toast-warning', 'toast-error', 'toast-success', 'toast-info');
  toastContainer.classList.add(`toast-${type}`);
  
  toastContainer.style.display = 'block';
  
  // ✅ v7.2.57: REGLA #3 - Countdown con auto-hide
  if (autoHideMs) {
    const seconds = Math.ceil(autoHideMs / 1000);
    let remaining = seconds;
    
    // Crear/obtener elemento de countdown
    let countdownEl = toastContainer.querySelector('.toast-countdown');
    if (!countdownEl) {
      countdownEl = document.createElement('span');
      countdownEl.className = 'toast-countdown';
      // ✅ v7.2.57: Insertar ANTES del botón cerrar (inline)
      const content = toastContainer.querySelector('.ml-panel-toast-content');
      if (content && closeBtn) {
        content.insertBefore(countdownEl, closeBtn);
      }
    }
    
    countdownEl.textContent = `(${remaining}s)`;
    countdownEl.style.display = 'inline';
    
    // Actualizar cada segundo
    const interval = setInterval(() => {
      remaining--;
      if (remaining > 0) {
        countdownEl.textContent = `(${remaining}s)`;
      } else {
        clearInterval(interval);
        countdownEl.style.display = 'none';
      }
    }, 1000);
    
    setTimeout(() => {
      clearInterval(interval);
      hidePanelToast();
    }, autoHideMs);
  } else {
    // ✅ v7.2.57: Ocultar countdown si no hay autoHideMs
    const countdownEl = toastContainer.querySelector('.toast-countdown');
    if (countdownEl) {
      countdownEl.style.display = 'none';
    }
  }
  
  // Cerrar al hacer click en X
  if (closeBtn) {
    closeBtn.onclick = () => hidePanelToast();
  }
  
}

/**
 * Oculta el toast del panel
 */
function hidePanelToast() {
  const toastContainer = document.getElementById('ml-panel-toasts');
  if (toastContainer) {
    toastContainer.style.display = 'none';
  }
}

// ============================================================
// ✅ v6.5.5: DETECCIÓN DE CAMBIO DE PESTAÑA
// ============================================================

let _publishingActive = false;
let _tabChangeListenerInstalled = false;
let _tabVisibilityIgnoreUntil = 0; // ✅ v6.6.0: Timestamp para ignorar mensajes iniciales

/**
 * Marca que la publicación está activa (para mostrar aviso si cambia de pestaña)
 */
function setPublishingActive(active) {
  _publishingActive = active;
  
  // ✅ v1.5.0: ÚNICA FUENTE DE VERDAD para processingCount
  // Controla visibilidad de botones Reanudar/Cancelar en ButtonManager
  ButtonManager.updateState({ processingCount: active ? 1 : 0 });
  
  // ✅ v6.6.0: Ignorar cambios de visibilidad durante los primeros 3 segundos
  if (active) {
    _tabVisibilityIgnoreUntil = Date.now() + 3000;
  }
  
  // ✅ v10.5.44: Notificar al content_script para activar protección beforeunload
  chrome.runtime.sendMessage({
    type: 'PUBLISH.BROADCAST_TO_WALLAPOP',
    payload: {
      type: 'PUBLISH.SET_ACTIVE',
      active: active
    }
  }).catch(() => {});
  
  if (active) {
    // ✅ v7.2.57: Toast naranja con countdown de 10s
    showPanelToast(
      '⚠️ Publicación en curso - No cambies de pestaña', 
      '🚀', 
      'warning', 
      10000 // ✅ Auto-close en 10 segundos
    );
    _installTabChangeListener();
  } else {
    hidePanelToast();
  }
}

/**
 * Instala listener para detectar cuando el usuario cambia de pestaña
 * ✅ v10.5.49: Eliminados toasts molestos - el usuario ve el progreso en pestaña "En Curso"
 */
function _installTabChangeListener() {
  if (_tabChangeListenerInstalled) return;
  
  // Escuchar mensajes del SW sobre cambio de pestaña activa
  chrome.runtime.onMessage.addListener((msg) => {
    if (msg.type === 'TAB_VISIBILITY_CHANGED' && _publishingActive) {
      // ✅ v6.6.0: Ignorar mensajes durante los primeros 3 segundos después de activar publicación
      if (Date.now() < _tabVisibilityIgnoreUntil) {
        return;
      }
      
      // ✅ v10.5.49: Toasts eliminados - el progreso es visible en pestaña "En Curso"
      // El usuario no necesita avisos constantes
    }
  });
  
  _tabChangeListenerInstalled = true;
}

// Exponer funciones globalmente
window.showPanelToast = showPanelToast;
window.hidePanelToast = hidePanelToast;
window.setPublishingActive = setPublishingActive;

// ✅ v6.6.0: Exponer función para parar publicación (usado por monitor)
window.stopPublishing = function() {
  stopContinuousProcessing();
  setPublishingActive(false);
  toast('⏹️ Publicación pausada', 'warning');
};

// ✅ v6.6.0: Escuchar evento personalizado del monitor
window.addEventListener('pm-stop-publishing', () => {
  window.stopPublishing();
});

// Inicializar debug toggle cuando cargue Config tab
document.addEventListener("DOMContentLoaded", () => {
  // Inicializar toggle de debug (si existe en Config tab)
  initDebugToggle();
  
  // IMPORTANTE: Actualizar banner debug SIEMPRE al cargar
  updateDebugBanner();
  loadExtensionVersion();
  
  // ✅ v7.0.1: Listener para cambio de cuenta (sincronizar ProgressTab)
  const selAccount = document.getElementById('sel-account');
  if (selAccount) {
    selAccount.addEventListener('change', async (e) => {
      const accountId = parseInt(e.target.value);
      
      // Sincronizar ProgressTab
      if (window.ProgressTab?.switchAccount && accountId > 0) {
        await window.ProgressTab.switchAccount(accountId);
      }
    });
  }
  
});

// Ahora se gestionan en listings/events.js (módulo centralizado)
   
// Exponer módulos globalmente (para compatibilidad)
window.AUTH = AUTH;
window.LISTINGS = LISTINGS;
window.RESUME = RESUME;
window.ProgressTab = ProgressTab;  // ✅ v6.12.0: Exponer ProgressTab
window.ONBOARDING = ONBOARDING;  // ✅ v10.5.146: Exponer Onboarding para testing
window.toast = toast;
window.updateDebugBanner = updateDebugBanner; // Para testing


// ============================================================
// BOTONES DE PUBLICACIÓN (flujo completo)
// ============================================================

/**
 * Helper: Verificar que hay pestaña de Wallapop en la URL correcta
 */

/**
 * Loop de procesamiento (llama a RUN.RESUME repetidamente)
 */

// ============================================================
// BOTÓN PUBLICAR ÚNICO (Publicar Todos o Seleccionados)
// ============================================================
// ============================================================
// BOTÓN ELIMINAR SELECCIONADOS
// ============================================================
document.getElementById('btn-delete-selected')?.addEventListener('click', async () => {
  
  try {
    const { handleDeleteSelected } = await import('./scripts/panel/delete-handler.js');
    await handleDeleteSelected();
  } catch (error) {
    console.error('[PANEL] ❌ Error importando delete-handler:', error);
    console.error('[PANEL] Stack:', error.stack);
    console.error('[PANEL] Message:', error.message);
    logger.error('[PANEL] ❌ Error importando delete-handler:', error.message);
    toast('Error al eliminar anuncios', 'err');
  }
});

// ============================================================
// ============================================================
document.getElementById('btn-publish-dynamic')?.addEventListener('click', async () => {
  
  const accountId = AUTH.getActiveAccountId();
  if (!accountId) return toast('Selecciona una cuenta');
  
  // Obtener acción del botón (data-action: 'publish' o 'republish')
  const btn = document.getElementById('btn-publish-dynamic');
  const action = btn?.getAttribute('data-action') || 'publish';
  const isDeletedMode = action === 'republish';
  
  // Obtener IDs seleccionados
  const selectedIds = Array.from(LISTINGS.getSelectedListingIds());
  
  if (selectedIds.length === 0) {
    return toast('Selecciona al menos un anuncio', 'warn');
  }
  
  // ✅ Usar flujo unificado
  await handlePublishFlow({
    accountId,
    selectedIds,
    isDeletedMode
  });
});

// ============================================================
// ============================================================
document.getElementById('btn-resume-pending')?.addEventListener('click', async () => {
  
  const accountId = AUTH.getActiveAccountId();
  if (!accountId) return toast('Selecciona una cuenta');
  
  try {
    // Obtener estado actual de publicaciones
    showLoader('Verificando procesos pendientes...', {
      timeout: 10000,
      onTimeout: () => toast('⏱️ Tiempo agotado verificando procesos', 'error', 5000)
    });
    
    const statusRes = await chrome.runtime.sendMessage({
      type: 'API.FETCH_JSON',
      url: `/api/walla/publish2/status?account_id=${accountId}`,
      method: 'GET'
    });
    
    hideLoader();
    
    if (!statusRes?.ok || !statusRes.data) {
      return toast('❌ Error obteniendo estado de publicaciones', 'err');
    }
    
    const { pending, processing } = statusRes.data;
    const totalPending = (pending || 0) + (processing || 0);
    
    if (totalPending === 0) {
      return toast('No hay procesos pendientes para reanudar', 'info');
    }
    
    // Confirmar reanudación
    const result = await showModal({
      title: '⏭️ Reanudar publicaciones',
      html: `
        <p style="margin-bottom: 16px;">
          Hay <strong>${totalPending} anuncio${totalPending !== 1 ? 's' : ''}</strong> pendiente${totalPending !== 1 ? 's' : ''} de publicar.
        </p>
        <p style="color: #3b82f6; font-size: 14px; margin: 0;">
          ¿Deseas reanudar el proceso de publicación?
        </p>
      `,
      buttons: [
        { text: 'Cancelar', value: 'cancel', primary: false },
        { text: 'Reanudar', value: 'confirm', primary: true }
      ]
    });
    
    if (result !== 'confirm') return;
    
    // ✅ Usar flujo unificado en modo reanudar
    await handlePublishFlow({
      accountId,
      selectedIds: [], // Vacío = procesar todos los pendientes
      isDeletedMode: false,
      isResumeMode: true,
      resumePendingCount: totalPending,
      skipConfirmation: true // Ya confirmamos arriba
    });
    
  } catch (e) {
    hideLoader();
    logger.error('[PANEL] Error reanudando publicaciones:', e);
    toast(`❌ Error: ${e.message}`, 'err');
  }
});

// ✅ v1.3.0: Botón Reanudar en tab "En Curso" - dispara el mismo flujo (REGLA #41)
document.getElementById('btn-resume-progress')?.addEventListener('click', () => {
  // Reutilizar el botón original para mantener flujo centralizado
  document.getElementById('btn-resume-pending')?.click();
});

// ============================================================
// BOTÓN PUBLICAR DELETED SELECCIONADOS
// ============================================================
// ============================================================
// BOTÓN CANCELAR PROCESOS PENDIENTES
// ============================================================
document.getElementById('btn-cancel-pending')?.addEventListener('click', async () => {
  const accountId = AUTH.getActiveAccountId();
  if (!accountId) return toast('Selecciona una cuenta');
  
  // ✅ Consultar anuncios pendientes para detectar los eliminados
  let deletedPendingItems = [];
  try {
    const pendingRes = await chrome.runtime.sendMessage({
      type: 'API.FETCH_JSON',
      url: `/api/walla/publish2/pending-by-account/${accountId}`,
      method: 'GET'
    });
    
    if (pendingRes?.ok && pendingRes.data?.items) {
      // Filtrar los que tienen error "deleted_pending_publish"
      deletedPendingItems = pendingRes.data.items.filter(item => {
        const errorStr = item.publish_error || '';
        return errorStr.includes('deleted_pending_publish') || 
               errorStr.includes('Anuncio eliminado') ||
               errorStr.includes('pendiente republicar');
      });
    }
  } catch (e) {
  }
  
  // Construir HTML de la modal
  let modalHtml = `
    <p style="margin-bottom: 16px;">¿Estás seguro de cancelar todos los procesos de publicación?</p>
  `;
  
  // ⚠️ Si hay anuncios eliminados pendientes, mostrar advertencia grave
  if (deletedPendingItems.length > 0) {
    modalHtml += `
      <div class="deleted-items-alert">
        <p class="deleted-items-alert__title">
          🚨 ¡ATENCIÓN! ${deletedPendingItems.length} anuncio(s) YA ELIMINADO(S) de Wallapop:
        </p>
        <div class="deleted-items-alert__list">
    `;
    
    deletedPendingItems.forEach(item => {
      // Extraer datos del item
      let title = 'Sin título';
      let imageUrl = null;
      let wallapopId = item.id_wallapop || 'N/A';
      let price = null;
      
      // Intentar parsear JSON para obtener título e imagen
      try {
        if (item.json_data) {
          const snapshot = JSON.parse(item.json_data);
          title = snapshot?.title?.original || snapshot?.title || item.title || 'Sin título';
          // Extraer precio (varias posibles estructuras)
          price = snapshot?.price?.cash?.amount 
               || snapshot?.price?.amount 
               || snapshot?.sale_price 
               || snapshot?.price
               || null;
          // Asegurar que es número
          if (price && typeof price === 'object') {
            price = price.amount || price.cash?.amount || null;
          }
          // Construir URL de imagen local
          const img = snapshot?.images?.[0];
          if (img) {
            const localUrl = img.local_url || img.file_url;
            if (localUrl) {
              const base = (window.CONFIG?.API_BASE || '').replace(/\/$/, '');
              imageUrl = base + (localUrl.startsWith('/') ? localUrl : '/' + localUrl);
            } else {
              imageUrl = img.urls?.small || img.url;
            }
          }
        }
      } catch (e) {
        title = item.title || 'Sin título';
      }
      
      // Formatear precio
      let priceDisplay = '';
      if (price && !isNaN(Number(price))) {
        priceDisplay = `${Number(price).toFixed(2).replace('.', ',')} €`;
      }
      
      modalHtml += `
        <div class="deleted-items-alert__item">
          ${imageUrl 
            ? `<img src="${imageUrl}" class="deleted-items-alert__thumbnail" alt="Miniatura" />`
            : `<div class="deleted-items-alert__thumbnail-placeholder"><span>📦</span></div>`
          }
          <div class="deleted-items-alert__info">
            <div class="deleted-items-alert__header">
              <p class="deleted-items-alert__item-title">${title}</p>
              ${priceDisplay ? `<span class="deleted-items-alert__item-price">${priceDisplay}</span>` : ''}
            </div>
            <p class="deleted-items-alert__item-id">ID: ${wallapopId}</p>
          </div>
        </div>
      `;
    });
    
    modalHtml += `
        </div>
        <p class="deleted-items-alert__warning">
          ⚠️ Estos anuncios <strong>YA NO EXISTEN en Wallapop</strong> pero aún no se han republicado.<br>
          Si cancelas, <strong>SE PERDERÁN DEFINITIVAMENTE</strong>.
        </p>
      </div>
    `;
  } else {
    // Advertencia normal (sin anuncios eliminados)
    modalHtml += `
      <p style="color: #fbbf24; font-size: 14px; margin: 0;">
        <strong>⚠️ Atención:</strong> Los anuncios en proceso podrían perderse. Tendrás que volver a publicarlos.
      </p>
    `;
  }
  
  // Confirmación
  const result = await showModal({
    title: deletedPendingItems.length > 0 ? '🚨 Cancelar procesos pendientes' : '⚠️ Cancelar procesos pendientes',
    html: modalHtml,
    buttons: [
      { text: 'No', value: 'cancel', primary: false },
      { text: deletedPendingItems.length > 0 ? '⚠️ Sí, cancelar todo' : 'Sí, cancelar todo', value: 'confirm', primary: true }
    ]
  });
  
  if (result !== 'confirm') return;
  
  try {
    showLoader('Cancelando procesos...', {
      timeout: 30000,
      onTimeout: () => toast('⏱️ Tiempo agotado cancelando procesos', 'error', 5000)
    });
    
    const cancelRes = await chrome.runtime.sendMessage({
      type: 'API.FETCH_JSON',
      url: `/api/walla/publish2/cancel/${accountId}`,
      method: 'POST'
    });
    
    hideLoader();
    
    if (cancelRes?.ok && cancelRes.data) {
      const { cancelled, details } = cancelRes.data;
      
      // ✅ v2.6.0: Notificar fin de delay por cancelación
      try {
        await chrome.runtime.sendMessage({
          type: 'API.FETCH_JSON',
          url: `/api/walla/publish2/session/end-delay`,
          method: 'POST',
          body: { account_id: accountId, reason: 'cancelled' }
        });
      } catch (endErr) {
        console.error('[Panel] ⚠️ Error notificando cancelación:', endErr);
      }
      
      toast(`✅ ${cancelled} procesos cancelados`);
      
      // ✅ v10.5.144: Usar función centralizada (REGLA #11)
      if (window.ProgressTab?.resetPublishState) {
        window.ProgressTab.resetPublishState();
      }
      
      // Recargar estado del panel
      await loadAccountState(accountId);
      
      // Recargar tabla de listings
      if (LISTINGS?.loadListings) {
        await LISTINGS.loadListings(accountId);
      }
    } else {
      toast(`❌ Error: ${cancelRes?.error || 'No se pudo cancelar'}`);
    }
    
  } catch (e) {
    hideLoader();
    toast(`❌ Error: ${e.message}`);
  }
});

// ============================================================
// PUBLICAR SELECCIONADOS
// ============================================================
// ============================================================
// LOGOUT CON AUTO-CANCELACIÓN
// ============================================================


// ============================================
// LOGOUT - Limpia estados locales
// ============================================
async function handleLogout() {
  const result = await showModal({
    title: '👋 Cerrar sesión',
    html: `
      <p>¿Estás seguro de que deseas cerrar sesión?</p>
    `,
    buttons: [
      { text: 'Cancelar', value: 'cancel', primary: false },
      { text: 'Cerrar Sesión', value: 'confirm', primary: true }
    ]
  });
  
  if (result !== 'confirm') return;
  
  showLoader('Cerrando sesión...', {
    timeout: 30000,
    onTimeout: () => toast('⏱️ Tiempo agotado cerrando sesión', 'error', 5000)
  });
  
  try {
    // 1. Detener publicación en curso si existe (sin resetear estados)
    stopContinuousProcessing();
    
    // 2. Limpiar estados locales UI (solo frontend)
    STATE.setBackupState({ 
      status: 'idle', 
      pending: 0, 
      errors: 0, 
      seen: 0, 
      total: 0, 
      run_id: null 
    });
    ButtonManager.updateState({
      hasPendingProcess: false,
      pendingProcessId: null,
      pendingType: null
    });
    
    
    // 3. Logout en backend (revoca dispositivo automáticamente)
    try {
      await RUNS.sw({ type: 'SESSION.LOGOUT' });
    } catch (e) {
    }
    
    // 5. Limpiar storage local (incluir selectedPlatform para volver al selector)
    await stRemove(['jwt', 'user', 'active_account', 'selectedPlatform']);
    
    // 6. Actualizar UI
    STATE.set_AUTH({ state: 'UNAUTHENTICATED', on: false, user: null });
    UI.setAuthUI(false);
    
    hideLoader();
    toast('👋 Sesión cerrada correctamente');
    
    // 7. Recargar panel para mostrar selector de plataforma
    setTimeout(() => {
      window.location.href = 'panel.html';
    }, 500);
    
  } catch (e) {
    hideLoader();
    console.error('[LOGOUT] Error:', e);
    toast('❌ Error al cerrar sesión: ' + e.message, 'err');
  }
}

// Exportar handleLogout globalmente para que menu.js pueda usarla
window.handleLogout = handleLogout;

// Conectar solo btn-logout-tab (candado en tabs)
// mi-logout ya tiene listener en menu.js
document.getElementById('btn-logout-tab')?.addEventListener('click', handleLogout);

// ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
// 🆕 FUNCIÓN YA NO USADA - Se reemplazó por recarga automática simple
// ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
// Esta función intentaba actualizar el panel sin recargar, pero causaba problemas
// con la renderización de la tabla de anuncios. La solución final es simplemente
// recargar la página después del primer backup (solo pasa UNA vez por usuario).
//
// La función updatePanelAfterFirstBackup() fue eliminada completamente.
// Ahora solo se hace: window.location.reload() después de 1.5 segundos.


// ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
// 🆕 MILANUNCIOS - ROUTER DE PLATAFORMAS
// ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
// Sistema que permite seleccionar plataforma (Wallapop/Milanuncios) en login
// y carga el panel correspondiente. 100% AISLADO - no afecta código Wallapop.
// ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

(async function() {
  console.log('[Platform Router] Inicializando...');
  
  // 1. Verificar si venimos de un cambio de plataforma intencional
  const urlParams = new URLSearchParams(window.location.search);
  const fromPlatformChange = urlParams.get('fromChange') === 'true';
  
  if (fromPlatformChange) {
    console.log('[Platform Router] Cambio de plataforma detectado, limpiando selección...');
    await chrome.storage.local.remove(['selectedPlatform']);
    // NO hacer return - continuar para mostrar selector
  }
  
  // 2. Esperar a que el DOM esté listo
  if (document.readyState === 'loading') {
    await new Promise(resolve => {
      document.addEventListener('DOMContentLoaded', resolve);
    });
  }
  
  // 3. Al cargar, verificar si ya hay sesión de Milanuncios (solo si NO viene de cambio)
  const stored = await chrome.storage.local.get(['jwt', 'selectedPlatform']);
  
  // SOLO redirigir si:
  // - NO venimos de cambio de plataforma
  // - Hay JWT (está logueado)
  // - Y plataforma es Milanuncios
  // - Y el login ya está visible (para evitar flash)
  if (!fromPlatformChange && stored.jwt && stored.selectedPlatform === 'milanuncios') {
    const loginBlock = document.getElementById('block-login');
    if (loginBlock && loginBlock.style.display !== 'none') {
      console.log('[Platform Router] Sesión Milanuncios detectada, redirigiendo...');
      window.location.href = 'milanuncios/panel-milanuncios.html';
      return;
    }
  }
  
  // 4. Guardar plataforma seleccionada cuando cambia
  const platformRadios = document.querySelectorAll('input[name="platform"]');
  
  if (platformRadios.length > 0) {
    // Listener para cambios
    platformRadios.forEach(radio => {
      radio.addEventListener('change', async (e) => {
        if (e.target.checked) {
          const selectedPlatform = e.target.value;
          console.log('[Platform Router] Plataforma seleccionada:', selectedPlatform);
          await chrome.storage.local.set({ selectedPlatform });
        }
      });
    });
    
    // Establecer valor por defecto si no existe
    if (!stored.selectedPlatform) {
      const wallapopRadio = document.querySelector('input[name="platform"][value="wallapop"]');
      if (wallapopRadio) {
        wallapopRadio.checked = true;
        await chrome.storage.local.set({ selectedPlatform: 'wallapop' });
      }
    } else {
      // Marcar el radio correcto
      const selectedRadio = document.querySelector(`input[name="platform"][value="${stored.selectedPlatform}"]`);
      if (selectedRadio) {
        selectedRadio.checked = true;
      }
    }
  }
  
  // 5. Observar cambios en el DOM para detectar login exitoso
  const observer = new MutationObserver(async (mutations) => {
    const loginBlock = document.getElementById('block-login');
    const appBlock = document.getElementById('block-app');
    
    // Detectar cuando login desaparece y app aparece = login exitoso
    if (loginBlock && loginBlock.style.display === 'none' && 
        appBlock && appBlock.style.display !== 'none') {
      
      const { selectedPlatform } = await chrome.storage.local.get(['selectedPlatform']);
      
      console.log('[Platform Router] Login exitoso detectado, plataforma:', selectedPlatform);
      
      if (selectedPlatform === 'milanuncios') {
        console.log('[Platform Router] Redirigiendo a Milanuncios...');
        observer.disconnect();
        
        // Pequeño delay para que termine el flujo de login
        setTimeout(() => {
          window.location.href = 'milanuncios/panel-milanuncios.html';
        }, 300);
      }
      // Si es Wallapop, no hacer nada (flujo normal)
    }
  });
  
  // Observar cambios en todo el body
  observer.observe(document.body, { 
    childList: true, 
    subtree: true,
    attributes: true,
    attributeFilter: ['style', 'class']
  });
  
  console.log('[Platform Router] ✅ Inicializado correctamente');
})();

