/**
 * MitikLive Milanuncios - UI Helpers v5.8.3
 */

import { state } from './state.js';
import { createLogger } from '../../scripts/utils.js';

const logger = createLogger('Milanuncios', 'UI');

/**
 * Muestra mensaje de estado
 */
export function showStatus(title, type = 'info', message = '') {
  const statusEl = document.getElementById('mil-status');
  const adsContainer = document.getElementById('mil-ads-container');
  
  if (statusEl) statusEl.style.display = 'block';
  if (adsContainer) adsContainer.style.display = 'none';
  
  if (statusEl) {
    statusEl.className = `ml-alert ${type}`;
    
    const icons = {
      info: 'ℹ️',
      warning: '⚠️',
      error: '❌',
      success: '✅'
    };
    
    statusEl.innerHTML = `
      <span class="icon">${icons[type] || 'ℹ️'}</span>
      <div class="message">
        <strong>${title}</strong>
        ${message ? `<p>${message}</p>` : ''}
      </div>
    `;
  }
}

/**
 * Oculta mensaje de estado
 */
export function hideStatus() {
  const statusEl = document.getElementById('mil-status');
  const adsContainer = document.getElementById('mil-ads-container');
  const toolbar = document.getElementById('mil-toolbar-container');
  
  if (statusEl) statusEl.style.display = 'none';
  if (adsContainer) adsContainer.style.display = 'block';
  if (toolbar) toolbar.style.display = 'block';
}

/**
 * Cambia de tab y guarda preferencia
 */
export async function switchTab(tabName) {
  // Ocultar todos los contenidos
  document.querySelectorAll('.tab-content').forEach(tab => {
    tab.style.display = 'none';
  });
  
  // Desactivar todos los botones
  document.querySelectorAll('.tab-button').forEach(btn => {
    btn.classList.remove('active');
  });
  
  // Mostrar tab seleccionada
  const tab = document.getElementById(`tab-${tabName}`);
  if (tab) tab.style.display = 'block';
  
  // Activar botón
  const btn = document.querySelector(`[data-tab="${tabName}"]`);
  if (btn) btn.classList.add('active');
  
  // Controlar toolbar
  const toolbar = document.getElementById('mil-toolbar-container');
  const adsContainer = document.getElementById('mil-ads-container');
  if (toolbar) {
    const showToolbar = tabName === 'renovaciones' && 
                        adsContainer && 
                        adsContainer.style.display !== 'none';
    toolbar.style.display = showToolbar ? 'block' : 'none';
  }
  
  // ✅ v5.8.3: Guardar tab activa en backend (import dinámico para evitar circular)
  try {
    const { saveActiveTab } = await import('./config.js');
    saveActiveTab(tabName);
  } catch (e) {
    logger.warn('No se pudo guardar tab activa:', e.message);
  }
}

/**
 * Reproduce sonido de completado
 */
export function playCompletionSound() {
  if (!state.config.sound) return;
  
  try {
    const audioCtx = new (window.AudioContext || window.webkitAudioContext)();
    const frequencies = [523.25, 659.25, 783.99]; // C5, E5, G5
    
    frequencies.forEach((freq, i) => {
      setTimeout(() => {
        const osc = audioCtx.createOscillator();
        const gain = audioCtx.createGain();
        osc.connect(gain);
        gain.connect(audioCtx.destination);
        osc.frequency.value = freq;
        osc.type = 'sine';
        gain.gain.setValueAtTime(0.3, audioCtx.currentTime);
        gain.gain.exponentialRampToValueAtTime(0.01, audioCtx.currentTime + 0.3);
        osc.start(audioCtx.currentTime);
        osc.stop(audioCtx.currentTime + 0.3);
      }, i * 150);
    });
    
    logger.log('🔊 Sonido reproducido');
  } catch (e) {
    logger.warn('No se pudo reproducir sonido:', e.message);
  }
}

// ==================== COUNTDOWN ====================

let countdownInterval = null;

/**
 * Muestra el contador de cuenta atrás en el footer
 */
export function showCountdown() {
  const el = document.getElementById('mil-footer-countdown');
  if (el) el.style.display = 'flex';
}

/**
 * Oculta el contador de cuenta atrás
 */
export function hideCountdown() {
  const el = document.getElementById('mil-footer-countdown');
  if (el) el.style.display = 'none';
  stopCountdown();
}

/**
 * Inicia la cuenta atrás
 * @param {number} seconds - Segundos iniciales
 * @param {Function} onComplete - Callback al terminar (opcional)
 */
export function startCountdown(seconds, onComplete = null) {
  stopCountdown(); // Limpiar cualquier intervalo anterior
  
  let remaining = seconds;
  const valueEl = document.getElementById('mil-countdown-value');
  
  // Mostrar valor inicial
  if (valueEl) valueEl.textContent = `${remaining}s`;
  showCountdown();
  
  countdownInterval = setInterval(() => {
    remaining--;
    
    if (valueEl) valueEl.textContent = `${remaining}s`;
    
    if (remaining <= 0) {
      stopCountdown();
      hideCountdown();
      if (onComplete) onComplete();
    }
  }, 1000);
}

/**
 * Detiene la cuenta atrás
 */
export function stopCountdown() {
  if (countdownInterval) {
    clearInterval(countdownInterval);
    countdownInterval = null;
  }
}
