/**
 * MitikLive Milanuncios - Estadísticas
 */

import { state } from './state.js';
import { getStorage, setStorage, createLogger } from '../../scripts/utils.js';

const logger = createLogger('Milanuncios', 'Stats');

/**
 * Carga estadísticas desde storage
 */
export async function loadStats() {
  const stored = await getStorage(['milStats', 'milStatsDate']);
  const today = new Date().toDateString();
  
  // Si es un nuevo día, resetear
  if (stored.milStatsDate !== today) {
    state.stats = {
      renovatedToday: 0,
      cooldownToday: 0,
      errorsToday: 0,
      lastRenewal: null
    };
    await setStorage({ milStats: state.stats, milStatsDate: today });
  } else if (stored.milStats) {
    state.stats = stored.milStats;
  }
  
  renderStats();
}

/**
 * Actualiza y guarda estadísticas
 */
export async function updateStats(type) {
  const today = new Date().toDateString();
  
  if (type === 'renovated') {
    state.stats.renovatedToday++;
    state.stats.lastRenewal = new Date().toISOString();
  } else if (type === 'cooldown') {
    state.stats.cooldownToday++;
  } else if (type === 'error') {
    state.stats.errorsToday++;
  }
  
  await setStorage({ milStats: state.stats, milStatsDate: today });
  renderStats();
}

/**
 * Renderiza estadísticas en la UI
 */
export function renderStats() {
  const renovatedEl = document.getElementById('mil-stats-renovated');
  const cooldownEl = document.getElementById('mil-stats-cooldown');
  const errorsEl = document.getElementById('mil-stats-errors');
  const totalEl = document.getElementById('mil-stats-total');
  const lastTimeEl = document.getElementById('mil-stats-last-time');
  
  if (renovatedEl) renovatedEl.textContent = state.stats.renovatedToday;
  if (cooldownEl) cooldownEl.textContent = state.stats.cooldownToday;
  if (errorsEl) errorsEl.textContent = state.stats.errorsToday;
  if (totalEl) totalEl.textContent = state.totalActiveAds || state.ads.length;
  
  if (lastTimeEl) {
    if (state.stats.lastRenewal) {
      const date = new Date(state.stats.lastRenewal);
      lastTimeEl.textContent = date.toLocaleTimeString('es-ES', {
        hour: '2-digit',
        minute: '2-digit'
      });
    } else {
      lastTimeEl.textContent = '--';
    }
  }
}
