/**
 * MitikLive Milanuncios - Sistema de Modales
 * Unifica todas las funciones de modal en un solo lugar
 */

import { createLogger } from '../../scripts/utils.js';

const logger = createLogger('Milanuncios', 'Modals');

// Variable para tracking de pestaña existente
let existingMilTab = null;

/**
 * Oculta todos los modales
 */
export function hideAllModals() {
  const modals = document.querySelectorAll('.mil-modal');
  modals.forEach(modal => modal.style.display = 'none');
  existingMilTab = null;
}

/**
 * Oculta modal específico de tab
 */
export function hideModal() {
  const modal = document.getElementById('mil-modal-open-tab');
  if (modal) modal.style.display = 'none';
  existingMilTab = null;
}

/**
 * Muestra modal de login requerido
 */
export function showLoginModal() {
  hideAllModals();
  const modal = document.getElementById('mil-modal-login');
  if (modal) modal.style.display = 'flex';
}

/**
 * Muestra modal de exportación
 */
export function showExportModal() {
  hideAllModals();
  const modal = document.getElementById('mil-modal-export');
  if (modal) {
    modal.style.display = 'flex';
    updateExportProgress(0, 0, 'Preparando exportación...');
  }
}

/**
 * Oculta modal de exportación
 */
export function hideExportModal() {
  const modal = document.getElementById('mil-modal-export');
  if (modal) modal.style.display = 'none';
}

/**
 * Actualiza progreso de exportación
 */
export function updateExportProgress(current, total, text) {
  const progressText = document.getElementById('mil-export-progress-text');
  const progressBar = document.getElementById('mil-export-progress-bar');
  const progressDetail = document.getElementById('mil-export-progress-detail');
  
  if (progressText) progressText.textContent = text;
  if (progressDetail) progressDetail.textContent = `${current} de ${total} anuncios`;
  if (progressBar) {
    const percent = total > 0 ? (current / total) * 100 : 0;
    progressBar.style.width = `${percent}%`;
  }
}

/**
 * Muestra modal de confirmación
 * @returns {Promise<boolean>}
 */
export function showConfirm(options = {}) {
  return new Promise((resolve) => {
    const {
      title = 'Confirmar',
      message = '¿Estás seguro?',
      details = null,
      icon = '❓',
      confirmText = 'Confirmar',
      cancelText = 'Cancelar',
      confirmClass = 'btn-primary',
      dangerous = false
    } = options;
    
    const modal = document.getElementById('mil-modal-generic');
    if (!modal) {
      resolve(confirm(message));
      return;
    }
    
    const titleEl = document.getElementById('mil-modal-title');
    const messageEl = document.getElementById('mil-modal-message');
    const iconEl = document.getElementById('mil-modal-icon');
    const detailsEl = document.getElementById('mil-modal-details');
    const footerEl = document.getElementById('mil-modal-footer');
    
    titleEl.textContent = title;
    messageEl.textContent = message;
    iconEl.textContent = icon;
    
    if (details) {
      detailsEl.innerHTML = details;
      detailsEl.style.display = 'block';
    } else {
      detailsEl.style.display = 'none';
    }
    
    footerEl.innerHTML = `
      <button class="btn btn-secondary" data-action="cancel">${cancelText}</button>
      <button class="btn ${dangerous ? 'btn-danger' : confirmClass}" data-action="confirm">${confirmText}</button>
    `;
    
    modal.style.display = 'flex';
    
    const cleanup = () => {
      modal.style.display = 'none';
      footerEl.removeEventListener('click', handleClick);
    };
    
    const handleClick = (e) => {
      const action = e.target.dataset.action;
      if (action === 'confirm') {
        cleanup();
        resolve(true);
      } else if (action === 'cancel') {
        cleanup();
        resolve(false);
      }
    };
    
    const handleOverlay = () => {
      cleanup();
      resolve(false);
    };
    
    footerEl.addEventListener('click', handleClick);
    modal.querySelector('.mil-modal__overlay')?.addEventListener('click', handleOverlay, { once: true });
  });
}

/**
 * Muestra modal genérico con botones personalizados
 * @returns {Promise<string>}
 */
export function showGenericModal(options = {}) {
  return new Promise((resolve) => {
    const {
      title = 'Aviso',
      message = '',
      details = null,
      icon = 'ℹ️',
      buttons = [{ text: 'Aceptar', value: 'ok', primary: true }]
    } = options;
    
    const modal = document.getElementById('mil-modal-generic');
    if (!modal) {
      resolve('cancel');
      return;
    }
    
    const titleEl = document.getElementById('mil-modal-title');
    const messageEl = document.getElementById('mil-modal-message');
    const iconEl = document.getElementById('mil-modal-icon');
    const detailsEl = document.getElementById('mil-modal-details');
    const footerEl = document.getElementById('mil-modal-footer');
    
    titleEl.textContent = title;
    messageEl.textContent = message;
    iconEl.textContent = icon;
    
    if (details) {
      detailsEl.innerHTML = details;
      detailsEl.style.display = 'block';
    } else {
      detailsEl.style.display = 'none';
    }
    
    footerEl.innerHTML = buttons.map(btn => `
      <button class="btn ${btn.primary ? 'btn-primary' : 'btn-secondary'}" data-value="${btn.value}">
        ${btn.text}
      </button>
    `).join('');
    
    modal.style.display = 'flex';
    
    const handleClick = (e) => {
      const value = e.target.dataset.value;
      if (value) {
        modal.style.display = 'none';
        resolve(value);
      }
    };
    
    const handleOverlay = () => {
      modal.style.display = 'none';
      resolve('cancel');
    };
    
    footerEl.addEventListener('click', handleClick);
    modal.querySelector('.mil-modal__overlay')?.addEventListener('click', handleOverlay, { once: true });
  });
}

/**
 * Muestra alerta (solo OK)
 */
export function showAlert(options = {}) {
  return new Promise((resolve) => {
    const {
      title = 'Aviso',
      message = '',
      details = null,
      icon = 'ℹ️',
      okText = 'Aceptar'
    } = options;
    
    const modal = document.getElementById('mil-modal-generic');
    if (!modal) {
      alert(message);
      resolve();
      return;
    }
    
    const titleEl = document.getElementById('mil-modal-title');
    const messageEl = document.getElementById('mil-modal-message');
    const iconEl = document.getElementById('mil-modal-icon');
    const detailsEl = document.getElementById('mil-modal-details');
    const footerEl = document.getElementById('mil-modal-footer');
    
    titleEl.textContent = title;
    messageEl.textContent = message;
    iconEl.textContent = icon;
    
    if (details) {
      detailsEl.innerHTML = details;
      detailsEl.style.display = 'block';
    } else {
      detailsEl.style.display = 'none';
    }
    
    footerEl.innerHTML = `<button class="btn btn-primary" data-action="ok">${okText}</button>`;
    
    modal.style.display = 'flex';
    
    const cleanup = () => modal.style.display = 'none';
    
    const handleClick = (e) => {
      if (e.target.dataset.action === 'ok') {
        cleanup();
        resolve();
      }
    };
    
    footerEl.addEventListener('click', handleClick, { once: true });
    modal.querySelector('.mil-modal__overlay')?.addEventListener('click', () => {
      cleanup();
      resolve();
    }, { once: true });
  });
}

/**
 * Getter/Setter para existingMilTab
 */
export function getExistingMilTab() {
  return existingMilTab;
}

export function setExistingMilTab(tab) {
  existingMilTab = tab;
}
