/**
 * MitikLive Milanuncios - Sistema de Créditos
 */

import { state, API_BASE_URL, RENOVATIONS_PER_CREDIT } from './state.js';
import { createLogger } from '../../scripts/utils.js';

const logger = createLogger('Milanuncios', 'Credits');

/**
 * Carga créditos del usuario desde la API
 */
export async function loadCredits() {
  if (!state.userId) {
    logger.warn('No hay userId para cargar créditos');
    return;
  }
  
  if (!state.jwt) {
    logger.warn('No hay JWT para cargar créditos');
    return;
  }
  
  try {
    logger.log('Cargando créditos para userId:', state.userId);
    
    const response = await fetch(`${API_BASE_URL}/billing/check-credits?user_id=${state.userId}`, {
      headers: {
        'Authorization': `Bearer ${state.jwt}`,
        'Content-Type': 'application/json',
        'X-Extension-Version': chrome.runtime.getManifest().version
      }
    });
    
    if (!response.ok) {
      const errorText = await response.text().catch(() => 'Sin detalles');
      throw new Error(`HTTP ${response.status}: ${errorText}`);
    }
    
    const data = await response.json();
    state.credits = data.credits_available || 0;
    
    logger.log('✅ Créditos cargados:', state.credits);
    
    updateCreditsUI();
    
  } catch (error) {
    logger.error('Error cargando créditos:', error.message);
  }
}

/**
 * Descuenta un crédito (cada 5 renovaciones)
 */
export async function deductCredit() {
  if (!state.userId || !state.jwt) return;
  
  state.renovationCount++;
  
  if (state.renovationCount < RENOVATIONS_PER_CREDIT) {
    logger.log(`Renovaciones: ${state.renovationCount}/${RENOVATIONS_PER_CREDIT} - Sin descuento`);
    return;
  }
  
  state.renovationCount = 0;
  
  try {
    const response = await fetch(`${API_BASE_URL}/billing/deduct-credit`, {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${state.jwt}`,
        'Content-Type': 'application/json',
        'X-Extension-Version': chrome.runtime.getManifest().version
      },
      body: JSON.stringify({ user_id: state.userId })
    });
    
    if (!response.ok) {
      throw new Error(`HTTP ${response.status}`);
    }
    
    const data = await response.json();
    state.credits = data.credits_remaining;
    
    updateCreditsUI();
    
    logger.log('✅ Crédito descontado. Quedan:', state.credits);
    
  } catch (error) {
    logger.error('Error descontando crédito:', error.message);
  }
}

/**
 * Actualiza la UI de créditos
 */
function updateCreditsUI() {
  const euroValue = (state.credits * 0.10).toFixed(2).replace('.', ',');
  
  // Header
  const creditsContainer = document.getElementById('mil-credits-container');
  const creditsBadge = document.getElementById('mil-credits-badge');
  const creditsCount = document.getElementById('mil-credits-count');
  
  if (creditsContainer && creditsBadge && creditsCount) {
    creditsCount.textContent = state.credits;
    creditsContainer.style.display = 'flex';
    
    if (state.credits === 0) {
      creditsBadge.classList.add('no-credits');
    } else {
      creditsBadge.classList.remove('no-credits');
    }
  }
  
  // Config tab
  const configCredits = document.getElementById('mil-config-credit-count');
  if (configCredits) {
    configCredits.textContent = state.credits;
  }
  
  const configEuro = document.getElementById('mil-config-credit-euro');
  if (configEuro) {
    configEuro.textContent = `(${euroValue}€)`;
  }
}
