/**
 * MitikLive Milanuncios - Conexión con Content Script v2
 * Usa chrome.tabs listeners + mensajes (no puertos)
 */

import { state } from './state.js';
import { createLogger } from '../../scripts/utils.js';

const logger = createLogger('Milanuncios', 'Connection');

// Callbacks registrados
const listeners = {
  onStatusChange: null,
  onAdsUpdate: null,
  onRenewResult: null,
  onDisconnect: null
};

// Estados posibles
export const ConnectionState = {
  DISCONNECTED: 'disconnected',
  CONNECTED_NO_LOGIN: 'no_login',
  CONNECTED_WRONG_PAGE: 'wrong_page',
  READY: 'ready',
  RENEWING: 'renewing'
};

// Estado actual
let connectionState = ConnectionState.DISCONNECTED;
let checkInterval = null;

/**
 * Obtiene el estado actual de conexión
 */
export function getConnectionState() {
  return connectionState;
}

/**
 * Verifica si está listo
 */
export function isReady() {
  return connectionState === ConnectionState.READY;
}

/**
 * Verifica si hay conexión
 */
export function isConnected() {
  return state.currentTabId !== null && connectionState !== ConnectionState.DISCONNECTED;
}

/**
 * Registra callbacks para eventos
 */
export function setListeners(callbacks) {
  Object.assign(listeners, callbacks);
}

/**
 * Inicializa listeners de tabs y verificación periódica
 */
export function initConnectionListener() {
  logger.log('Inicializando listeners de tabs...');
  
  // Escuchar cambios en pestañas
  chrome.tabs.onUpdated.addListener(handleTabUpdated);
  chrome.tabs.onRemoved.addListener(handleTabRemoved);
  chrome.tabs.onActivated.addListener(handleTabActivated);
  
  // Verificación inicial
  checkMilanunciosTabs();
  
  // Verificación periódica cada 2 segundos
  checkInterval = setInterval(checkMilanunciosTabs, 2000);
}

/**
 * Detiene la verificación periódica y limpia listeners
 */
export function stopConnectionListener() {
  if (checkInterval) {
    clearInterval(checkInterval);
    checkInterval = null;
  }
  
  // Limpiar listeners (importante para evitar memory leaks)
  try {
    chrome.tabs.onUpdated.removeListener(handleTabUpdated);
    chrome.tabs.onRemoved.removeListener(handleTabRemoved);
    chrome.tabs.onActivated.removeListener(handleTabActivated);
  } catch (e) {
    // Ignorar errores si ya fueron removidos
  }
}

/**
 * Verifica pestañas de Milanuncios
 * Busca la mejor pestaña, verifica CS, reinyecta si necesario
 */
async function checkMilanunciosTabs() {
  try {
    const tabs = await chrome.tabs.query({ url: '*://*.milanuncios.com/*' });
    
    // Sin pestañas de Milanuncios
    if (tabs.length === 0) {
      if (connectionState !== ConnectionState.DISCONNECTED) {
        logger.log('No hay pestañas de Milanuncios');
        setDisconnected();
      }
      return;
    }
    
    // Buscar la mejor pestaña (prioridad: mis-anuncios > otras)
    let bestTab = tabs.find(t => t.url?.includes('/mis-anuncios')) || tabs[0];
    
    // Si la pestaña actual ya no existe o cambió, actualizar
    if (state.currentTabId && state.currentTabId !== bestTab.id) {
      const oldTabExists = tabs.some(t => t.id === state.currentTabId);
      if (!oldTabExists) {
        logger.log('Pestaña anterior ya no existe, cambiando a:', bestTab.id);
        state.currentTabId = null;
      }
    }
    
    // Intentar comunicarse con el content script
    const status = await pingContentScript(bestTab.id);
    
    if (status && status.pong) {
      // CS responde correctamente
      state.currentTabId = bestTab.id;
      updateConnectionState(status);
    } else {
      // CS no responde - intentar inyectar
      logger.log('CS no responde en tab:', bestTab.id);
      await injectContentScript(bestTab.id);
    }
    
  } catch (error) {
    logger.error('Error verificando tabs:', error.message);
  }
}

/**
 * Envía ping al content script
 */
async function pingContentScript(tabId) {
  return new Promise((resolve) => {
    let resolved = false;
    
    // Timeout de seguridad
    const timeoutId = setTimeout(() => {
      if (!resolved) {
        resolved = true;
        resolve(null);
      }
    }, 1500);
    
    try {
      chrome.tabs.sendMessage(tabId, { action: 'mil_ping' }, (response) => {
        if (!resolved) {
          resolved = true;
          clearTimeout(timeoutId);
          
          if (chrome.runtime.lastError) {
            resolve(null);
          } else {
            resolve(response);
          }
        }
      });
    } catch {
      if (!resolved) {
        resolved = true;
        clearTimeout(timeoutId);
        resolve(null);
      }
    }
  });
}

/**
 * Inyecta content script si es necesario
 * Con reintentos y mejor manejo de errores
 */
async function injectContentScript(tabId, retryCount = 0) {
  const MAX_RETRIES = 2;
  
  try {
    // Verificar que la pestaña aún existe y está lista
    const tab = await chrome.tabs.get(tabId).catch(() => null);
    if (!tab) {
      logger.warn('Pestaña no existe:', tabId);
      return;
    }
    
    // No inyectar en páginas del sistema
    if (tab.url?.startsWith('chrome://') || tab.url?.startsWith('about:')) {
      return;
    }
    
    await chrome.scripting.executeScript({
      target: { tabId },
      files: ['milanuncios/content-script.js']
    });
    logger.log('Content script inyectado en tab:', tabId);
    
    // Esperar a que el CS se inicialice y verificar
    setTimeout(async () => {
      const status = await pingContentScript(tabId);
      if (status && status.pong) {
        state.currentTabId = tabId;
        updateConnectionState(status);
      } else if (retryCount < MAX_RETRIES) {
        // Reintentar
        logger.log(`CS no responde, reintentando (${retryCount + 1}/${MAX_RETRIES})`);
        setTimeout(() => injectContentScript(tabId, retryCount + 1), 1000);
      } else {
        logger.error('CS no responde después de reintentos');
      }
    }, 800);
    
  } catch (e) {
    // Errores comunes:
    // - "Cannot access a chrome:// URL"
    // - "The extensions gallery cannot be scripted"
    // - "Frame not found"
    if (e.message?.includes('Cannot access') || e.message?.includes('cannot be scripted')) {
      logger.log('Página no scriptable:', tabId);
    } else {
      logger.warn('Error inyectando CS:', e.message);
    }
  }
}

/**
 * Actualiza el estado de conexión basado en respuesta del CS
 */
function updateConnectionState(status) {
  const { isLogged, isMyAdsPage, userInfo } = status;
  
  let newState;
  
  if (!isLogged) {
    newState = ConnectionState.CONNECTED_NO_LOGIN;
  } else if (!isMyAdsPage) {
    newState = ConnectionState.CONNECTED_WRONG_PAGE;
  } else {
    newState = ConnectionState.READY;
  }
  
  // Detectar si cambió el usuario (aunque el estado sea el mismo)
  const currentUserId = state.currentMilUserId || null;
  const newUserId = userInfo?.userId || null;
  const userChanged = currentUserId !== newUserId && newUserId !== null;
  
  if (userChanged) {
    logger.log('Usuario cambió:', currentUserId, '→', newUserId);
    state.currentMilUserId = newUserId;
    state.currentMilUsername = userInfo?.username || null;
    // Limpiar anuncios al cambiar de usuario
    state.ads = [];
  }
  
  // Notificar si:
  // 1. Es la primera conexión (connectionState era DISCONNECTED)
  // 2. Cambió el estado
  // 3. Cambió el usuario
  const wasDisconnected = connectionState === ConnectionState.DISCONNECTED;
  const stateChanged = newState !== connectionState;
  const shouldNotify = wasDisconnected || stateChanged || userChanged;
  
  if (shouldNotify) {
    logger.log('Notificando - Estado:', connectionState, '→', newState, 
               '| Primera conexión:', wasDisconnected, 
               '| Usuario cambió:', userChanged);
    connectionState = newState;
    
    if (listeners.onStatusChange) {
      listeners.onStatusChange({
        state: newState,
        user: userInfo,
        url: status.url,
        userChanged
      });
    }
  }
  
  // Si está listo y no hay anuncios (o cambió usuario), solicitarlos
  if (newState === ConnectionState.READY && (state.ads.length === 0 || userChanged)) {
    requestScanAds();
  }
}

/**
 * Establece estado desconectado
 */
function setDisconnected() {
  connectionState = ConnectionState.DISCONNECTED;
  state.currentTabId = null;
  state.currentMilUserId = null;  // Limpiar usuario de Milanuncios
  state.currentMilUsername = null;  // Limpiar username
  state.ads = [];  // Limpiar anuncios
  
  if (listeners.onDisconnect) {
    listeners.onDisconnect();
  }
}

/**
 * Maneja actualización de pestaña
 * Detecta: recarga, cambio de URL (SPA), carga completa
 */
function handleTabUpdated(tabId, changeInfo, tab) {
  const url = tab?.url || '';
  
  // Solo nos interesan pestañas de Milanuncios
  if (!url.includes('milanuncios.com')) return;
  
  // Detectar cambio de URL (navegación SPA)
  if (changeInfo.url) {
    logger.log('URL cambió:', changeInfo.url);
    // Esperar a que el DOM se actualice
    setTimeout(() => checkMilanunciosTabs(), 800);
  }
  
  // Detectar carga completa
  if (changeInfo.status === 'complete') {
    logger.log('Pestaña Milanuncios cargada:', tabId);
    // Esperar un poco más para que el CS se inicialice
    setTimeout(() => checkMilanunciosTabs(), 1000);
  }
}

/**
 * Maneja cierre de pestaña
 */
async function handleTabRemoved(tabId) {
  if (tabId === state.currentTabId) {
    logger.log('Pestaña actual cerrada');
    state.currentTabId = null;
    setDisconnected();
  }
  
  // Verificar si quedan otras pestañas de Milanuncios
  setTimeout(() => checkMilanunciosTabs(), 100);
}

/**
 * Maneja cambio de pestaña activa
 * Detecta cuando el usuario cambia a/desde pestaña de Milanuncios
 */
function handleTabActivated(activeInfo) {
  // Verificar si la pestaña activa es de Milanuncios
  chrome.tabs.get(activeInfo.tabId, (tab) => {
    if (chrome.runtime.lastError) return;
    
    if (tab?.url?.includes('milanuncios.com')) {
      logger.log('Usuario cambió a pestaña Milanuncios');
      // Si es diferente a la actual, verificar
      if (tab.id !== state.currentTabId) {
        setTimeout(() => checkMilanunciosTabs(), 300);
      }
    }
  });
}

// ==================== COMANDOS ====================

/**
 * Solicita escaneo de anuncios
 */
export function requestScanAds() {
  if (!state.currentTabId) {
    logger.error('No hay tabId para scanAds');
    return false;
  }
  
  logger.log('Solicitando escaneo de anuncios...');
  
  chrome.tabs.sendMessage(state.currentTabId, { action: 'mil_scrapeAds' }, (response) => {
    if (chrome.runtime.lastError) {
      logger.error('Error:', chrome.runtime.lastError.message);
      return;
    }
    
    if (response && response.ads) {
      logger.log(`Recibidos ${response.ads.length} anuncios`);
      
      if (listeners.onAdsUpdate) {
        listeners.onAdsUpdate({
          ads: response.ads,
          pagination: response.pagination,
          totalActive: response.totalActiveAds
        });
      }
    }
  });
  
  return true;
}

/**
 * Solicita renovación
 */
export function requestRenew(adId) {
  if (!state.currentTabId) return false;
  
  logger.log('Solicitando renovación:', adId);
  
  chrome.tabs.sendMessage(state.currentTabId, { action: 'mil_renewAd', adId }, (response) => {
    if (chrome.runtime.lastError) {
      if (listeners.onRenewResult) {
        listeners.onRenewResult({ adId, success: false, error: 'Error de conexión' });
      }
      return;
    }
    
    if (listeners.onRenewResult) {
      listeners.onRenewResult({
        adId,
        success: response?.success || false,
        error: response?.error
      });
    }
  });
  
  return true;
}

/**
 * Solicita navegación
 */
export function requestNavigate(page) {
  if (!state.currentTabId) return false;
  
  chrome.tabs.sendMessage(state.currentTabId, { action: 'mil_navigateToPage', page });
  return true;
}

/**
 * Solicita estado actual
 */
export function requestStatus() {
  checkMilanunciosTabs();
}

/**
 * Fuerza reconexión - resetea estado y busca pestañas de nuevo
 * Útil como fallback si la detección automática falla
 */
export async function forceReconnect() {
  logger.log('Forzando reconexión...');
  
  // Resetear estado
  state.currentTabId = null;
  state.currentMilUserId = null;
  connectionState = ConnectionState.DISCONNECTED;
  
  // Buscar pestañas inmediatamente
  await checkMilanunciosTabs();
  
  return connectionState !== ConnectionState.DISCONNECTED;
}
