/**
 * MitikLive Milanuncios - Configuración v5.8.3
 * Sincroniza preferencias con backend (persistencia multi-dispositivo)
 */

import { state } from './state.js';
import { getStorage, setStorage, createLogger } from '../../scripts/utils.js';

const logger = createLogger('Milanuncios', 'Config');

let autoRefreshInterval = null;

// ==================== BACKEND SYNC ====================

/**
 * Obtiene la URL base de la API
 */
function getApiBase() {
  return state.API_BASE_URL || 'https://www.mitiklive.com/fa/api';
}

/**
 * Carga preferencias desde el backend
 */
async function loadPreferencesFromBackend() {
  const { jwt } = await getStorage(['jwt']);
  if (!jwt) {
    logger.log('Sin JWT, no se cargan preferencias del backend');
    return null;
  }
  
  try {
    const response = await fetch(`${getApiBase()}/user/preferences`, {
      method: 'GET',
      headers: {
        'Authorization': `Bearer ${jwt}`,
        'Content-Type': 'application/json'
      }
    });
    
    if (!response.ok) {
      logger.error('Error cargando preferencias:', response.status);
      return null;
    }
    
    const data = await response.json();
    logger.log('Preferencias del backend:', data.ui_preferences);
    return data.ui_preferences || {};
  } catch (e) {
    logger.error('Error en fetch preferencias:', e);
    return null;
  }
}

/**
 * Guarda preferencias en el backend
 */
async function savePreferencesToBackend(prefs) {
  const { jwt } = await getStorage(['jwt']);
  if (!jwt) {
    logger.log('Sin JWT, no se guardan preferencias en backend');
    return false;
  }
  
  try {
    const response = await fetch(`${getApiBase()}/user/preferences`, {
      method: 'PATCH',
      headers: {
        'Authorization': `Bearer ${jwt}`,
        'Content-Type': 'application/json'
      },
      body: JSON.stringify({
        ui_preferences: prefs
      })
    });
    
    if (!response.ok) {
      logger.error('Error guardando preferencias:', response.status);
      return false;
    }
    
    logger.log('Preferencias guardadas en backend');
    return true;
  } catch (e) {
    logger.error('Error en fetch guardar preferencias:', e);
    return false;
  }
}

// ==================== CONFIG MANAGEMENT ====================

/**
 * Carga configuración desde storage local y backend
 */
export async function loadConfig() {
  // 1. Cargar desde storage local (rápido)
  const stored = await getStorage(['milConfig']);
  if (stored.milConfig) {
    state.config = { ...state.config, ...stored.milConfig };
    logger.log('Config local cargada:', state.config);
  }
  
  // 2. Aplicar a UI inmediatamente (sin esperar backend)
  applyConfigToUI();
  
  // 3. Cargar desde backend (puede tardar más)
  const backendPrefs = await loadPreferencesFromBackend();
  if (backendPrefs) {
    // Mapear campos del backend a state.config
    if (backendPrefs.zoom_milanuncios !== undefined) {
      state.config.zoom = backendPrefs.zoom_milanuncios;
    }
    if (backendPrefs.delay_milanuncios !== undefined) {
      state.config.delay = backendPrefs.delay_milanuncios * 1000; // segundos a ms
    }
    if (backendPrefs.notifications_milanuncios !== undefined) {
      state.config.notifications = backendPrefs.notifications_milanuncios;
    }
    if (backendPrefs.auto_refresh_milanuncios !== undefined) {
      state.config.autoRefresh = backendPrefs.auto_refresh_milanuncios;
    }
    if (backendPrefs.sound_milanuncios !== undefined) {
      state.config.sound = backendPrefs.sound_milanuncios;
    }
    
    // Guardar en local y aplicar
    await setStorage({ milConfig: state.config });
    applyConfigToUI();
    logger.log('Config sincronizada con backend:', state.config);
  }
}

/**
 * Aplica la configuración actual a los elementos de la UI
 */
function applyConfigToUI() {
  const autoRefreshEl = document.getElementById('mil-config-auto-refresh');
  const delayEl = document.getElementById('mil-config-delay');
  const delayValueEl = document.getElementById('mil-delay-value');
  const notificationsEl = document.getElementById('mil-config-notifications');
  const soundEl = document.getElementById('mil-config-sound');
  const zoomEl = document.getElementById('mil-config-zoom');
  const zoomValueEl = document.getElementById('mil-zoom-value');
  
  if (autoRefreshEl) autoRefreshEl.checked = state.config.autoRefresh;
  if (delayEl) delayEl.value = Math.round(state.config.delay / 1000);
  if (delayValueEl) delayValueEl.textContent = Math.round(state.config.delay / 1000) + 's';
  if (notificationsEl) notificationsEl.checked = state.config.notifications;
  if (soundEl) soundEl.checked = state.config.sound;
  if (zoomEl) zoomEl.value = state.config.zoom;
  if (zoomValueEl) zoomValueEl.textContent = state.config.zoom + '%';
  
  applyZoom(state.config.zoom);
}

/**
 * Guarda configuración en storage local y backend
 */
export async function saveConfig() {
  // 1. Guardar en local (inmediato)
  await setStorage({ milConfig: state.config });
  logger.log('Config guardada localmente:', state.config);
  
  // 2. Guardar en backend (async, no bloquea)
  const backendPrefs = {
    zoom_milanuncios: state.config.zoom,
    delay_milanuncios: Math.round(state.config.delay / 1000), // ms a segundos
    notifications_milanuncios: state.config.notifications,
    auto_refresh_milanuncios: state.config.autoRefresh,
    sound_milanuncios: state.config.sound
  };
  
  // No esperamos la respuesta para no bloquear UI
  savePreferencesToBackend(backendPrefs).catch(e => {
    logger.error('Error guardando en backend:', e);
  });
}

/**
 * Guarda la pestaña activa
 */
export async function saveActiveTab(tabName) {
  const { jwt } = await getStorage(['jwt']);
  if (!jwt) return;
  
  // Guardar en backend
  savePreferencesToBackend({
    active_tab_milanuncios: tabName
  }).catch(e => {
    logger.error('Error guardando tab activa:', e);
  });
}

/**
 * Obtiene la pestaña activa guardada
 */
export async function getActiveTab() {
  const backendPrefs = await loadPreferencesFromBackend();
  return backendPrefs?.active_tab_milanuncios || 'renovaciones';
}

/**
 * Aplica nivel de zoom
 */
export function applyZoom(zoom) {
  document.body.style.zoom = zoom + '%';
}

/**
 * Inicia auto-refresh
 */
export function startAutoRefresh(scanAdsCallback) {
  if (autoRefreshInterval) return;
  
  autoRefreshInterval = setInterval(() => {
    if (!state.isRenewing && scanAdsCallback) {
      logger.log('Auto-refresh: escaneando...');
      scanAdsCallback();
    }
  }, 30000);
  
  logger.log('Auto-refresh iniciado');
}

/**
 * Detiene auto-refresh
 */
export function stopAutoRefresh() {
  if (autoRefreshInterval) {
    clearInterval(autoRefreshInterval);
    autoRefreshInterval = null;
    logger.log('Auto-refresh detenido');
  }
}
