/**
 * MitikLive - Panel de Milanuncios v2.1.0
 * Sistema de puerto persistente bidireccional
 */

// ==================== IMPORTS ====================

// Utilidades compartidas
import { 
  showToast, 
  delay, 
  createLogger,
  getStorage,
  showLoader,
  hideLoader,
  showModal
} from '../scripts/utils.js';

// Módulos de Milanuncios
import { state, DELAY_VARIANCE, MAX_RETRIES, API_BASE_URL } from './scripts/state.js';
import { loadConfig, saveConfig, applyZoom, startAutoRefresh, stopAutoRefresh, getActiveTab } from './scripts/config.js';
import { loadCredits, deductCredit } from './scripts/credits.js';
import { 
  hideAllModals, 
  showLoginModal, 
  showExportModal, 
  hideExportModal, 
  updateExportProgress,
  showConfirm,
  showAlert,
  getExistingMilTab,
  setExistingMilTab
} from './scripts/modals.js';
import { showStatus, hideStatus, switchTab, playCompletionSound, showCountdown, hideCountdown, startCountdown, stopCountdown } from './scripts/ui.js';
import { loadStats, updateStats, renderStats } from './scripts/stats.js';

// Sistema de conexión persistente
import {
  ConnectionState,
  getConnectionState,
  isReady,
  isConnected,
  setListeners,
  initConnectionListener,
  requestScanAds,
  requestRenew,
  requestNavigate,
  requestStatus,
  forceReconnect
} from './scripts/connection.js';

// ==================== LOGGER ====================

const logger = createLogger('Milanuncios', 'Panel');
logger.log('Inicializando v2.1...');

// ==================== HELPERS ====================

/**
 * Delay aleatorio para simular comportamiento humano
 * Muestra countdown en el footer durante la espera
 */
async function randomDelay(baseMs) {
  const variance = baseMs * DELAY_VARIANCE;
  const min = baseMs - variance;
  const max = baseMs + variance;
  const actualDelay = Math.floor(Math.random() * (max - min + 1)) + min;
  const seconds = Math.ceil(actualDelay / 1000);
  
  logger.log(`Delay: ${actualDelay}ms (${seconds}s)`);
  
  // Mostrar countdown
  startCountdown(seconds);
  
  // Esperar el delay real
  await delay(actualDelay);
  
  // Ocultar countdown
  hideCountdown();
}

// ==================== INIT ====================

document.addEventListener('DOMContentLoaded', async () => {
  logger.log('DOM cargado');
  
  // ✅ v5.8.3: Cargar versión desde manifest.json
  const versionEl = document.getElementById('mil-footer-version');
  if (versionEl) {
    try {
      const manifest = chrome.runtime.getManifest();
      versionEl.textContent = `v${manifest.version}`;
    } catch (e) {
      logger.error('Error obteniendo versión:', e);
    }
  }
  
  // Setup listeners PRIMERO
  setupListeners();
  
  // Configurar callbacks del sistema de conexión
  setupConnectionHandlers();
  
  // =====================================================================
  // Obtener JWT y user del storage ANTES de iniciar conexión
  // (handleConnectionStatusChange necesita state.jwt para verificar cuenta)
  // =====================================================================
  const stored = await getStorage([
    'jwt', 'user',
    'session.access', 'session.user'
  ]);
  
  const jwt = stored['session.access'] || stored.jwt;
  const user = stored['session.user'] || stored.user;
  
  logger.log('JWT obtenido:', jwt ? 'Sí ✅' : 'No ❌');
  
  state.jwt = jwt;
  
  // Obtener user_id
  if (user && user.id) {
    state.userId = parseInt(user.id);
  } else if (jwt) {
    try {
      const payload = JSON.parse(atob(jwt.split('.')[1]));
      state.userId = parseInt(payload.sub || payload.user_id || payload.id);
    } catch (e) {
      logger.error('Error decodificando JWT:', e);
    }
  }
  
  // Mostrar email del usuario MitikLive en header
  if (user && (user.email || user.username)) {
    const panelEmailEl = document.getElementById('mil-panel-email');
    if (panelEmailEl) {
      panelEmailEl.textContent = user.email || user.username;
    }
  }
  
  // Mostrar estado inicial ANTES de intentar conectar
  showStatus('🔍 Buscando Milanuncios...', 'info', 'Conectando con la pestaña...');
  
  // Solo mostrar mensaje de "Cargando" si no hay JWT
  if (!jwt) {
    showStatus('⚠️ Cargando...', 'info', 'Verificando sesión');
  }
  
  // Iniciar listener de conexiones DESPUÉS de tener JWT
  // (handleConnectionStatusChange puede dispararse inmediatamente)
  initConnectionListener();
  
  // Inicializar módulos
  await loadConfig();
  applyZoom(state.config.zoom);
  await loadCredits();
  await loadStats();
  
  // ✅ v5.8.3: Cargar tab activa guardada
  const savedTab = await getActiveTab();
  if (savedTab && savedTab !== 'renovaciones') {
    switchTab(savedTab);
  }
  
  // Verificar si ya hay pestañas abiertas (muestra modal si no conecta después de 5s)
  checkExistingTabs();
  
  // Iniciar refresh periódico de créditos (cada 60s)
  setInterval(() => loadCredits(), 60000);
  
  // Escuchar cambios en storage (para stats)
  chrome.storage.onChanged.addListener(handleStorageChange);
  
  // ✅ v5.8.4: Iniciar auto-refresh si está activado en config
  if (state.config.autoRefresh) {
    startAutoRefresh(scanAds);
  }
  
  logger.log('✅ Inicializado correctamente');
});

// ==================== CONNECTION HANDLERS ====================

/**
 * Configura los handlers para eventos de conexión
 */
function setupConnectionHandlers() {
  setListeners({
    onStatusChange: handleConnectionStatusChange,
    onAdsUpdate: handleAdsUpdate,
    onRenewResult: handleRenewResult,
    onDisconnect: handleDisconnect
  });
}

/**
 * Maneja cambios de estado de conexión
 */
async function handleConnectionStatusChange({ state: connState, user, url, userChanged }) {
  logger.log('📡 Estado de conexión:', connState, userChanged ? '(usuario cambió)' : '');
  
  // Ocultar modal de "abrir Milanuncios" si ya hay conexión
  if (connState !== ConnectionState.DISCONNECTED) {
    const modal = document.getElementById('mil-modal-open-tab');
    if (modal && modal.style.display !== 'none') {
      modal.style.display = 'none';
      logger.log('Modal ocultado - conexión establecida');
    }
  }
  
  // Siempre actualizar UI de usuario según el estado
  if (connState === ConnectionState.READY || connState === ConnectionState.CONNECTED_WRONG_PAGE) {
    // Logueado - mostrar info de usuario
    updateUserUI(user);
  } else {
    // No logueado o desconectado - limpiar
    updateUserUI(null);
  }
  
  // =====================================================================
  // 🔒 ANTI-ABUSO: Verificar cuenta al detectar nuevo usuario de Milanuncios
  // =====================================================================
  if (userChanged && connState === ConnectionState.READY && state.currentMilUserId) {
    logger.log('🔒 Verificando cuenta de Milanuncios (usuario nuevo)...');
    const accountCheck = await verifyMilanunciosAccount();
    
    if (!accountCheck.ok && accountCheck.error === 'duplicate') {
      // Cuenta duplicada - mostrar modal y desconectar
      await showModal({
        title: '⚠️ Cuenta ya vinculada',
        html: `
          <p>Esta cuenta de Milanuncios ya está vinculada a otro usuario.</p>
          <p style="margin-top: 12px; color: var(--text-secondary, #aaa);">
            Si eres el propietario legítimo y perdiste acceso a tu cuenta anterior, contacta con soporte.
          </p>
          <div style="margin-top: 16px; display: flex; gap: 12px; justify-content: center; flex-wrap: wrap;">
            <a href="https://wa.me/34618620664" target="_blank" style="color: #25D366; text-decoration: none; font-weight: 500;">
              💬 WhatsApp
            </a>
            <a href="mailto:soporte@mitiklive.com" target="_blank" style="color: #FF6B35; text-decoration: none; font-weight: 500;">
              📧 Email
            </a>
          </div>
        `,
        buttons: [{ text: 'Entendido', value: 'ok', primary: true }]
      });
      
      // Bloquear UI - no mostrar anuncios
      showStatus('🚫 Cuenta no autorizada', 'error', 'Contacta con soporte');
      hideAdsUI();
      return; // No continuar con la actualización de UI normal
    }
  }
  
  // Actualizar UI según estado
  switch (connState) {
    case ConnectionState.DISCONNECTED:
      showStatus('🔌 Milanuncios no detectado', 'warning', 
        'Abre una pestaña de Milanuncios');
      hideAdsUI();
      break;
      
    case ConnectionState.CONNECTED_NO_LOGIN:
      hideAllModals(); // Cerrar cualquier modal abierto
      showStatus('🔒 Inicia sesión en Milanuncios', 'warning',
        'Debes estar logueado para gestionar tus anuncios');
      hideAdsUI();
      break;
      
    case ConnectionState.CONNECTED_WRONG_PAGE:
      hideAllModals();
      showStatus('ℹ️ Navega a "Mis anuncios"', 'info',
        'Ve a tu página de anuncios en Milanuncios');
      hideAdsUI();
      break;
      
    case ConnectionState.READY:
      hideAllModals();
      hideStatus();
      showAdsUI();
      // El content script enviará ADS_UPDATE automáticamente
      break;
  }
}

/**
 * Maneja actualización de anuncios desde el CS
 */
function handleAdsUpdate({ ads, pagination, totalActive }) {
  logger.log(`📦 Anuncios recibidos: ${ads?.length || 0}`);
  
  state.ads = ads || [];
  
  if (totalActive !== undefined && totalActive !== null) {
    state.totalActiveAds = totalActive;
  }
  
  if (pagination) {
    state.pagination = {
      current: parseInt(pagination.current) || 1,
      total: parseInt(pagination.total) || 1
    };
  }
  
  // Ocultar mensajes de estado y loader
  hideStatus();
  hideLoader();
  hideAllModals();
  showAdsUI();
  
  renderAds();
}

/**
 * Maneja resultado de renovación
 */
function handleRenewResult({ adId, success, error }) {
  logger.log(`🔄 Resultado renovación ${adId}: ${success ? '✅' : '❌'}`);
  
  if (state.renewCallback) {
    state.renewCallback({ adId, success, error });
  }
}

/**
 * Maneja desconexión del content script
 */
function handleDisconnect() {
  logger.log('❌ Desconectado de Milanuncios');
  
  // Limpiar UI de usuario
  updateUserUI(null);
  
  showStatus('🔌 Conexión perdida', 'warning', 
    'Recarga la página de Milanuncios');
  hideAdsUI();
  
  // Verificar si hay otras pestañas
  checkExistingTabs();
}

/**
 * Maneja cambios en chrome.storage
 */
function handleStorageChange(changes, areaName) {
  if (areaName !== 'local') return;
  
  // Actualizar stats si cambiaron
  if (changes['mil.stats']) {
    loadStats();
  }
}

/**
 * Verifica si hay pestañas de Milanuncios existentes
 * Muestra modal si no hay conexión después de 3 segundos
 */
async function checkExistingTabs() {
  // Esperar 3 segundos para que connection.js intente conectar
  setTimeout(async () => {
    if (!isConnected()) {
      const tabs = await chrome.tabs.query({ url: '*://*.milanuncios.com/*' });
      logger.log('Sin conexión después de 3s, tabs:', tabs.length);
      
      // Siempre mostrar modal cuando no hay conexión
      showTabModal(tabs);
    }
  }, 3000);
}

/**
 * Actualiza la UI con info del usuario de Milanuncios
 */
function updateUserUI(user) {
  const milUserContainer = document.getElementById('mil-header-mil-user');
  const avatarEl = document.getElementById('mil-user-avatar');
  const usernameEl = document.getElementById('mil-username');
  
  if (user && user.isLogged) {
    // Mostrar sección de usuario Milanuncios
    if (milUserContainer) milUserContainer.style.display = 'flex';
    
    // Avatar
    if (avatarEl) {
      if (user.avatar) {
        // Añadir timestamp para evitar caché
        const avatarUrl = user.avatar.includes('?') 
          ? `${user.avatar}&_t=${Date.now()}` 
          : `${user.avatar}?_t=${Date.now()}`;
        avatarEl.src = avatarUrl;
        avatarEl.style.display = 'block';
      } else {
        avatarEl.style.display = 'none';
        avatarEl.src = '';
      }
    }
    
    // Username/alias de Milanuncios
    if (usernameEl) {
      usernameEl.textContent = user.username || 'Usuario';
    }
  } else {
    // Ocultar sección de usuario Milanuncios
    if (milUserContainer) milUserContainer.style.display = 'none';
    if (avatarEl) {
      avatarEl.style.display = 'none';
      avatarEl.src = '';
    }
    if (usernameEl) {
      usernameEl.textContent = '';
    }
  }
}

/**
 * Muestra la UI de anuncios
 */
function showAdsUI() {
  const container = document.getElementById('mil-ads-container');
  const toolbar = document.getElementById('mil-toolbar-container');
  if (container) container.style.display = 'block';
  if (toolbar) toolbar.style.display = 'block';
}

/**
 * Oculta la UI de anuncios
 */
function hideAdsUI() {
  const container = document.getElementById('mil-ads-container');
  const toolbar = document.getElementById('mil-toolbar-container');
  if (container) container.style.display = 'none';
  if (toolbar) toolbar.style.display = 'none';
}

// ==================== CONNECTION (Legacy - Compatibilidad) ====================

/**
 * Verifica estado de página - wrapper para checkExistingTabs
 */
async function checkPageStatus() {
  logger.log('Verificando estado de página...');
  checkExistingTabs();
}

/**
 * Inyecta content script si es necesario
 */
async function injectContentScriptIfNeeded(tabId) {
  try {
    await chrome.scripting.executeScript({
      target: { tabId },
      files: ['milanuncios/content-script.js']
    });
    logger.log('Content script inyectado');
  } catch (e) {
    logger.warn('No se pudo inyectar:', e.message);
  }
}

function showTabModal(existingTabs) {
  const modal = document.getElementById('mil-modal-open-tab');
  if (!modal) return;
  
  const titleEl = document.getElementById('mil-modal-tab-title');
  const messageEl = document.getElementById('mil-modal-tab-message');
  const hintEl = document.getElementById('mil-modal-tab-hint');
  const gotoBtn = document.getElementById('mil-modal-goto-tab');
  const openNewBtn = document.getElementById('mil-modal-open-new-tab');
  
  if (existingTabs && existingTabs.length > 0) {
    // Hay pestañas pero no conectadas
    if (titleEl) titleEl.textContent = 'Conexión perdida';
    if (messageEl) messageEl.textContent = 'Tienes Milanuncios abierto pero no se pudo conectar.';
    if (hintEl) hintEl.textContent = 'Recarga la página de Milanuncios o abre una nueva.';
    if (gotoBtn) {
      gotoBtn.style.display = 'block';
      gotoBtn.textContent = '🔄 Ir a Milanuncios y recargar';
      gotoBtn.className = 'btn btn-primary btn-lg';
    }
    if (openNewBtn) {
      openNewBtn.style.display = 'block';
      openNewBtn.className = 'btn btn-secondary';
    }
    setExistingMilTab(existingTabs[0]);
  } else {
    // No hay pestañas - ofrecer abrir nueva
    if (titleEl) titleEl.textContent = 'Abre Milanuncios';
    if (messageEl) messageEl.textContent = 'Para gestionar tus anuncios, necesitas tener Milanuncios abierto.';
    if (hintEl) hintEl.textContent = '¿Quieres abrir Milanuncios ahora?';
    if (gotoBtn) gotoBtn.style.display = 'none';
    if (openNewBtn) {
      openNewBtn.style.display = 'block';
      openNewBtn.textContent = '🚀 Abrir Milanuncios';
      openNewBtn.className = 'btn btn-primary btn-lg';
    }
    setExistingMilTab(null);
  }
  
  modal.style.display = 'flex';
  logger.log('Modal mostrado:', existingTabs?.length > 0 ? 'con pestañas' : 'sin pestañas');
}

/**
 * Verifica que Milanuncios esté activo
 */
async function requireMilanunciosActive() {
  logger.log('🔍 Verificando Milanuncios...');
  
  if (isReady()) {
    return { ok: true, tabId: state.currentTabId };
  }
  
  const connState = getConnectionState();
  
  if (connState === ConnectionState.DISCONNECTED) {
    return { ok: false, reason: 'NO_TAB' };
  }
  
  if (connState === ConnectionState.CONNECTED_NO_LOGIN) {
    return { ok: false, reason: 'NO_LOGIN' };
  }
  
  if (connState === ConnectionState.CONNECTED_WRONG_PAGE) {
    return { ok: false, reason: 'WRONG_PAGE' };
  }
  
  return { ok: false, reason: 'UNKNOWN' };
}

function showTabRequiredModal(existingTabs) {
  return new Promise((resolve) => {
    hideAllModals();
    
    const modal = document.getElementById('mil-modal-open-tab');
    if (!modal) {
      resolve({ ok: false, reason: 'no_modal' });
      return;
    }
    
    const titleEl = document.getElementById('mil-modal-tab-title');
    const messageEl = document.getElementById('mil-modal-tab-message');
    const gotoBtn = document.getElementById('mil-modal-goto-tab');
    
    if (existingTabs && existingTabs.length > 0) {
      if (titleEl) titleEl.textContent = 'Milanuncios detectado';
      if (messageEl) messageEl.textContent = 'Tienes Milanuncios abierto pero no se pudo conectar.';
      if (gotoBtn) {
        gotoBtn.style.display = 'block';
        gotoBtn.textContent = '🔄 Cambiar a Milanuncios';
      }
      setExistingMilTab(existingTabs[0]);
    } else {
      if (titleEl) titleEl.textContent = 'Milanuncios no detectado';
      if (messageEl) messageEl.textContent = 'Para gestionar anuncios, necesitas tener Milanuncios abierto.';
      if (gotoBtn) gotoBtn.style.display = 'none';
      setExistingMilTab(null);
    }
    
    modal.style.display = 'flex';
    
    // Handlers
    const openNewBtn = document.getElementById('mil-modal-open-new-tab');
    const verifyBtn = document.getElementById('mil-modal-verify');
    const cancelBtn = document.getElementById('mil-modal-cancel');
    
    const cleanup = () => {
      hideAllModals();
    };
    
    openNewBtn?.addEventListener('click', async () => {
      cleanup();
      showToast('Abriendo Milanuncios...', 'info');
      await chrome.tabs.create({ 
        url: 'https://www.milanuncios.com/mis-anuncios',
        active: true
      });
      await delay(3000);
      const result = await requireMilanunciosActive();
      resolve(result);
    }, { once: true });
    
    gotoBtn?.addEventListener('click', async () => {
      const existingTab = getExistingMilTab();
      if (existingTab) {
        cleanup();
        await chrome.tabs.update(existingTab.id, { active: true });
        await chrome.windows.update(existingTab.windowId, { focused: true });
        await delay(1000);
        const result = await requireMilanunciosActive();
        resolve(result);
      }
    }, { once: true });
    
    verifyBtn?.addEventListener('click', async () => {
      cleanup();
      showToast('Verificando...', 'info');
      await delay(500);
      const result = await requireMilanunciosActive();
      resolve(result);
    }, { once: true });
    
    cancelBtn?.addEventListener('click', () => {
      cleanup();
      resolve({ ok: false, reason: 'cancelled' });
    }, { once: true });
  });
}

function showLoginRequiredModal(tab) {
  return new Promise((resolve) => {
    hideAllModals();
    
    const modal = document.getElementById('mil-modal-login');
    if (!modal) {
      resolve({ ok: false, reason: 'no_modal' });
      return;
    }
    
    modal.style.display = 'flex';
    
    const gotoBtn = document.getElementById('mil-modal-login-goto');
    const verifyBtn = document.getElementById('mil-modal-login-verify');
    const cancelBtn = document.getElementById('mil-modal-login-cancel');
    
    gotoBtn?.addEventListener('click', async () => {
      hideAllModals();
      await chrome.tabs.update(tab.id, { active: true });
      await chrome.windows.update(tab.windowId, { focused: true });
      showToast('Inicia sesión y pulsa Verificar', 'info', 5000);
    }, { once: true });
    
    verifyBtn?.addEventListener('click', async () => {
      hideAllModals();
      showToast('Verificando login...', 'info');
      await delay(500);
      const result = await requireMilanunciosActive();
      resolve(result);
    }, { once: true });
    
    cancelBtn?.addEventListener('click', () => {
      hideAllModals();
      resolve({ ok: false, reason: 'cancelled' });
    }, { once: true });
  });
}

// ==================== ADS ====================

/**
 * Solicita escaneo de anuncios al content script
 * Si no hay conexión, intenta reconectar automáticamente
 */
async function scanAds() {
  // Si no hay conexión, intentar reconectar primero
  if (!isConnected()) {
    logger.log('Sin conexión, intentando reconectar...');
    showLoader('Reconectando...');
    
    const reconnected = await forceReconnect();
    
    if (!reconnected) {
      hideLoader();
      showStatus('❌ Sin conexión', 'error', 'Abre una pestaña de Milanuncios');
      const tabs = await chrome.tabs.query({ url: '*://*.milanuncios.com/*' });
      if (tabs.length === 0) {
        showTabModal([]);
      }
      return;
    }
    
    // Esperar un momento para que se actualice el estado
    await delay(500);
  }
  
  logger.log('Solicitando escaneo de anuncios...');
  showLoader('Cargando anuncios...');
  
  // Enviar comando al CS - la respuesta llegará por handleAdsUpdate
  const sent = requestScanAds();
  
  if (!sent) {
    hideLoader();
    showStatus('❌ Error de conexión', 'error', 'Recarga la página de Milanuncios');
  }
  
  // Timeout de seguridad
  setTimeout(() => {
    if (state.ads.length === 0) {
      hideLoader();
    }
  }, 5000);
}

/**
 * Versión síncrona de scanAds (espera la respuesta)
 */
async function scanAdsSync() {
  return new Promise((resolve) => {
    if (!isConnected()) {
      resolve(false);
      return;
    }
    
    let resolved = false;
    
    // Temporalmente guardar el callback
    const originalHandler = state.adsUpdateCallback;
    
    state.adsUpdateCallback = (data) => {
      if (!resolved) {
        resolved = true;
        state.adsUpdateCallback = originalHandler;
        resolve(true);
      }
    };
    
    const sent = requestScanAds();
    if (!sent) {
      state.adsUpdateCallback = originalHandler;
      resolve(false);
      return;
    }
    
    // Timeout
    setTimeout(() => {
      if (!resolved) {
        resolved = true;
        state.adsUpdateCallback = originalHandler;
        resolve(false);
      }
    }, 5000);
  });
}

function renderAds() {
  const container = document.getElementById('mil-ads-container');
  const toolbar = document.getElementById('mil-toolbar-container');
  const grid = document.getElementById('mil-ads-grid');
  const countBadge = document.getElementById('mil-ads-count');
  const totalCount = document.getElementById('mil-total-count');
  
  if (!container || !grid) return;
  
  container.style.display = 'block';
  if (toolbar) toolbar.style.display = 'block';
  
  if (countBadge) countBadge.textContent = `${state.ads.length} en esta página`;
  if (totalCount) totalCount.textContent = state.totalActiveAds || state.ads.length;
  
  renderPagination();
  grid.innerHTML = '';
  
  state.ads.forEach((ad, index) => {
    const card = document.createElement('div');
    card.className = 'ad-card';
    card.dataset.adId = ad.id;
    
    const wasRenovated = state.renovatedIds.has(ad.id);
    const thumbnail = ad.image 
      ? `<img class="ad-thumbnail" src="${ad.image}" alt="" loading="lazy" />`
      : `<div class="ad-thumbnail ad-thumbnail--empty">📷</div>`;
    
    const canTakeAction = ad.canRenew && !wasRenovated;
    const footerHtml = canTakeAction 
      ? `<div class="ad-footer">
          <button class="btn btn-renew btn-primary" data-ad-id="${ad.id}">🔄 Renovar</button>
        </div>`
      : '';
    
    card.innerHTML = `
      <div class="ad-header">
        <span class="ad-number">#${index + 1}</span>
        <span class="ad-status-badge ${wasRenovated ? 'cooldown' : (ad.canRenew ? 'can-renew' : 'cooldown')}">
          ${wasRenovated ? '✅ Renovado' : (ad.canRenew ? '🟢 Disponible' : '⏳ Cooldown')}
        </span>
      </div>
      <div class="ad-body">
        <div class="ad-content">
          ${thumbnail}
          <div class="ad-info">
            <h4 class="ad-title">${ad.title}</h4>
            <div class="ad-meta">
              <span class="ad-price">${ad.price}</span>
              <span class="ad-id">ID: ${ad.id}</span>
            </div>
          </div>
        </div>
      </div>
      ${footerHtml}
    `;
    
    grid.appendChild(card);
  });
  
  document.querySelectorAll('.btn-renew').forEach(btn => {
    btn.addEventListener('click', async () => {
      await renewSingle(btn.dataset.adId);
    });
  });
  
  renderStats();
}

function renderPagination() {
  const container = document.getElementById('mil-pagination');
  const numbersDiv = document.getElementById('mil-page-numbers');
  
  if (!container || !numbersDiv) return;
  
  const current = parseInt(state.pagination.current) || 1;
  const total = parseInt(state.pagination.total) || 1;
  
  if (total <= 1) {
    container.style.display = 'none';
    return;
  }
  
  container.style.display = 'flex';
  numbersDiv.innerHTML = '';
  
  const pages = generatePageNumbers(current, total);
  
  pages.forEach(page => {
    if (page === '...') {
      const ellipsis = document.createElement('span');
      ellipsis.className = 'mil-pagination__ellipsis';
      ellipsis.textContent = '...';
      numbersDiv.appendChild(ellipsis);
    } else {
      const span = document.createElement('span');
      span.className = `mil-pagination__num ${page === current ? 'mil-pagination__num--active' : ''}`;
      span.textContent = page;
      numbersDiv.appendChild(span);
    }
  });
}

function generatePageNumbers(current, total) {
  const pages = [];
  
  if (total <= 7) {
    for (let i = 1; i <= total; i++) pages.push(i);
  } else if (current <= 4) {
    for (let i = 1; i <= 4; i++) pages.push(i);
    pages.push('...', total);
  } else if (current >= total - 3) {
    pages.push(1, '...');
    for (let i = total - 3; i <= total; i++) pages.push(i);
  } else {
    pages.push(1, '...', current - 1, current, current + 1, '...', total);
  }
  
  return pages;
}

async function navigateToPage(page) {
  if (!state.currentTabId) return;
  
  return new Promise((resolve) => {
    chrome.tabs.sendMessage(state.currentTabId, { action: 'mil_navigateToPage', page }, (response) => {
      resolve(response?.success || false);
    });
  });
}

// ==================== RENEWAL ====================

async function renewSingle(adId) {
  if (!state.currentTabId || state.isRenewing) return;
  
  logger.log('Renovando:', adId);
  state.isRenewing = true;
  
  const btn = document.querySelector(`[data-ad-id="${adId}"]`);
  if (btn) {
    btn.disabled = true;
    btn.textContent = '⏳ Renovando...';
  }
  
  chrome.tabs.sendMessage(state.currentTabId, { action: 'mil_renewAd', adId }, async (response) => {
    state.isRenewing = false;
    
    if (chrome.runtime.lastError) {
      if (state.config.notifications) showToast('Error de conexión', 'error');
      if (btn) {
        btn.disabled = false;
        btn.textContent = '🔄 Renovar';
      }
      return;
    }
    
    if (response?.success) {
      state.renovatedIds.add(adId);
      await deductCredit();
      if (state.config.notifications) showToast('Anuncio renovado', 'success');
      if (btn) {
        btn.textContent = '✅ Renovado';
        btn.classList.add('btn-disabled');
      }
    } else {
      const errorMsg = response?.error || 'Error desconocido';
      if (errorMsg.includes('422') || errorMsg.includes('renovado')) {
        state.renovatedIds.add(adId);
        if (btn) {
          btn.textContent = '✅ Ya renovado';
          btn.classList.add('btn-disabled');
        }
      } else {
        if (state.config.notifications) showToast(errorMsg, 'error');
        if (btn) {
          btn.disabled = false;
          btn.textContent = '🔄 Renovar';
        }
      }
    }
  });
}

// ==================== ANTI-ABUSO: VERIFICACIÓN DE CUENTA ====================

/**
 * Verifica/registra la cuenta de Milanuncios en el backend
 * Si la cuenta ya está asociada a otro usuario, muestra error y bloquea
 * 
 * @returns {Promise<{ok: boolean, error?: string}>}
 */
async function verifyMilanunciosAccount() {
  const milUserId = state.currentMilUserId;
  const milUsername = state.currentMilUsername;
  
  if (!milUserId) {
    logger.error('No hay mil_user_id para verificar');
    return { ok: false, error: 'No se pudo obtener tu ID de Milanuncios' };
  }
  
  if (!state.jwt) {
    logger.warn('No hay JWT disponible para verificar cuenta');
    return { ok: false, error: 'No hay sesión activa' };
  }
  
  try {
    logger.log(`Verificando cuenta Milanuncios: ${milUserId} (${milUsername})`);
    
    const response = await fetch(
      `${API_BASE_URL}/milanuncios/accounts?mil_user_id=${encodeURIComponent(milUserId)}&alias=${encodeURIComponent(milUsername || 'Usuario')}`,
      {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${state.jwt}`,
          'Content-Type': 'application/json'
        }
      }
    );
    
    if (response.status === 409) {
      // Cuenta ya asociada a otro usuario
      logger.warn('⚠️ Cuenta de Milanuncios ya asociada a otro usuario');
      return { 
        ok: false, 
        error: 'duplicate',
        message: 'Esta cuenta de Milanuncios ya está vinculada a otro usuario. Si eres el propietario legítimo y perdiste acceso a tu cuenta anterior, contacta con soporte.'
      };
    }
    
    if (!response.ok) {
      const errorText = await response.text();
      logger.error('Error verificando cuenta:', response.status, errorText);
      return { ok: false, error: `Error del servidor: ${response.status}` };
    }
    
    const data = await response.json();
    logger.log('✅ Cuenta verificada:', data);
    
    return { ok: true, data };
    
  } catch (e) {
    logger.error('Error en verifyMilanunciosAccount:', e);
    return { ok: false, error: e.message };
  }
}

async function startMassiveRenewal() {
  if (state.isRenewing) {
    showToast('Ya hay una renovación en curso', 'warning');
    return;
  }
  
  if (state.ads.length === 0) {
    showToast('No hay anuncios para renovar', 'warning');
    return;
  }
  
  const totalPages = parseInt(state.pagination.total) || 1;
  
  const confirmed = await showConfirm({
    title: 'Renovación masiva',
    icon: '🚀',
    message: '¿Empezar renovación masiva?',
    details: `
      <p>📄 <strong>${totalPages}</strong> páginas</p>
      <p>⏱️ Delay: <strong>${state.config.delay / 1000}s</strong></p>
      <p>💰 Créditos: <strong>${state.credits}</strong></p>
    `,
    confirmText: '🚀 Empezar',
    cancelText: 'Cancelar'
  });
  
  if (!confirmed) return;
  
  // =====================================================================
  // 🔒 ANTI-ABUSO: Verificar cuenta de Milanuncios antes de renovar
  // =====================================================================
  showLoader('Verificando cuenta...');
  const accountCheck = await verifyMilanunciosAccount();
  hideLoader();
  
  if (!accountCheck.ok) {
    if (accountCheck.error === 'duplicate') {
      // Cuenta duplicada - mostrar modal y bloquear (igual que Wallapop)
      await showModal({
        title: '⚠️ Cuenta ya vinculada',
        html: `
          <p>Esta cuenta de Milanuncios ya está vinculada a otro usuario.</p>
          <p style="margin-top: 12px; color: var(--text-secondary, #aaa);">
            Si eres el propietario legítimo y perdiste acceso a tu cuenta anterior, contacta con soporte.
          </p>
          <div style="margin-top: 16px; display: flex; gap: 12px; justify-content: center; flex-wrap: wrap;">
            <a href="https://wa.me/34618620664" target="_blank" style="color: #25D366; text-decoration: none; font-weight: 500;">
              💬 WhatsApp
            </a>
            <a href="mailto:soporte@mitiklive.com" target="_blank" style="color: #FF6B35; text-decoration: none; font-weight: 500;">
              📧 Email
            </a>
          </div>
        `,
        buttons: [{ text: 'Entendido', value: 'ok', primary: true }]
      });
      return;
    } else {
      // Otro error
      showToast(accountCheck.error || 'Error verificando cuenta', 'error');
      return;
    }
  }
  
  logger.log('🚀 Iniciando renovación masiva');
  
  state.isRenewing = true;
  state.stopRequested = false;
  state.renovatedIds.clear();
  
  const startBtn = document.getElementById('mil-btn-start-renewal');
  const stopBtn = document.getElementById('mil-btn-stop-renewal');
  
  if (startBtn) startBtn.style.display = 'none';
  if (stopBtn) stopBtn.style.display = 'inline-flex';
  
  let totalRenovated = 0, totalSkipped = 0, totalErrors = 0;
  let consecutiveEmptyPages = 0;
  const MAX_EMPTY_PAGES = 3;
  
  while (!state.stopRequested) {
    const currentTotalPages = parseInt(state.pagination.total) || 1;
    const targetPage = Math.max(1, currentTotalPages - consecutiveEmptyPages);
    
    logger.log(`Página ${targetPage}/${currentTotalPages}`);
    
    if (stopBtn) {
      const pageLabel = ['última', 'penúltima', 'antepenúltima'][consecutiveEmptyPages] || 'página';
      stopBtn.innerHTML = `⏹️ Parar <span class="mil-btn__progress">(${pageLabel})</span>`;
    }
    
    await navigateToPage(targetPage);
    await delay(2500);
    await scanAdsSync();
    renderAds();
    
    const adsToProcess = state.ads.filter(ad => !state.renovatedIds.has(ad.id));
    
    if (adsToProcess.length === 0) {
      consecutiveEmptyPages++;
      if (consecutiveEmptyPages >= MAX_EMPTY_PAGES) break;
      continue;
    }
    
    let pageRenovated = 0;
    
    for (const ad of adsToProcess) {
      if (state.stopRequested) break;
      
      setCardRenewing(ad.id, true);
      const result = await renewAdWithRetry(ad.id);
      setCardRenewing(ad.id, false);
      
      if (result === true) {
        pageRenovated++;
        totalRenovated++;
        state.renovatedIds.add(ad.id);
        updateAdCardStatus(ad.id, 'renovated');
        await updateStats('renovated');
      } else if (result === 'cooldown') {
        totalSkipped++;
        state.renovatedIds.add(ad.id);
        updateAdCardStatus(ad.id, 'cooldown');
        await updateStats('cooldown');
      } else {
        totalErrors++;
        await updateStats('error');
      }
      
      await randomDelay(state.config.delay);
    }
    
    if (state.stopRequested) break;
    
    if (pageRenovated > 0) {
      consecutiveEmptyPages = 0;
      await delay(1500);
    } else {
      consecutiveEmptyPages++;
      if (consecutiveEmptyPages >= MAX_EMPTY_PAGES) break;
    }
  }
  
  // Finalización
  state.isRenewing = false;
  state.stopRequested = false;
  
  if (startBtn) startBtn.style.display = 'inline-flex';
  if (stopBtn) stopBtn.style.display = 'none';
  
  playCompletionSound();
  
  await showAlert({
    title: 'Renovación completada',
    icon: '✅',
    message: 'Proceso finalizado',
    details: `
      <p>✅ Renovados: <strong>${totalRenovated}</strong></p>
      <p>⏳ Cooldown: <strong>${totalSkipped}</strong></p>
      <p>❌ Errores: <strong>${totalErrors}</strong></p>
    `
  });
  
  await loadCredits();
  await navigateToPage(1);
  await delay(2000);
  await scanAds();
}

function stopMassiveRenewal() {
  state.stopRequested = true;
  showToast('Deteniendo...', 'info');
  hideCountdown(); // Ocultar cuenta atrás
  
  const stopBtn = document.getElementById('mil-btn-stop-renewal');
  if (stopBtn) {
    stopBtn.disabled = true;
    stopBtn.textContent = '⏳ Deteniendo...';
  }
}

async function renewAdWithRetry(adId, retries = 0) {
  return new Promise((resolve) => {
    chrome.tabs.sendMessage(state.currentTabId, { action: 'mil_renewAd', adId }, async (response) => {
      if (chrome.runtime.lastError) {
        if (retries < MAX_RETRIES) {
          await delay(1000);
          resolve(await renewAdWithRetry(adId, retries + 1));
        } else {
          resolve(false);
        }
        return;
      }
      
      if (response?.success) {
        await deductCredit();
        resolve(true);
      } else if (response?.error?.includes('422') || response?.error?.includes('renovado')) {
        resolve('cooldown');
      } else if (retries < MAX_RETRIES) {
        await delay(1000);
        resolve(await renewAdWithRetry(adId, retries + 1));
      } else {
        resolve(false);
      }
    });
  });
}

function updateAdCardStatus(adId, status) {
  const card = document.querySelector(`[data-ad-id="${adId}"]`);
  if (!card) return;
  
  const cardContainer = card.closest('.ad-card') || card;
  const badge = cardContainer.querySelector('.ad-status-badge');
  const btn = cardContainer.querySelector('.btn-renew');
  
  cardContainer.classList.remove('ad-card--renewing', 'ad-card--renovated', 'ad-card--cooldown');
  
  if (status === 'renovated') {
    cardContainer.classList.add('ad-card--renovated');
    if (badge) {
      badge.className = 'ad-status-badge renovated';
      badge.textContent = '✅ Renovado';
    }
    if (btn) btn.remove();
  } else if (status === 'cooldown') {
    cardContainer.classList.add('ad-card--cooldown');
    if (badge) {
      badge.className = 'ad-status-badge cooldown';
      badge.textContent = '⏳ Cooldown';
    }
    if (btn) btn.remove();
  }
}

function setCardRenewing(adId, isRenewing) {
  const card = document.querySelector(`[data-ad-id="${adId}"]`);
  if (!card) return;
  
  const cardContainer = card.closest('.ad-card') || card;
  
  if (isRenewing) {
    cardContainer.classList.add('ad-card--renewing');
  } else {
    cardContainer.classList.remove('ad-card--renewing');
  }
}

// ==================== EXPORT ====================

async function exportAllAdsToCSV() {
  const result = await requireMilanunciosActive();
  if (!result.ok) {
    showToast('No se pudo obtener usuario', 'error', 5000);
    return;
  }
  
  showExportModal();
  
  const allAds = [];
  const totalPages = parseInt(state.pagination.total) || 1;
  let cancelled = false;
  
  const cancelBtn = document.getElementById('mil-export-cancel');
  cancelBtn?.addEventListener('click', () => {
    cancelled = true;
    showToast('Exportación cancelada', 'warning');
  }, { once: true });
  
  try {
    for (let page = 1; page <= totalPages && !cancelled; page++) {
      updateExportProgress(page, totalPages, `Página ${page} de ${totalPages}...`);
      
      await navigateToPage(page);
      await delay(2000);
      
      const result = await new Promise(resolve => {
        chrome.tabs.sendMessage(state.currentTabId, { action: 'mil_scrapeAds' }, resolve);
      });
      
      if (result?.ads) {
        allAds.push(...result.ads);
      }
    }
    
    if (!cancelled && allAds.length > 0) {
      const csv = generateCSV(allAds);
      const blob = new Blob([csv], { type: 'text/csv;charset=utf-8;' });
      const url = URL.createObjectURL(blob);
      const link = document.createElement('a');
      link.href = url;
      link.download = `milanuncios_${new Date().toISOString().split('T')[0]}.csv`;
      link.click();
      
      showToast(`✅ ${allAds.length} anuncios exportados`, 'success', 5000);
    }
  } catch (e) {
    showToast('Error durante exportación', 'error');
  }
  
  hideExportModal();
}

function generateCSV(ads) {
  const headers = ['ID', 'Título', 'Precio', 'Puede Renovar', 'Imagen'];
  const rows = ads.map(ad => [
    ad.id,
    `"${(ad.title || '').replace(/"/g, '""')}"`,
    `"${(ad.price || '').replace(/"/g, '""')}"`,
    ad.canRenew ? 'Sí' : 'No',
    ad.image || ''
  ]);
  
  return [headers.join(','), ...rows.map(r => r.join(','))].join('\n');
}

// ==================== LISTENERS ====================

function setupListeners() {
  // Tabs (excluir el botón de logout que tiene clase especial)
  document.querySelectorAll('.tab-button:not(.mil-tab-special)').forEach(btn => {
    btn.addEventListener('click', () => switchTab(btn.dataset.tab));
  });
  
  // Refresh
  document.getElementById('mil-btn-refresh')?.addEventListener('click', scanAds);
  
  // Renovación
  document.getElementById('mil-btn-start-renewal')?.addEventListener('click', startMassiveRenewal);
  document.getElementById('mil-btn-stop-renewal')?.addEventListener('click', stopMassiveRenewal);
  
  // Export
  document.getElementById('mil-btn-export')?.addEventListener('click', exportAllAdsToCSV);
  
  // Config
  document.getElementById('mil-config-auto-refresh')?.addEventListener('change', (e) => {
    state.config.autoRefresh = e.target.checked;
    if (e.target.checked) {
      startAutoRefresh(scanAds);
    } else {
      stopAutoRefresh();
    }
    saveConfig();
  });
  
  document.getElementById('mil-config-delay')?.addEventListener('input', (e) => {
    const seconds = parseInt(e.target.value) || 3;
    state.config.delay = seconds * 1000;
    const valueEl = document.getElementById('mil-delay-value');
    if (valueEl) valueEl.textContent = seconds + 's';
    saveConfig();
  });
  
  document.getElementById('mil-config-notifications')?.addEventListener('change', (e) => {
    state.config.notifications = e.target.checked;
    saveConfig();
  });
  
  document.getElementById('mil-config-sound')?.addEventListener('change', (e) => {
    state.config.sound = e.target.checked;
    saveConfig();
  });
  
  document.getElementById('mil-config-zoom')?.addEventListener('input', (e) => {
    const zoom = parseInt(e.target.value) || 100;
    state.config.zoom = zoom;
    applyZoom(zoom);
    const valueEl = document.getElementById('mil-zoom-value');
    if (valueEl) valueEl.textContent = zoom + '%';
    saveConfig();
  });
  
  // Modal listeners
  document.getElementById('mil-modal-open-new-tab')?.addEventListener('click', async () => {
    hideAllModals();
    showToast('Abriendo Milanuncios...', 'info');
    await chrome.tabs.create({ url: 'https://www.milanuncios.com/mis-anuncios', active: true });
    await delay(3000);
    await checkPageStatus();
  });
  
  document.getElementById('mil-modal-goto-tab')?.addEventListener('click', async () => {
    const existingTab = getExistingMilTab();
    if (existingTab) {
      hideAllModals();
      await chrome.tabs.update(existingTab.id, { active: true });
      await chrome.windows.update(existingTab.windowId, { focused: true });
      showToast('Recarga la página (F5)', 'info', 4000);
    }
  });
  
  document.getElementById('mil-modal-verify')?.addEventListener('click', async () => {
    hideAllModals();
    showToast('Verificando...', 'info');
    await delay(500);
    await checkPageStatus();
  });
  
  document.getElementById('mil-modal-cancel')?.addEventListener('click', hideAllModals);
  
  // Login modal
  document.getElementById('mil-modal-login-goto')?.addEventListener('click', async () => {
    const existingTab = getExistingMilTab();
    if (existingTab) {
      await chrome.tabs.update(existingTab.id, { active: true });
      await chrome.windows.update(existingTab.windowId, { focused: true });
    }
    showToast('Inicia sesión y pulsa Verificar', 'info', 5000);
  });
  
  document.getElementById('mil-modal-login-verify')?.addEventListener('click', async () => {
    hideAllModals();
    showToast('Verificando login...', 'info');
    await delay(500);
    await checkPageStatus();
  });
  
  document.getElementById('mil-modal-login-cancel')?.addEventListener('click', hideAllModals);
  
  // Overlays
  document.querySelectorAll('.mil-modal__overlay').forEach(overlay => {
    overlay.addEventListener('click', hideAllModals);
  });
  
  // Refresh créditos
  document.getElementById('mil-btn-refresh-credits')?.addEventListener('click', async () => {
    showToast('Actualizando créditos...', 'info');
    await loadCredits();
    showToast('Créditos actualizados', 'success');
  });
  
  // Logout
  // Logout (ambos botones)
  document.getElementById('mil-btn-logout')?.addEventListener('click', handleLogout);
  document.getElementById('mil-btn-logout-tab')?.addEventListener('click', handleLogout);
  
  // Comprar créditos
  document.getElementById('mil-btn-buy-credits')?.addEventListener('click', () => {
    chrome.tabs.create({ url: 'https://mitiklive.com/wallapop/dashboard/billing' });
  });
  
  // Manual
  document.getElementById('mil-btn-manual')?.addEventListener('click', () => {
    chrome.tabs.create({ url: 'https://mitiklive.com/wallapop/manual' });
  });
}

async function handleLogout() {
  const confirmed = await showConfirm({
    title: 'Cerrar sesión',
    message: '¿Seguro que quieres cerrar sesión?',
    icon: '🚪',
    confirmText: 'Cerrar sesión',
    cancelText: 'Cancelar',
    dangerous: true
  });
  
  if (confirmed) {
    try {
      // 1. Enviar SESSION.LOGOUT al SW (limpia session.*, revoca dispositivo)
      await chrome.runtime.sendMessage({ type: 'SESSION.LOGOUT' });
      
      // 2. Limpiar keys adicionales locales
      await chrome.storage.local.remove([
        'jwt', 'user', 'active_account',
        'milConfig', 'mil.renovatedAds', 'mil.lastRenewalTime'
      ]);
      
      // 3. Mantener plataforma seleccionada
      await chrome.storage.local.set({ selectedPlatform: 'milanuncios' });
      
      showToast('Sesión cerrada', 'info');
      
      // 4. Redirigir con delay
      setTimeout(() => {
        window.location.href = '../panel.html';
      }, 300);
      
    } catch (error) {
      logger.error('Error en logout:', error);
      showToast('Error al cerrar sesión', 'error');
    }
  }
}
