# 📱 GUÍA MAESTRA - MILANUNCIOS MODULE v1.3

**⚠️ ÚNICA FUENTE DE VERDAD - MÓDULO MILANUNCIOS**

**Fecha actualización:** 2025-12-05  
**Versión:** v2.8.0  
**Extensión base:** v5.8.11  

**Estado:** 🟢 ACTIVO - Aplicar en TODOS los cambios de Milanuncios

---

## 🤖 NOTA IMPORTANTE PARA USO CON CLAUDE

> **INSTRUCCIÓN CRÍTICA:**  
> Esta guía debe consultarse ANTES de hacer cualquier modificación al módulo de Milanuncios.
> 
> **RECORDATORIOS CLAVE:**
> - **Antes de modificar código:** "Lee la sección X de la guía de Milanuncios"
> - **Antes de añadir funciones:** "Verifica si existe código reutilizable en la guía"
> - **Si surgen dudas de arquitectura:** "Consulta las reglas de aislamiento de la guía"
>
> **NO ASUMAS** que las reglas de Wallapop aplican a Milanuncios. Cada plataforma tiene sus propias reglas.

---

## 📋 ÍNDICE

### 🏗️ [ARQUITECTURA](#arquitectura)
- [Principio de Aislamiento](#principio-de-aislamiento)
- [Estructura de Carpetas](#estructura-de-carpetas)
- [Componentes del Sistema](#componentes-del-sistema)
- [Sistema de Conexión](#2-sistema-de-conexión-scriptsconnectionjs--v582) ⭐ v5.8.2+
- [Header del Panel](#4-header-del-panel-v585) ⭐ v5.8.5+

### 🎨 [DISEÑO Y UI](#diseño-y-ui)
- [Sistema de Colores](#sistema-de-colores)
- [Componentes Reutilizables](#componentes-reutilizables)
- [Nomenclatura CSS](#nomenclatura-css)

### 💳 [SISTEMA DE CRÉDITOS](#sistema-de-créditos-v556)
- [Configuración](#configuración)
- [Flujo de Autenticación](#flujo-de-autenticación)
- [Endpoints Backend](#endpoints-backend)

### 🔧 [API MILANUNCIOS](#api-milanuncios)
- [Endpoint de Renovación](#endpoint-de-renovación)
- [Manejo de Errores](#manejo-de-errores)
- [Rate Limiting](#rate-limiting)
- [Algoritmo de Renovación Masiva](#algoritmo-de-renovación-masiva-v558)

### 📝 [REGLAS DE CÓDIGO](#reglas-de-código)
- [Content Script](#content-script)
- [Panel JavaScript](#panel-javascript)
- [Estilos CSS](#estilos-css)

### 🐛 [DEBUGGING](#debugging)
- [Console Logs](#console-logs)
- [Testing Manual](#testing-manual)
- [Errores Comunes](#errores-comunes)
- [Edge Cases Cubiertos](#edge-cases-cubiertos-v588) ⭐ v5.8.8+

---

## 🏗️ ARQUITECTURA

### Principio de Aislamiento

**REGLA CRÍTICA #1:** El código de Milanuncios debe estar 100% aislado de Wallapop.

```
✅ PERMITIDO:
- Reutilizar funciones de scripts/utils.js (OBLIGATORIO si existe)
- Usar mismo sistema de storage (keys diferentes)

❌ PROHIBIDO:
- Importar archivos específicos de Wallapop
- Modificar código de Wallapop
- Duplicar código que ya existe en scripts/utils.js
- Crear dependencias cruzadas
- Usar colores hardcodeados (SIEMPRE usar variables CSS)
```

**REGLA CRÍTICA #1.1:** ANTES de crear una función, verificar si ya existe en `scripts/utils.js`

### Estructura de Carpetas

```
extension/
├── manifest.json              # Permisos compartidos
├── panel.html                 # Login compartido + Router
├── panel.js                   # Router de plataformas
│
├── scripts/
│   └── utils.js              # ← FUNCIONES COMPARTIDAS (Wallapop + Milanuncios)
│
├── milanuncios/               # ← TODO MILANUNCIOS AQUÍ (AISLADO)
│   ├── content-script.js     # Content script MIL
│   ├── panel-milanuncios.html # UI del panel
│   ├── panel-milanuncios.js  # Lógica principal
│   ├── scripts/              # ← MÓDULOS (v5.8.5)
│   │   ├── state.js          # Estado global y constantes
│   │   ├── connection.js     # ⭐ Sistema de conexión con CS (v5.8.2+)
│   │   ├── config.js         # Configuración
│   │   ├── credits.js        # Sistema de créditos
│   │   ├── modals.js         # Sistema de modales unificado
│   │   ├── stats.js          # Estadísticas
│   │   └── ui.js             # Helpers de UI
│   └── styles/
│       └── milanuncios.css   # ← ÚNICO archivo CSS
│
└── ... (Wallapop sin tocar)
```

**REGLA #2:** Todos los archivos nuevos de Milanuncios van en `milanuncios/`

---

## Estilos CSS

### Archivo Único: milanuncios.css

**REGLA CRÍTICA CSS:** Milanuncios tiene UN SOLO archivo CSS: `milanuncios/styles/milanuncios.css`

```
milanuncios/styles/
└── milanuncios.css    # Variables + Componentes + Todo
```

### Variables CSS (OBLIGATORIO)

**REGLA #3:** SIEMPRE usar variables CSS, NUNCA colores hardcodeados.

```css
/* ✅ CORRECTO */
.mi-clase {
  background: var(--bg-primary);
  color: var(--text-primary);
  border: 1px solid var(--border-primary);
}

/* ❌ INCORRECTO */
.mi-clase {
  background: #0d1320;
  color: #e6eaf2;
  border: 1px solid #2d3548;
}
```

### Variables Disponibles

```css
/* Brand Colors - Milanuncios */
--mil-primary: #FF6B35;      /* Naranja principal */
--mil-primary-dark: #E55A2B;
--mil-success: #27AE60;      /* Verde éxito */
--mil-warning: #F39C12;      /* Amarillo warning */
--mil-error: #E74C3C;        /* Rojo error */
--mil-info: #3498DB;         /* Azul info */

/* Background */
--bg-primary: #0d1320;       /* Fondo principal */
--bg-secondary: #1a1f2e;     /* Fondo secundario */
--bg-tertiary: #252b3d;      /* Fondo elevado */

/* Text */
--text-primary: #e6eaf2;     /* Texto principal */
--text-secondary: #a0a8b8;   /* Texto secundario */
--text-muted: #6b7280;       /* Texto apagado */

/* Borders */
--border-primary: #2d3548;
--border-secondary: #3d4558;
```

### Nomenclatura BEM

**REGLA #4:** Usar nomenclatura BEM con prefijo `mil-`

```css
/* Bloque */
.mil-card { }

/* Elemento */
.mil-card__header { }
.mil-card__body { }
.mil-card__footer { }

/* Modificador */
.mil-card--active { }
.mil-card--disabled { }
```

### Componentes Base Disponibles

```css
/* Botones */
.btn                    /* Base */
.btn-primary           /* Naranja Milanuncios */
.btn-secondary         /* Gris */
.btn-success           /* Verde */
.btn-danger            /* Rojo */
.btn-ghost             /* Transparente */
.btn-sm, .btn-lg       /* Tamaños */
.btn-block             /* Full width */

/* Inputs */
.inp                   /* Input base con variables */

/* Alertas */
.ml-alert              /* Base */
.ml-alert.info         /* Azul */
.ml-alert.warning      /* Amarillo */
.ml-alert.error        /* Rojo */
.ml-alert.success      /* Verde */

/* Toast */
.ml-toast              /* Notificación flotante */

/* Modales */
.mil-modal             /* Base modal */
.mil-modal__overlay    /* Fondo oscuro */
.mil-modal__content    /* Contenido */
.mil-modal__header     /* Cabecera */
.mil-modal__body       /* Cuerpo */
.mil-modal__footer     /* Botones */
```

### Checklist CSS

Antes de añadir CSS nuevo:

- [ ] ¿Usé variables CSS en vez de colores hardcodeados?
- [ ] ¿Usé nomenclatura BEM con prefijo `mil-`?
- [ ] ¿Verifiqué que no existe un componente similar?
- [ ] ¿Lo añadí a `milanuncios.css` (no crear archivos nuevos)?

---

## Código Compartido (scripts/utils.js)

**REGLA CRÍTICA #3:** SIEMPRE usar funciones de `scripts/utils.js` cuando existan.

> **Nota:** Antes existía `shared/utils.js` pero se eliminó. Ahora Milanuncios usa el mismo `scripts/utils.js` que Wallapop para evitar duplicación.

#### Funciones Disponibles en scripts/utils.js:

```javascript
// === Funciones originales de Wallapop ===
toast(msg, type, options)           // Toast nativo Wallapop
showModal({title, html, buttons})   // Modal con botones
showLoader(text, options)           // Loader con opciones avanzadas
hideLoader()
setLoaderText(text)
sleep(ms)                           // Delay
normalize(str)                      // Normaliza texto
debounce(func, wait)

// === Funciones añadidas para Milanuncios (v5.7.5) ===
showToast(message, type, duration)  // ✅ Alias de toast
hideToast()                         // ✅ Oculta toasts
delay(ms)                           // ✅ Alias de sleep
getActiveTab()                      // ✅ Obtiene tab activa
sendMessageToActiveTab(message)     // ✅ Envía mensaje a tab activa
getStorage(keys)                    // ✅ chrome.storage.local.get
setStorage(data)                    // ✅ chrome.storage.local.set
removeStorage(keys)                 // ✅ chrome.storage.local.remove
clearStorage()                      // ✅ chrome.storage.local.clear
createLogger(platform, module)      // ✅ Logger con prefijo
isValidId(id)                       // ✅ Valida ID numérico
isValidEmail(email)                 // ✅ Valida email
formatDate(date)                    // ✅ Formatea fecha
truncate(text, maxLength)           // ✅ Trunca texto
```

#### Ejemplo de Uso CORRECTO:

```javascript
// ✅ CORRECTO - Importar de scripts/utils.js
import { showToast, delay, createLogger } from '../scripts/utils.js';

const logger = createLogger('Milanuncios', 'Panel');

async function renovar() {
  logger.log('Renovando...');
  showToast('Renovando...', 'info');
  await delay(3000);
  showToast('Renovado', 'success');
}
```

#### Ejemplo de Uso INCORRECTO:

```javascript
// ❌ INCORRECTO - Duplicar código
function showToast(msg) {  // ← Ya existe en scripts/utils.js
  // ...código duplicado...
}

function delay(ms) {       // ← Ya existe en scripts/utils.js
  return new Promise(resolve => setTimeout(resolve, ms));
}

console.log('[Milanuncios]', ...); // ← Usar createLogger de scripts/utils.js
```

#### Estados de Cards (v5.5.0+)

```css
/* Card renovando (pulsante, muy visible) */
.ad-card--renewing {
  border: 4px solid var(--mil-primary);
  background: rgba(255, 107, 53, 0.15);
  animation: pulse-renewing 1s infinite;
}

/* Card renovada (permanente, verde) */
.ad-card--renovated {
  border: 3px solid var(--mil-success);
  background: rgba(39, 174, 96, 0.12);
}

/* Card en cooldown */
.ad-card--cooldown {
  border: 2px solid var(--mil-warning);
  background: rgba(243, 156, 18, 0.08);
  opacity: 0.85;
}
```

```javascript
// Uso en JS
setCardRenewing(adId, true);   // Marca como renovando
setCardRenewing(adId, false);  // Quita marca renovando
updateAdCardStatus(adId, 'renovated');  // Marca permanente renovado
updateAdCardStatus(adId, 'cooldown');   // Marca cooldown
```

#### Optimización UI de Cards (v5.6.3+)

**REGLA:** No mostrar botones deshabilitados redundantes. El badge ya indica el estado.

```javascript
// ✅ CORRECTO: Botón solo cuando hay acción disponible
const canTakeAction = ad.canRenew && !wasRenovated;
const footerHtml = canTakeAction 
  ? `<div class="ad-footer"><button class="btn btn-renew">🔄 Renovar</button></div>`
  : '';  // Sin footer si no hay acción

// ❌ INCORRECTO: Botón deshabilitado duplicando info del badge
<button class="btn-disabled" disabled>✅ Renovado</button>  // Redundante con badge
```

**Estructura visual:**
```
┌─────────────────────────────┐
│ #1              🟢 Disponible │  ← Badge indica estado
├─────────────────────────────┤
│ [img] Título del anuncio    │
│       10 € · ID: 123456     │
├─────────────────────────────┤
│     [ 🔄 Renovar ]          │  ← Solo si puede renovar
└─────────────────────────────┘

┌─────────────────────────────┐
│ #2              ⏳ Cooldown  │  ← Badge indica estado
├─────────────────────────────┤
│ [img] Otro anuncio          │
│       5 € · ID: 789012      │
└─────────────────────────────┘  ← Sin footer (no hay acción)
```

### Componentes del Sistema

#### 1. Content Script (`content-script.js`)

**Responsabilidades:**
- Detectar si está en "Mis anuncios"
- Detectar si usuario está logueado (múltiples métodos)
- Extraer IDs de anuncios del DOM
- Llamar a API de renovación
- Responder a mensajes del panel

**Acciones soportadas:**
```javascript
// Mensajes que el CS responde:
'mil_ping'          // Verificar conexión + estado
'mil_getStatus'     // Obtener estado actual
'mil_scrapeAds'     // Extraer anuncios
'mil_renewAd'       // Renovar un anuncio
'mil_navigateToPage' // Navegar a página
```

**Selectores DOM de Milanuncios (v5.8.8+):**
```javascript
// Avatar del usuario (NO el logo)
'.ma-UserAvatar img.sui-AtomImage-image'
// - El atributo `alt` contiene el nombre del usuario
// - El atributo `src` contiene la URL del avatar

// Detectar si está logueado
'.ma-UserAvatar'  // Si existe, usuario logueado

// Detectar página "Mis anuncios"
window.location.pathname.includes('/mis-anuncios')
```

**Archivos relacionados:**
- `milanuncios/content-script.js`

#### 2. Sistema de Conexión (`scripts/connection.js`) ⭐ v5.8.2+

**IMPORTANTE:** NO usa puertos (`chrome.runtime.onConnect`). Usa `chrome.tabs.sendMessage` + verificación periódica.

**Responsabilidades:**
- Detectar pestañas de Milanuncios abiertas
- Verificar estado del Content Script cada 2 segundos
- Detectar cambio de usuario (cuenta diferente)
- Inyectar CS si es necesario
- Notificar cambios de estado al panel

**Estados de conexión:**
```javascript
ConnectionState = {
  DISCONNECTED: 'disconnected',      // No hay pestaña/CS
  CONNECTED_NO_LOGIN: 'no_login',    // CS vivo, no logueado
  CONNECTED_WRONG_PAGE: 'wrong_page', // Logueado, no en mis-anuncios
  READY: 'ready',                    // Listo para operar
  RENEWING: 'renewing'               // Renovación en curso
}
```

**Funciones exportadas:**
```javascript
import {
  ConnectionState,
  getConnectionState,
  isReady,
  isConnected,
  setListeners,
  initConnectionListener,
  stopConnectionListener,
  requestScanAds,
  requestRenew,
  requestNavigate,
  requestStatus
} from './scripts/connection.js';
```

**Detección de cambio de cuenta:**
```javascript
// En state.js
currentMilUserId: null  // userId del usuario de Milanuncios

// connection.js compara userId en cada ping
// Si cambia → limpia anuncios → notifica → recarga
```

#### 3. Panel (`panel-milanuncios.html` + `.js`)

**Responsabilidades:**
- Mostrar interfaz de usuario
- Gestionar estado de anuncios
- Coordinar renovaciones
- Configuración de usuario

**Archivos relacionados:**
- `milanuncios/panel-milanuncios.html`
- `milanuncios/panel-milanuncios.js`
- `milanuncios/styles/milanuncios.css`

#### 4. Header del Panel (v5.8.5+)

**Estructura del header:**
```
[● email@mitiklive.com] [Avatar + AliasMil] [💳 Créditos | 💰 Recargar]
        ↑                      ↑                      ↑
   Usuario MitikLive    Usuario Milanuncios      Créditos
```

**IDs del HTML:**
```html
<!-- Sección 1: Estado + Email usuario MitikLive -->
<div class="ml-header__user-panel">
  <span class="ml-status-dot"></span>
  <span id="mil-panel-email">test@mitiklive.com</span>
</div>

<!-- Sección 2: Avatar + Alias usuario Milanuncios -->
<div id="mil-header-mil-user" class="ml-header__mil-user">
  <img id="mil-user-avatar" class="ml-header__avatar" />
  <span id="mil-username">NombreUsuario</span>
</div>

<!-- Sección 3: Créditos -->
<div id="mil-credits-container" class="ml-header__credits">
  <button id="mil-credits-badge">...</button>
  <a id="mil-btn-recargar">...</a>
</div>
```

**Clases CSS del header:**
```css
.ml-header              /* Contenedor principal */
.ml-header__user-panel  /* Sección 1: email MitikLive */
.ml-header__email       /* Email del panel */
.ml-header__mil-user    /* Sección 2: usuario Milanuncios */
.ml-header__avatar      /* Avatar de Milanuncios */
.ml-header__username    /* Alias de Milanuncios */
.ml-header__credits     /* Sección 3: créditos */
```

**Actualización de usuario Milanuncios:**
```javascript
function updateUserUI(user) {
  // user = null → Oculta sección 2 y limpia avatar
  // user.isLogged = true → Muestra avatar + username
  // Avatar usa ?_t=timestamp para evitar caché
}
```

#### 5. Router de Plataformas (`panel.js`)

**Responsabilidades:**
- Detectar plataforma seleccionada en login
- Redirigir al panel correspondiente
- Mantener sesión entre cambios

**Archivos relacionados:**
- `panel.js` (solo el código del router al final)

---

## 🎨 DISEÑO Y UI

### Sistema de Colores

**REGLA #3:** Usar siempre las variables CSS definidas, NUNCA colores hardcodeados.

```css
/* milanuncios/styles/milanuncios.css */

:root {
  --mil-primary: #FF6B00;           /* Naranja Milanuncios */
  --mil-primary-hover: #E65C00;     /* Naranja oscuro */
  --mil-secondary: #2C3E50;         /* Gris oscuro */
  --mil-success: #27AE60;           /* Verde */
  --mil-warning: #F39C12;           /* Amarillo */
  --mil-error: #E74C3C;             /* Rojo */
  --mil-info: #3498DB;              /* Azul */
}

/* USO CORRECTO */
.btn-primary {
  background: var(--mil-primary);  /* ✅ */
}

/* USO INCORRECTO */
.btn-primary {
  background: #FF6B00;             /* ❌ */
}
```

### Componentes Reutilizables

#### Botones

```html
<!-- Botón primario -->
<button class="btn btn-primary">Renovar</button>

<!-- Botón secundario -->
<button class="btn btn-secondary">Cancelar</button>

<!-- Botón pequeño -->
<button class="btn btn-sm">Pequeño</button>

<!-- Botón grande -->
<button class="btn btn-lg">Grande</button>
```

#### Toolbar Sticky - Una Línea (v5.6.0+)

Barra compacta con todo en una línea:

```
📦850 anuncios - 30 en esta página - [1][2][3][4][5] - [🚀 Renovar todo]
```

```html
<div class="mil-toolbar">
  <div class="mil-toolbar__row">
    <span class="mil-toolbar__stat">📦 <span id="mil-total-count">0</span> anuncios</span>
    <span class="mil-toolbar__separator">-</span>
    <span class="mil-toolbar__stat mil-toolbar__stat--muted">30 en esta página</span>
    <div class="mil-toolbar__pagination">
      <span class="mil-toolbar__separator">-</span>
      <div class="mil-pagination__numbers">...</div>
    </div>
    <div class="mil-toolbar__actions">
      <button class="btn btn-primary btn-sm">🚀 Renovar todo</button>
    </div>
  </div>
</div>
```

**CSS clave:**
```css
.mil-toolbar {
  position: sticky;
  top: 108px;
  padding: 10px 20px;
}
.mil-toolbar__row {
  display: flex;
  align-items: center;
  gap: 8px;
}
.mil-toolbar__actions {
  margin-left: auto;
}
```

#### Tabs Centradas con Botones Especiales (v5.5.9+)

```html
<nav class="ml-tabs-container">
  <div class="ml-tabs-group">
    <button class="tab-button active" data-tab="renovaciones">📱 Renovaciones</button>
    <button class="tab-button" data-tab="estadisticas">📊 Estadísticas</button>
    <button class="tab-button" data-tab="config">⚙️ Config</button>
    <!-- Botones especiales -->
    <button id="mil-btn-logout-tab" class="tab-button mil-tab-special" title="Cerrar sesión">🔐</button>
    <button id="mil-btn-popout-tab" class="tab-button mil-tab-special mil-tab-popout" title="Abrir en ventana">🗗</button>
  </div>
</nav>
```

**CSS clave:**
```css
.ml-tabs-container {
  justify-content: center; /* Centrar tabs */
}

.ml-tabs-group {
  display: flex;
  gap: 4px;
}

.mil-tab-special {
  padding: 14px 12px;
  opacity: 0.7;
}

.mil-tab-special:hover:not(.mil-tab-popout) {
  color: var(--mil-error);
}

.mil-tab-popout:hover {
  color: var(--mil-info);
}
```

**JS Popout:**
```javascript
document.getElementById('mil-btn-popout-tab')?.addEventListener('click', () => {
  chrome.windows.create({
    url: chrome.runtime.getURL('milanuncios/panel-milanuncios.html'),
    type: 'popup',
    width: 450,
    height: 700
  });
});
```

#### Alerts/Status

```html
<div class="ml-alert info">
  <span class="icon">ℹ️</span>
  <div class="message">
    <strong>Título</strong>
    <p>Descripción</p>
  </div>
</div>

<!-- Tipos: info, success, warning, error -->
```

#### Toast Notifications

```javascript
// Función en panel-milanuncios.js
showToast('Mensaje', 'success');  // success, error, warning, info
```

#### Modales (Sistema Centralizado v5.5.0+)

**REGLA #3.1:** Usar modales personalizados en lugar de alert/confirm nativos.

```javascript
// Modal de confirmación (retorna Promise<boolean>)
const confirmed = await showConfirm({
  title: 'Título',
  icon: '❓',
  message: 'Mensaje principal',
  details: '<p>Detalles HTML opcionales</p>',
  confirmText: 'Confirmar',
  cancelText: 'Cancelar',
  dangerous: false  // true = botón rojo
});

// Modal de alerta (solo OK)
await showAlert({
  title: 'Aviso',
  icon: 'ℹ️',
  message: 'Mensaje',
  details: null,
  okText: 'Aceptar'
});
```

**Clases CSS de modales (BEM):**
```css
.mil-modal                    /* Contenedor principal */
.mil-modal__overlay           /* Fondo oscuro */
.mil-modal__content           /* Caja del modal */
.mil-modal__content--sm       /* Variante pequeña */
.mil-modal__header            /* Cabecera */
.mil-modal__icon              /* Icono del header */
.mil-modal__body              /* Cuerpo */
.mil-modal__details           /* Detalles adicionales */
.mil-modal__footer            /* Pie con botones */
.mil-modal__footer--column    /* Botones en columna */
```

### Nomenclatura CSS

**REGLA #4:** Seguir nomenclatura BEM para componentes de Milanuncios.

```css
/* CORRECTO */
.mil-panel { }                    /* Bloque */
.mil-panel__header { }            /* Elemento */
.mil-panel--compact { }           /* Modificador */

.ad-card { }
.ad-card__title { }
.ad-card--renewable { }

/* INCORRECTO */
.milanuncios-panel { }            /* No usar nombre largo */
.panel-mil { }                    /* No mezclar estilos */
```

**Prefijos específicos:**
- `mil-` - Componentes específicos Milanuncios
- `ad-` - Componentes relacionados con anuncios
- `ma-` - Clases originales de Milanuncios (DOM existente)

---

## 💳 SISTEMA DE CRÉDITOS (v5.5.6+)

### Configuración

```javascript
const API_BASE_URL = 'https://www.mitiklive.com/fa/api';
const RENOVATIONS_PER_CREDIT = 5; // 1 crédito cada 5 renovaciones
```

### Flujo de Autenticación

```javascript
// Obtener JWT y User de múltiples fuentes (legacy + actual)
const stored = await getStorage([
  'jwt', 'user',                    // Legacy
  'session.access', 'session.user'  // Arquitectura actual
]);

const jwt = stored['session.access'] || stored.jwt;
const user = stored['session.user'] || stored.user;
```

### Endpoints Backend

**Consultar créditos:**
```
GET /api/billing/check-credits?user_id={userId}
Authorization: Bearer {jwt}
```

**Descontar crédito:**
```
POST /api/billing/deduct-credit
Authorization: Bearer {jwt}
Body: { "user_id": {userId} }
```

### Lógica de Descuento

```javascript
// Solo descontar cada 5 renovaciones exitosas
state.renovationCount++;

if (state.renovationCount >= RENOVATIONS_PER_CREDIT) {
  state.renovationCount = 0;
  await deductCredit();
}
```

### UI Header Créditos

```html
<div id="mil-credits-container" class="mil-credits-container">
  <button id="mil-credits-badge" class="mil-credits-badge">
    <span class="mil-credits-icon">💳</span>
    <span id="mil-credits-count">0</span>
    <span class="mil-credits-label">CRÉDITOS</span>
  </button>
  <a href="..." class="mil-btn-recargar">💰 Recargar</a>
</div>
```

**Estados visuales:**
- Normal: degradado morado
- Sin créditos: `.no-credits` → degradado rojo

---

## 🔧 API MILANUNCIOS

### Endpoint de Renovación

**URL:** `https://www.milanuncios.com/api/v4/ads/{ad_id}/renew`

**REGLA #5:** Siempre usar esta estructura exacta para renovación.

```javascript
async function renewAd(adId) {
  const url = `https://www.milanuncios.com/api/v4/ads/${adId}/renew`;
  
  const response = await fetch(url, {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json',
    },
    body: JSON.stringify({
      tokenRenew: ""  // ← IMPORTANTE: siempre vacío
    }),
    credentials: 'include'  // ← CRÍTICO: incluye cookies
  });
  
  // Ver sección "Manejo de Errores" para procesar respuesta
}
```

### Manejo de Errores

**REGLA #6:** Manejar TODOS los códigos de respuesta posibles.

```javascript
if (response.status === 200) {
  // ✅ Renovación exitosa
  return { success: true, data: await response.json() };
}

if (response.status === 422) {
  // ⏳ Cooldown - Ya renovado recientemente (18-21h)
  throw new Error('Este anuncio ya fue renovado recientemente. Espera 18-21 horas.');
}

if (response.status === 403) {
  // 🔒 Sin permisos - No es tu anuncio
  throw new Error('No tienes permisos para renovar este anuncio.');
}

if (response.status === 401) {
  // 🔐 No autenticado - Sesión expirada
  throw new Error('Sesión expirada. Recarga la página e inicia sesión.');
}

// Otros errores
throw new Error(`Error ${response.status}: ${response.statusText}`);
```

### Rate Limiting

**REGLA #7:** Respetar siempre delays entre renovaciones.

```javascript
// Delay mínimo recomendado: 3 segundos
const DEFAULT_DELAY = 3000;

// Delay máximo seguro: 10 segundos
const MAX_DELAY = 10000;

// En renovación masiva
for (const ad of ads) {
  await renewSingle(ad.id);
  await delay(state.config.delay || DEFAULT_DELAY);  // ← OBLIGATORIO
}
```

**NUNCA:**
- Renovar sin delay
- Usar delays menores a 2 segundos
- Renovar más de 50 anuncios seguidos sin pausa

### Algoritmo de Renovación Masiva (v5.5.8+)

**IMPORTANTE:** Milanuncios reordena los anuncios al renovarlos. Los renovados van a página 1, empujando los no renovados hacia las últimas páginas.

**Algoritmo Loop Inteligente:**

```
┌─────────────────────────────────────────────────────────┐
│                    LOOP PRINCIPAL                        │
├─────────────────────────────────────────────────────────┤
│  1. Ir a ÚLTIMA página                                  │
│  2. Renovar todo lo posible                             │
│  3. ¿Renovamos ≥1?                                      │
│     ├─ SÍ → Refrescar página → Volver a paso 1         │
│     └─ NO → Ir a penúltima (fallback)                  │
│  4. ¿Renovamos ≥1 en penúltima?                        │
│     ├─ SÍ → Volver a última (paso 1)                   │
│     └─ NO → Ir a antepenúltima                         │
│  5. ¿Renovamos ≥1 en antepenúltima?                    │
│     ├─ SÍ → Volver a última (paso 1)                   │
│     └─ NO → FIN (3 páginas vacías = nada renovable)    │
└─────────────────────────────────────────────────────────┘
```

**Código clave:**

```javascript
let consecutiveEmptyPages = 0;
const MAX_EMPTY_PAGES = 3;

while (!state.stopRequested) {
  const targetPage = Math.max(1, totalPages - consecutiveEmptyPages);
  
  // Navegar y renovar...
  
  if (pageRenovated > 0) {
    consecutiveEmptyPages = 0; // Resetear, volver a última
  } else {
    consecutiveEmptyPages++;   // Probar página anterior
    if (consecutiveEmptyPages >= MAX_EMPTY_PAGES) break;
  }
}
```

**¿Por qué este algoritmo?**
- Los anuncios renovados van a pág 1 → los NO renovados se acumulan en la última
- Por eso siempre volvemos a la última después de cada ciclo exitoso
- Fallback a penúltima/antepenúltima para casos especiales (coches max 3/día, etc.)

---

## 📝 REGLAS DE CÓDIGO

### Content Script

**Archivo:** `milanuncios/content-script.js`

#### Detección de Login

**REGLA #8:** Usar avatar como indicador principal de login.

```javascript
function isLoggedIn() {
  // ✅ CORRECTO: Buscar avatar del usuario
  const avatar = document.querySelector('img.sui-AtomImage-image[alt*=" "]');
  return avatar && avatar.alt && avatar.alt.trim().length > 0;
}

// ❌ INCORRECTO: Usar solo cookies
function isLoggedIn() {
  return document.cookie.includes('session');  // Poco confiable
}
```

#### Extracción de IDs

**REGLA #9:** Extraer ID del href, NO de atributos data.

```javascript
// ✅ CORRECTO
const href = card.querySelector('a[href*=".htm"]').getAttribute('href');
// href = "/otros-motor/titulo-383480696.htm"
const idMatch = href.match(/(\d+)\.htm$/);
const adId = idMatch[1];  // "383480696"

// ❌ INCORRECTO
const adId = card.getAttribute('data-adid');  // No existe en Milanuncios
```

#### Scraping de Anuncios

**REGLA #10:** Buscar siempre en el contenedor principal.

```javascript
// Estructura DOM de Milanuncios
const mainContainer = document.querySelector('.ma-ContentMyAdsV2-mainAdList');
const adCards = mainContainer.querySelectorAll('article[class*="ma-AdCard"]');

// Selectores específicos
const title = card.querySelector('.ma-AdCardV2-title');
const price = card.querySelector('.ma-AdCardV2-price');
const image = card.querySelector('img.ma-AdCardV2-photo');  // v5.6.1+
const timeText = card.querySelector('.ma-AdCardV2-time').textContent;
```

#### Objeto Anuncio (v5.6.1+)

```javascript
{
  id: "383480696",
  title: "Título del anuncio",
  price: "10 €",
  image: "https://images.milanuncios.com/...",  // URL de miniatura
  status: "Activo",
  daysRemaining: 116,
  canRenew: true,
  href: "/categoria/titulo-383480696.htm"
}
```

### Panel JavaScript

**Archivo:** `milanuncios/panel-milanuncios.js`

#### Estado Global

**REGLA #11:** Mantener estado en objeto único.

```javascript
// milanuncios/scripts/state.js
export const state = {
  // Conexión
  currentTabId: null,         // ID de pestaña activa
  currentMilUserId: null,     // ⭐ userId del usuario de Milanuncios (v5.8.4+)
  
  // Anuncios
  ads: [],                    // Anuncios detectados
  totalActiveAds: 0,          // Total de anuncios activos
  pagination: { current: 1, total: 1 },
  
  // Renovación
  isRenewing: false,          // Flag de renovación en curso
  renewQueue: [],             // Cola de renovaciones
  renewedToday: 0,            // Contador diario
  
  // Usuario MitikLive
  jwt: null,
  userId: null,               // userId de MitikLive (no confundir con currentMilUserId)
  
  // Config
  config: {
    autoRefresh: false,
    delay: 3000,
    delayBetweenAds: 2000,
    zoom: 100,
    notifications: true
  }
};
```

**IMPORTANTE - Dos userIds diferentes:**
```javascript
state.userId          // Usuario de MitikLive (del JWT)
state.currentMilUserId // Usuario de Milanuncios (de la web)
```

```javascript
// ✅ Acceso correcto
state.ads.push(newAd);

// ❌ Variables globales sueltas
let ads = [];                // NO hacer esto
let isRenewing = false;      // NO hacer esto
```

#### Funciones Async

**REGLA #12:** Siempre usar async/await, nunca .then()

```javascript
// ✅ CORRECTO
async function renewSingle(adId) {
  try {
    const result = await someAsyncOperation();
    return result;
  } catch (error) {
    console.error('Error:', error);
  }
}

// ❌ INCORRECTO
function renewSingle(adId) {
  someAsyncOperation()
    .then(result => { })
    .catch(error => { });
}
```

#### Logout (Cerrar Sesión)

**REGLA #12b:** Al cerrar sesión, mantener `selectedPlatform` para volver al mismo panel.

```javascript
// ✅ CORRECTO: Mantener plataforma seleccionada
async function handleLogout() {
  await chrome.storage.local.remove([
    'jwt', 'user', 'active_account',
    'session.access', 'session.user',
    'milConfig'
  ]);
  // Mantener Milanuncios seleccionado
  await chrome.storage.local.set({ selectedPlatform: 'milanuncios' });
  window.location.href = '../panel.html';
}

// ❌ INCORRECTO: Borrar selectedPlatform (redirige a Wallapop)
await chrome.storage.local.remove([..., 'selectedPlatform']);
```

#### Comunicación con Content Script (v5.8.2+)

**IMPORTANTE:** El sistema usa `chrome.tabs.sendMessage`, NO puertos.

**REGLA #13:** Usar funciones de `connection.js`, no llamar directamente.

```javascript
// ✅ CORRECTO: Usar funciones de connection.js
import { requestScanAds, requestRenew } from './scripts/connection.js';

requestScanAds();  // Solicita anuncios
requestRenew(adId); // Solicita renovación

// ❌ INCORRECTO: Llamar directamente
chrome.tabs.sendMessage(tabId, { action: 'mil_scrapeAds' }, ...);
```

**Verificar estado antes de operar:**
```javascript
import { isConnected, isReady, getConnectionState } from './scripts/connection.js';

if (!isConnected()) {
  showStatus('No hay conexión con Milanuncios');
  return;
}

if (!isReady()) {
  const state = getConnectionState();
  // state puede ser: 'no_login', 'wrong_page'
  return;
}

// Ahora sí operar
requestScanAds();
```

**Escuchar cambios de estado:**
```javascript
import { setListeners, ConnectionState } from './scripts/connection.js';

setListeners({
  onStatusChange: ({ state, user, url, userChanged }) => {
    // state: ConnectionState
    // user: { username, avatar, userId, isLogged }
    // userChanged: boolean - true si cambió de cuenta
    
    if (userChanged) {
      // Limpiar UI, recargar datos
    }
    
    updateUserUI(user);
  },
  
  onAdsUpdate: ({ ads, pagination, totalActive }) => {
    // Anuncios recibidos del CS
  },
  
  onRenewResult: ({ adId, success, error }) => {
    // Resultado de renovación
  },
  
  onDisconnect: () => {
    // Pestaña cerrada o CS no responde
    updateUserUI(null);  // Limpiar avatar
  }
});
```

### Estilos CSS

**Archivo:** `milanuncios/styles/milanuncios.css`

#### Organización del Archivo

**REGLA #14:** Mantener estructura por secciones.

```css
/* 1. Variables */
:root { }

/* 2. Platform Badge */
.platform-badge { }

/* 3. Alerts */
.ml-alert { }

/* 4. Toasts */
.ml-toast { }

/* 5. Componentes específicos */
.ad-card { }

/* 6. Responsive (al final) */
@media (max-width: 768px) { }
```

#### Especificidad

**REGLA #15:** Evitar !important, usar especificidad correcta.

```css
/* ✅ CORRECTO */
.ad-card.ad-card--renewable {
  border-color: var(--mil-success);
}

/* ❌ INCORRECTO */
.ad-card {
  border-color: green !important;
}
```

---

## 🐛 DEBUGGING

### Console Logs

**REGLA #16:** Usar prefijo `[Milanuncios]` en TODOS los logs.

```javascript
// Content Script
console.log('[Milanuncios CS] Mensaje aquí');
console.error('[Milanuncios CS] Error aquí');

// Panel
console.log('[Milanuncios Panel] Mensaje aquí');

// Scraper
console.log('[Milanuncios Scraper] Encontrados X anuncios');

// API
console.log('[Milanuncios API] Renovando anuncio', adId);
```

### Testing Manual

**REGLA #17:** Checklist obligatorio antes de commit.

```
✅ Checklist de Testing:

Panel:
- [ ] Login funciona con Milanuncios seleccionado
- [ ] Redirige correctamente al panel MIL
- [ ] Botón "Cambiar plataforma" vuelve al login
- [x] Botón "Cerrar sesión" limpia storage (mantiene selectedPlatform)

Detección:
- [ ] Detecta correctamente página "Mis anuncios"
- [ ] Detecta login (avatar presente)
- [ ] Muestra mensaje correcto si no hay login
- [ ] Muestra mensaje correcto si no es "Mis anuncios"

Scraping:
- [ ] Extrae todos los anuncios visibles
- [ ] IDs extraídos correctamente (numéricos)
- [ ] Títulos y precios correctos
- [ ] Estado "canRenew" correcto

Renovación:
- [ ] Renovación individual funciona
- [ ] Renovación masiva funciona
- [ ] Delays entre renovaciones
- [ ] Manejo de error 422 (cooldown)
- [ ] Manejo de error 403/401
- [ ] Toast muestra éxito/error

Config:
- [ ] Auto-refresh funciona
- [ ] Delay se guarda
- [ ] Notificaciones on/off funciona

Wallapop:
- [ ] TODO sigue funcionando igual
- [ ] Login Wallapop no afectado
- [ ] Panel Wallapop intacto
```

### Errores Comunes

#### Error: "No se detectan anuncios"

**Causa:** Selectores CSS incorrectos o DOM cambió.

**Solución:**
1. Abrir DevTools en página "Mis anuncios"
2. Inspeccionar estructura actual
3. Actualizar selectores en `scrapeAdIds()`

#### Error: "Runtime error: Could not establish connection"

**Causa:** Content script no cargado en la página.

**Solución:**
1. Verificar `manifest.json` tiene matches correctos
2. Recargar extensión (chrome://extensions/)
3. Recargar página de Milanuncios

#### Error: "401 Unauthorized al renovar"

**Causa:** Usuario no está realmente logueado.

**Solución:**
1. Verificar función `isLoggedIn()` funciona
2. Comprobar que avatar está presente en DOM
3. Usuario debe iniciar sesión en Milanuncios

### Edge Cases Cubiertos (v5.8.8+)

El sistema de conexión maneja estos casos automáticamente:

| # | Caso | Comportamiento | Estado |
|---|------|----------------|--------|
| 1 | Abre panel SIN pestaña Milanuncios | Modal "Abrir Milanuncios" después de 3s | ✅ |
| 2 | Abre panel CON pestaña abierta | Conecta automáticamente | ✅ |
| 3 | Cierra pestaña con panel abierto | `onRemoved` → `setDisconnected()` | ✅ |
| 4 | Abre pestaña con panel ya abierto | `onUpdated` detecta y conecta | ✅ |
| 5 | Login en Milanuncios con panel abierto | Ping cada 2s detecta `isLogged=true` | ✅ |
| 6 | Logout en Milanuncios con panel abierto | Ping detecta y muestra "Inicia sesión" | ✅ |
| 7 | Cambia de cuenta en Milanuncios | Compara `userId`, recarga anuncios | ✅ |
| 8 | Navega a "mis-anuncios" | `onUpdated` con `changeInfo.url` | ✅ |
| 9 | Navega FUERA de "mis-anuncios" | Ping detecta `isMyAdsPage=false` | ✅ |
| 10 | Recarga página Milanuncios | `onUpdated` status=complete, reinyecta CS | ✅ |
| 11 | Múltiples pestañas Milanuncios | Prioriza la que tiene `/mis-anuncios` | ✅ |
| 12 | CS no carga (error/bloqueado) | Reinyecta con reintentos (max 2) | ✅ |
| 13 | Usuario cambia de pestaña | `onActivated` verifica si es Milanuncios | ✅ |
| 14 | Sesión Milanuncios expira | Ping detecta `isLogged=false` | ✅ |
| 15 | Clic en "Actualizar lista" sin conexión | `forceReconnect()` + reintento | ✅ |
| 16 | Avatar cacheado tras cambio cuenta | URL con `?_t=timestamp` | ✅ |
| 17 | Conexión establecida con modal abierto | Modal se cierra automáticamente | ✅ |

**Modal "Abrir Milanuncios" (v5.8.8+):**
```
Flujo cuando no hay pestaña:
1. Panel abre → "🔍 Buscando Milanuncios..."
2. Después de 3s sin conexión → Modal aparece
3. Usuario hace clic en "🚀 Abrir Milanuncios"
4. Se abre nueva pestaña en /mis-anuncios
5. connection.js detecta y conecta
6. Modal se cierra automáticamente
7. Panel muestra anuncios
```

**Listeners activos:**
```javascript
// connection.js registra estos listeners:
chrome.tabs.onUpdated   // Detecta: recarga, navegación SPA, carga completa
chrome.tabs.onRemoved   // Detecta: cierre de pestaña
chrome.tabs.onActivated // Detecta: cambio de pestaña activa

// Verificación periódica cada 2 segundos:
setInterval(checkMilanunciosTabs, 2000);
```

**Función de reconexión manual:**
```javascript
import { forceReconnect } from './scripts/connection.js';

// Resetea estado y busca pestañas de nuevo
const reconnected = await forceReconnect();
```

---

## 📚 EJEMPLOS DE CÓDIGO

### Ejemplo Completo: Añadir Nueva Función

```javascript
// 1. Añadir al content-script.js
async function pauseAd(adId) {
  console.log('[Milanuncios API] Pausando anuncio', adId);
  
  const response = await fetch(`https://www.milanuncios.com/api/v4/ads/${adId}/pause`, {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    credentials: 'include'
  });
  
  if (!response.ok) {
    throw new Error(`Error ${response.status}`);
  }
  
  return { success: true };
}

// 2. Añadir listener
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'mil_pauseAd') {
    pauseAd(request.adId)
      .then(result => sendResponse(result))
      .catch(error => sendResponse({ success: false, error: error.message }));
    return true;
  }
});

// 3. Usar en panel-milanuncios.js
async function pauseSingle(adId) {
  chrome.tabs.sendMessage(state.currentTabId, { 
    action: 'mil_pauseAd', 
    adId 
  }, (response) => {
    if (response?.success) {
      showToast('Anuncio pausado', 'success');
    }
  });
}

// 4. Añadir botón en HTML
<button onclick="pauseSingle('123456')">Pausar</button>
```

---

## 🔄 VERSIONADO

**Formato:** v{MAJOR}.{MINOR}.{PATCH}

- **MAJOR:** Cambios que rompen compatibilidad
- **MINOR:** Nuevas funcionalidades
- **PATCH:** Bug fixes

**Ejemplo:**
- v1.0.0 - Release inicial
- v1.1.0 - Añadir pausar/despausar anuncios
- v1.1.1 - Fix detección de login

---

## 📞 TROUBLESHOOTING RÁPIDO

| Síntoma | Causa Probable | Solución |
|---------|----------------|----------|
| No redirige a panel MIL | Router no funciona | Ver logs `[Platform Router]` |
| No detecta anuncios | Selectores incorrectos | Actualizar `scrapeAdIds()` |
| Error 401 al renovar | No hay sesión | Verificar `isLoggedIn()` |
| Panel en blanco | Error JS | Ver consola del panel |
| Wallapop roto | Código modificado | Revertir cambios |

---

## ✅ CHECKLIST PRE-COMMIT

```
Antes de hacer commit, verificar:

Código:
- [ ] Sin console.logs de debug olvidados
- [ ] Todos los errores manejados
- [ ] Variables con nombres descriptivos
- [ ] Funciones documentadas

Testing:
- [ ] Probado en página real de Milanuncios
- [ ] Wallapop sigue funcionando
- [ ] No hay errores en consola

Documentación:
- [ ] Guía actualizada si hay cambios arquitectónicos
- [ ] Comentarios en código complejo
- [ ] CHANGELOG.md actualizado

Arquitectura:
- [ ] Código aislado (no mezcla con Wallapop)
- [ ] Usa funciones reutilizables cuando posible
- [ ] Sigue nomenclatura establecida
```

---

## 🚀 ROADMAP FUTURO

### v1.1.0 (Próximo)
- [ ] Pausar/despausar anuncios
- [ ] Editar título y descripción
- [ ] Gestión de imágenes

### v1.2.0
- [ ] Estadísticas de visualizaciones
- [ ] Historial de renovaciones
- [ ] Exportar/importar anuncios

### v2.0.0
- [ ] Publicación de anuncios
- [ ] Sincronización con backend
- [ ] Multi-cuenta

---

**Última actualización:** 2024-12-04  
**Mantenedor:** Javier (@MitikLive)  
**Versión guía:** 1.0.0

---

## 🆕 CAMBIOS v5.7.0 (04 Dic 2025)

### Sistema de Verificación Inteligente

Nueva función centralizada `requireMilanunciosActive()` que:
- Detecta automáticamente si hay pestaña de Milanuncios
- Verifica si el usuario está logueado
- Muestra modales inteligentes para guiar al usuario
- Permite abrir pestaña en segundo plano
- Botón "Verificar de nuevo" para revalidar

### Extracción de userId (v5.7.2)

Nueva función `getUserId()` en content-script.js que obtiene el ID del usuario de Milanuncios:

```javascript
// Fuentes de datos (en orden de prioridad):
// 1. Cookie ma_uid (la más directa y confiable)
// 2. Cookie ajs_user_id (formato: sdrn:milanuncios.com:user:11535792)
// 3. localStorage ajs_user_id

function getUserId() {
  // Lee cookie ma_uid=11535792
  const cookies = document.cookie.split(';');
  for (const cookie of cookies) {
    const [name, value] = cookie.trim().split('=');
    if (name === 'ma_uid' && /^\d+$/.test(value)) {
      return value;
    }
  }
  // ... fallbacks
}
```

**Uso en panel-milanuncios.js:**
```javascript
const status = await checkTabStatus(tabId);
const userId = status?.userInfo?.userId;  // "11535792"

// Usar en API call
fetch(`https://classifieds.gw.milanuncios.com/api/v1/users/${userId}/classifieds`, ...)
```

### Logout correcto (v5.7.3)

**CRÍTICO:** El logout de Milanuncios DEBE enviar `SESSION.LOGOUT` al Service Worker para limpiar completamente la sesión.

```javascript
async function handleLogout() {
  // 1. Enviar SESSION.LOGOUT al SW (limpia session.*, revoca dispositivo)
  await chrome.runtime.sendMessage({ type: 'SESSION.LOGOUT' });
  
  // 2. Limpiar keys adicionales locales
  await chrome.storage.local.remove([
    'jwt', 'user', 'active_account',
    'milConfig', 'mil.renovatedAds', 'mil.lastRenewalTime'
  ]);
  
  // 3. Mantener plataforma seleccionada
  await chrome.storage.local.set({ selectedPlatform: 'milanuncios' });
  
  // 4. Redirigir con delay
  setTimeout(() => window.location.href = '../panel.html', 300);
}
```

**Sin `SESSION.LOGOUT`, el SW sigue intentando refrescar tokens causando:**
- Errores de `refresh_cooldown`
- Redirección al panel de Milanuncios en loop
- Necesidad de hacer logout múltiples veces

### Nuevos Modales

1. **Modal "Milanuncios no detectado"**
   - Botón "Cambiar a Milanuncios" (si hay pestaña)
   - Botón "Abrir Milanuncios" (nueva pestaña)
   - Botón "Verificar de nuevo"

2. **Modal "Inicia sesión"**
   - Botón "Ir a Milanuncios"
   - Botón "Verificar de nuevo"

3. **Modal de Progreso Exportación**
   - Barra de progreso visual
   - Contador de anuncios
   - Botón cancelar

### Exportar a CSV

Nueva funcionalidad en Config:
- Botón "Exportar mis anuncios"
- Descarga TODOS los anuncios (paginación automática)
- Campos exportados:
  - ID, Título, Descripción, Precio
  - Categoría (breadcrumb completo)
  - Provincia, Localidad
  - Estado, Vendedor, Teléfono
  - URL, Imágenes (hasta 5)
  - Fecha caducidad, Fecha exportación

### Funciones Nuevas en panel-milanuncios.js

```javascript
// Verificación centralizada
requireMilanunciosActive() → {ok, tabId, reason}

// Modales
hideAllModals()
showLoginModal()
showExportModal()
hideExportModal()
updateExportProgress(current, total, text)
showTabRequiredModal(existingTabs)
showLoginRequiredModal(tab)
showGenericModal(options)

// Exportación
exportAllAdsToCSV()
generateCSV(ads)
```

### HTML Nuevos Elementos

```html
<!-- Config: Sección Exportar -->
<div class="mil-config__section">
  <h4>📤 Exportar Anuncios</h4>
  <button id="mil-btn-export-csv">📥 Exportar mis anuncios</button>
</div>

<!-- Modal Login -->
<div id="mil-modal-login">...</div>

<!-- Modal Exportación -->
<div id="mil-modal-export">...</div>
```

### CSS Nuevas Clases

```css
.mil-modal__hint         /* Texto de ayuda en modales */
.btn-info               /* Botón azul (verificar) */
.mil-progress-bar       /* Barra de progreso */
.mil-progress-bar__fill /* Relleno de progreso */
.mil-config__separator  /* Separador en config */
.mil-config__section-title /* Título de sección */
.mil-config__card--highlight /* Card destacada */
.mil-config__export-status /* Estado de exportación */
```

---

## 🆕 CAMBIOS v5.8.3 - v5.8.6 (05 Dic 2025)

### Preferencias Sincronizadas con Backend (v5.8.3)

Las preferencias del panel de Milanuncios ahora se guardan en el backend (columna `ui_preferences` de tabla `users`).

**Campos en backend:**
```javascript
{
  zoom_milanuncios: 100,          // Zoom panel (70-130%)
  active_tab_milanuncios: "renovaciones", // Tab activa
  delay_milanuncios: 3,           // Delay entre anuncios (2-30s)
  notifications_milanuncios: true, // Mostrar alertas
  auto_refresh_milanuncios: false, // Auto-actualizar cada 30s
  sound_milanuncios: true         // Sonido al terminar
}
```

**Flujo de carga (config.js):**
1. Carga de localStorage (inmediato)
2. Sincroniza con backend (sobrescribe si hay diferencias)
3. Aplica a UI

**Flujo de guardado:**
1. Guarda en localStorage (inmediato)
2. Envía al backend (async, no bloquea)

### Contador de Cuenta Atrás en Footer (v5.8.6)

Durante la renovación masiva, se muestra un contador en el footer indicando el tiempo restante hasta el próximo anuncio.

**HTML:**
```html
<div id="mil-footer-countdown" class="mil-footer__countdown">
  <span class="mil-footer__countdown-icon">⏳</span>
  <span id="mil-countdown-value" class="mil-footer__countdown-value">5s</span>
</div>
```

**Funciones en ui.js:**
```javascript
showCountdown()              // Muestra el contador
hideCountdown()              // Oculta el contador
startCountdown(seconds)      // Inicia cuenta atrás
stopCountdown()              // Detiene cuenta atrás
```

**Uso en panel-milanuncios.js:**
```javascript
// randomDelay ahora muestra el countdown automáticamente
async function randomDelay(baseMs) {
  const seconds = Math.ceil(actualDelay / 1000);
  startCountdown(seconds);
  await delay(actualDelay);
  hideCountdown();
}
```

### CSS Nuevas Clases (v5.8.6)

```css
.mil-footer__countdown       /* Contenedor del contador */
.mil-footer__countdown-icon  /* Icono ⏳ */
.mil-footer__countdown-value /* Valor (ej: "5s") */
```

### Slider Delay Extendido (v5.8.6)

El slider de delay ahora permite valores de **2-30 segundos** (antes era 2-10s).

```html
<input type="range" id="mil-config-delay" min="2" max="30" step="1" value="3">
```

> **Nota:** Con 30 segundos el Service Worker no se duerme (el límite es ~30s de inactividad).

### Mejoras UI Config (v5.8.4)

- Card "Exportar a CSV" ahora ocupa 1 columna (igual que las demás)
- Botones "Recargar créditos" y "Exportar" ya no ocupan todo el ancho
- Mayor separación entre grid y botones Manual/Cerrar sesión
- Línea separadora antes de botones de acción

### Versión Dinámica en Footer (v5.8.3)

El footer ahora lee la versión del manifest dinámicamente:

```javascript
const manifest = chrome.runtime.getManifest();
versionEl.textContent = `v${manifest.version}`;
```

---

## 🔒 SISTEMA ANTI-ABUSO (v5.8.7)

### Tabla milanuncios_accounts

Similar a `walla_accounts`, almacena las cuentas de Milanuncios asociadas a usuarios del panel.

```sql
CREATE TABLE `milanuncios_accounts` (
  `id` bigint(20) unsigned NOT NULL AUTO_INCREMENT,
  `user_id` bigint(20) unsigned NOT NULL,  -- FK a users (panel MitikLive)
  `mil_user_id` varchar(32) NOT NULL,      -- ID de Milanuncios (cookie ma_uid)
  `alias` varchar(80) DEFAULT NULL,        -- Nombre del usuario
  `created_at` timestamp NOT NULL DEFAULT current_timestamp(),
  `updated_at` timestamp NULL DEFAULT NULL,
  PRIMARY KEY (`id`),
  UNIQUE KEY `uq_user_milid` (`user_id`, `mil_user_id`),
  KEY `idx_mil_user_id` (`mil_user_id`),
  CONSTRAINT `fk_mil_accounts_user` FOREIGN KEY (`user_id`) REFERENCES `users` (`id`) ON DELETE CASCADE
);
```

### Endpoint POST /api/milanuncios/accounts

**Ubicación:** `app/routers/milanuncios.py`

**Parámetros:**
- `mil_user_id` (obligatorio): ID de usuario de Milanuncios (cookie ma_uid)
- `alias` (opcional): Nombre del usuario en Milanuncios

**Flujo:**
1. Verifica si `mil_user_id` ya existe en OTRO usuario (excepto admin)
2. Si existe: suspende usuario actual + devuelve error 409
3. Si ya existe en ESTE usuario: retorna cuenta existente
4. Si no existe: crea nueva cuenta

**Respuestas:**
- `200 OK`: `{ ok: true, id, alias, mil_user_id, is_new }`
- `409 Conflict`: Cuenta duplicada (usuario suspendido)

### Verificación en Panel

```javascript
// panel-milanuncios.js - Antes de renovación masiva
async function verifyMilanunciosAccount() {
  const milUserId = state.currentMilUserId;
  const milUsername = state.currentMilUsername;
  
  const response = await fetch(
    `${API_BASE}/milanuncios/accounts?mil_user_id=${milUserId}&alias=${milUsername}`,
    { method: 'POST', headers: { 'Authorization': `Bearer ${jwt}` } }
  );
  
  if (response.status === 409) {
    return { ok: false, error: 'duplicate' };
  }
  
  return { ok: true };
}

// En startMassiveRenewal():
const accountCheck = await verifyMilanunciosAccount();
if (!accountCheck.ok) {
  showAlert({ title: '⚠️ Cuenta ya vinculada', ... });
  return;
}
```

### Variables de Estado

```javascript
// state.js
state.currentMilUserId    // ID de Milanuncios (cookie ma_uid)
state.currentMilUsername  // Nombre/alias del usuario
```

### Notas Importantes

- El admin puede gestionar cualquier cuenta (no se verifica duplicados)
- `mil_user_id` es inmutable (viene de cookie `ma_uid`)
- **La verificación se hace en dos puntos:**
  1. Al detectar conexión con usuario nuevo de Milanuncios (bloqueo inmediato)
  2. Al iniciar renovación masiva (respaldo)
- **Modal de cuenta duplicada usa `showModal()` de `scripts/utils.js`** (compartido con Wallapop)

### ⚠️ Orden de Inicialización Crítico (v5.8.11)

En `panel-milanuncios.js`, el orden de inicialización es **crítico**:

```javascript
// ✅ CORRECTO: JWT ANTES de initConnectionListener
state.jwt = jwt;
state.userId = ...;
initConnectionListener();  // handleConnectionStatusChange puede usar state.jwt

// ❌ INCORRECTO: Causaba "No hay sesión activa"
initConnectionListener();  // dispara handleConnectionStatusChange SIN jwt
state.jwt = jwt;           // demasiado tarde
```

`handleConnectionStatusChange()` llama a `verifyMilanunciosAccount()` que necesita `state.jwt`.
