/**
 * MitikLive - Content Script para Milanuncios v2.1
 * Sistema de mensajes simplificado
 * Aislado - NO afecta código de Wallapop
 */

// Guard para evitar doble inyección
if (window.__MITIKLIVE_MIL_CS_LOADED__) {
  console.log('[Milanuncios CS] Ya cargado, ignorando re-inyección');
} else {
  window.__MITIKLIVE_MIL_CS_LOADED__ = true;

console.log('[Milanuncios CS] Content script v2.1 cargando...');

// ==================== DETECTORES ====================

/**
 * Detecta si estamos en la página "Mis anuncios"
 */
function isMyAdsPage() {
  return window.location.pathname.includes('/mis-anuncios') ||
         window.location.pathname.includes('/my-ads');
}

/**
 * Detecta si el usuario está logueado
 * Múltiples métodos de detección
 */
function isLoggedIn() {
  // Método 1: Buscar avatar del usuario (selector específico)
  const avatarImg = document.querySelector('.ma-UserAvatar img.sui-AtomImage-image');
  if (avatarImg && avatarImg.src) {
    console.log('[Milanuncios CS] Login detectado via avatar usuario');
    return true;
  }
  
  // Método 2: Buscar contenedor .ma-UserAvatar
  if (document.querySelector('.ma-UserAvatar')) {
    console.log('[Milanuncios CS] Login detectado via .ma-UserAvatar');
    return true;
  }
  
  // Método 3: Buscar cookie de sesión
  if (document.cookie.includes('ma_uid=')) {
    console.log('[Milanuncios CS] Login detectado via cookie ma_uid');
    return true;
  }
  
  console.log('[Milanuncios CS] Usuario NO logueado');
  return false;
}

/**
 * Obtiene información del usuario
 */
function getUserInfo() {
  // Selector específico del avatar de usuario (NO el logo)
  const avatarImg = document.querySelector('.ma-UserAvatar img.sui-AtomImage-image');
  
  let avatar = null;
  let username = 'Usuario';
  
  if (avatarImg) {
    avatar = avatarImg.src || null;
    // El alt contiene el nombre del usuario
    if (avatarImg.alt && avatarImg.alt.trim().length > 0) {
      username = avatarImg.alt.trim();
    }
  }
  
  return {
    username,
    isLogged: isLoggedIn(),
    avatar,
    userId: getUserId()
  };
}

/**
 * Obtiene el userId de Milanuncios desde cookies
 */
function getUserId() {
  const cookies = document.cookie.split(';');
  
  for (const cookie of cookies) {
    const [name, value] = cookie.trim().split('=');
    if (name === 'ma_uid' && value && /^\d+$/.test(value)) {
      return value;
    }
  }
  
  // Fallback: ajs_user_id
  for (const cookie of cookies) {
    const [name, value] = cookie.trim().split('=');
    if (name === 'ajs_user_id') {
      const decoded = decodeURIComponent(value).replace(/"/g, '');
      const match = decoded.match(/user:(\d+)/);
      if (match) return match[1];
    }
  }
  
  return null;
}

// ==================== SCRAPER ====================

/**
 * Extrae anuncios de la página actual
 */
function scrapeAdIds() {
  const ads = [];
  
  // Múltiples selectores para el contenedor principal
  const containerSelectors = [
    '.ma-ContentMyAdsV2-mainAdList',
    '.ma-ContentMyAds-mainAdList',
    '[class*="ContentMyAds"] [class*="mainAdList"]',
    '[class*="MyAds"] article[class*="AdCard"]',
    'main article[class*="AdCard"]'
  ];
  
  let mainContainer = null;
  for (const selector of containerSelectors) {
    mainContainer = document.querySelector(selector);
    if (mainContainer) {
      console.log('[Milanuncios CS] Contenedor encontrado con:', selector);
      break;
    }
  }
  
  // Si no hay contenedor específico, buscar directamente las tarjetas
  let adCards;
  if (mainContainer) {
    adCards = mainContainer.querySelectorAll('article[class*="ma-AdCard"]');
  } else {
    // Fallback: buscar en todo el documento
    adCards = document.querySelectorAll('article[class*="ma-AdCard"], article[class*="AdCard"]');
    console.log('[Milanuncios CS] Buscando tarjetas en todo el documento');
  }
  
  if (!adCards || adCards.length === 0) {
    console.warn('[Milanuncios CS] No se encontraron tarjetas de anuncios');
    return ads;
  }
  
  console.log(`[Milanuncios CS] Encontradas ${adCards.length} tarjetas`);
  
  adCards.forEach((card) => {
    const linkElement = card.querySelector('a[href*=".htm"]');
    const href = linkElement?.getAttribute('href');
    
    if (!href) return;
    
    const idMatch = href.match(/(\d+)\.htm$/);
    const adId = idMatch ? idMatch[1] : null;
    
    if (!adId) return;
    
    const titleElement = card.querySelector('.ma-AdCardV2-title, .ma-SharedText.ma-AdCardV2-title, [class*="AdCard"][class*="title"]');
    const title = titleElement?.textContent?.trim() || 'Sin título';
    
    const priceElement = card.querySelector('.ma-AdCardV2-price, [class*="AdCard"][class*="price"]');
    const price = priceElement?.textContent?.trim() || 'Sin precio';
    
    const imageElement = card.querySelector('img.ma-AdCardV2-photo, img[class*="AdCard"][class*="photo"], img[class*="photo"]');
    const image = imageElement?.src || null;
    
    const statusElement = card.querySelector('.ma-AdCardV2-complexButtonTriggerTextLabel, [class*="status"], [class*="Status"]');
    const status = statusElement?.textContent?.trim() || 'Activo';
    
    const timeElement = card.querySelector('.ma-AdCardV2-time, [class*="time"], [class*="Time"]');
    const timeText = timeElement?.textContent?.trim() || '';
    const daysMatch = timeText.match(/(\d+)\s*días/);
    const daysRemaining = daysMatch ? parseInt(daysMatch[1]) : 30; // Default 30 si no encuentra
    
    // Puede renovar si tiene días y está activo
    const canRenew = daysRemaining > 0 && (status.toLowerCase() === 'activo' || status.toLowerCase() === 'desconocido');
    
    ads.push({
      id: adId,
      title,
      price,
      image,
      status,
      daysRemaining,
      canRenew,
      href
    });
  });
  
  console.log(`[Milanuncios CS] Extraídos ${ads.length} anuncios`);
  return ads;
}

/**
 * Obtiene total de anuncios activos
 */
function getTotalActiveAds() {
  const selectors = [
    '.sui-MoleculeTabs-item.is-active span',
    '.sui-MoleculeTabs-item.is-active',
    '[role="tab"][aria-selected="true"] span',
    '[role="tab"][aria-selected="true"]'
  ];
  
  for (const selector of selectors) {
    const el = document.querySelector(selector);
    if (el) {
      const match = el.textContent.match(/\((\d+)\)/);
      if (match) return parseInt(match[1]);
    }
  }
  
  return null;
}

/**
 * Obtiene información de paginación
 */
function getPaginationInfo() {
  const urlParams = new URLSearchParams(window.location.search);
  const current = parseInt(urlParams.get('page') || '1');
  let total = current;
  
  // Buscar contenedor de paginación
  const paginationSelectors = [
    '[class*="ma-Pagination"]',
    '[class*="Pagination"]',
    '.pagination',
    'nav[aria-label*="paginación"]'
  ];
  
  let paginationContainer = null;
  for (const selector of paginationSelectors) {
    paginationContainer = document.querySelector(selector);
    if (paginationContainer) break;
  }
  
  if (paginationContainer) {
    paginationContainer.querySelectorAll('a, button, span, li').forEach(el => {
      const text = el.textContent.trim();
      if (/^\d+$/.test(text)) {
        const pageNum = parseInt(text);
        if (!isNaN(pageNum) && pageNum > total) total = pageNum;
      }
    });
  }
  
  // Fallback: buscar links con page=
  if (total === current) {
    document.querySelectorAll('a[href*="page="]').forEach(el => {
      const match = el.getAttribute('href')?.match(/page=(\d+)/);
      if (match) {
        const pageNum = parseInt(match[1]);
        if (pageNum > total) total = pageNum;
      }
    });
  }
  
  return { current, total };
}

// ==================== API MILANUNCIOS ====================

/**
 * Renueva un anuncio
 */
async function renewAd(adId) {
  try {
    const response = await fetch(`https://www.milanuncios.com/api/v4/ads/${adId}/renew`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ tokenRenew: "" }),
      credentials: 'include'
    });
    
    if (!response.ok) {
      if (response.status === 422) {
        return { success: false, error: 'Cooldown activo (422)' };
      }
      if (response.status === 401) {
        return { success: false, error: 'No autenticado (401)' };
      }
      return { success: false, error: `Error ${response.status}` };
    }
    
    const data = await response.json();
    return { success: true, data };
    
  } catch (error) {
    return { success: false, error: error.message };
  }
}

// ==================== LISTENER DE MENSAJES ====================

chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  console.log('[Milanuncios CS] Mensaje recibido:', request.action);
  
  // Ping para verificar conexión
  if (request.action === 'mil_ping') {
    const response = {
      pong: true,
      platform: 'milanuncios',
      isMyAdsPage: isMyAdsPage(),
      isLogged: isLoggedIn(),
      userInfo: getUserInfo(),
      url: window.location.href
    };
    console.log('[Milanuncios CS] Respondiendo ping:', response);
    sendResponse(response);
    return true;
  }
  
  // Obtener estado
  if (request.action === 'mil_getStatus') {
    sendResponse({
      platform: 'milanuncios',
      isMyAdsPage: isMyAdsPage(),
      isLogged: isLoggedIn(),
      userInfo: getUserInfo(),
      url: window.location.href
    });
    return true;
  }
  
  // Extraer anuncios
  if (request.action === 'mil_scrapeAds') {
    sendResponse({
      ads: scrapeAdIds(),
      pagination: getPaginationInfo(),
      totalActiveAds: getTotalActiveAds()
    });
    return true;
  }
  
  // Renovar anuncio
  if (request.action === 'mil_renewAd') {
    renewAd(request.adId).then(result => sendResponse(result));
    return true; // Async response
  }
  
  // Navegar a página
  if (request.action === 'mil_navigateToPage') {
    window.location.href = `https://www.milanuncios.com/mis-anuncios?page=${request.page}`;
    sendResponse({ success: true });
    return true;
  }
  
  return false;
});

// ==================== INICIALIZACIÓN ====================

function init() {
  console.log('[Milanuncios CS] ✅ Inicializado');
  console.log('[Milanuncios CS] - URL:', window.location.href);
  console.log('[Milanuncios CS] - Es "Mis anuncios":', isMyAdsPage());
  console.log('[Milanuncios CS] - Logueado:', isLoggedIn());
  
  if (isLoggedIn()) {
    const user = getUserInfo();
    console.log('[Milanuncios CS] - Usuario:', user.username);
  }
}

// Esperar DOM
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', init);
} else {
  init();
}

} // Fin del guard
