# 📱 GUÍA DE LA EXTENSIÓN CHROME - MitikLive v5.8

**⚠️ ÚNICA FUENTE DE VERDAD - EXTENSIÓN CHROME**

**Fecha actualización:** 2024-12-03  
**Extensión:** v4.0.2  
**Basada en:** GUIA-MAESTRA-MITIKLIVE-v5.8-LINUX.md

**Estado:** 🔴 CRÍTICO - Aplicar en TODOS los cambios de la extensión

---

# 📚 GUÍA MAESTRA DE DESARROLLO - MitikLive v5.8 LINUX

**⚠️ ÚNICA FUENTE DE VERDAD - ACTUALIZADA PARA ENTORNO LINUX**

**Fecha actualización:** 2025-11-30  
**Versiones consolidadas:** v3.2, v3.3, v3.4, v3.5, v3.5.1, v3.6.0, v3.6.2, v3.6.3, v4.1, v4.2, v5.0, v5.1, v5.2, v5.3, v5.4, v5.5, v5.6, v5.7, **v5.8**  
**Extensión:** v2.2.3 | **Backend:** v2.9.35  
**Entorno:** Ubuntu 24.04 LTS + Nginx + MariaDB + Python 3.x + Redis 7.1+

**Estado:** 🔴 CRÍTICO - Aplicar en TODOS los cambios

---

## 🤖 NOTA IMPORTANTE PARA USO CON CLAUDE

> **PROBLEMA:** En conversaciones largas, Claude tiende a olvidar consultar esta guía y comete errores que ya están documentados aquí.
>
> **SOLUCIÓN:** Recordar a Claude en momentos clave:
> - **Antes de correcciones:** "Revisa la guía antes de corregir"
> - **Antes de refactorizaciones:** "Lee las reglas X-Y de la guía primero"
> - **Si ves que se desvía:** "¿Has comprobado esto en la guía?"
> - **Para imports:** "Consulta las reglas 49-51 antes de tocar imports"
>
> **NO ASUMAS** que Claude recordará la guía durante toda la conversación. Su atención se fragmenta con muchos mensajes.

---

## 🐧 CAMBIOS RESPECTO A VERSIÓN WINDOWS

### Infraestructura
- ✅ **Sistema Operativo:** Windows Server → Ubuntu 24.04 LTS
- ✅ **Servidor Web:** Apache/XAMPP → Nginx
- ✅ **Base de Datos:** MySQL → MariaDB 10.11
- ✅ **PHP:** mod_php → PHP-FPM 8.3
- ✅ **Gestión Procesos:** Unicorn (1 worker) → Gunicorn (16 workers)
- ✅ **SSL:** Cloudflare Origin Certificate (mismo)
- ✅ **DNS Dinámico:** No-IP DUC (mismo)

### Rutas del Sistema
```
ANTES (Windows):
C:/xampp/htdocs/wallapop/          → Frontend PHP
C:/fastapi/wallapop_api/app/       → Backend FastAPI
C:/fastapi/data/data/products/          → Imágenes

AHORA (Linux):
/var/www/mitiklive/frontend/wallapop/  → Frontend PHP
/var/www/mitiklive/backend/app/        → Backend FastAPI  
/var/www/mitiklive/data/data/products/      → Imágenes
/var/www/mitiklive/extension/          → Extensión Chrome
```

### Servicio de Archivos Estáticos
```
ANTES: Backend montaba imágenes con FastAPI StaticFiles en /media o /public
AHORA: Nginx sirve imágenes directamente desde /data (más eficiente)
```

### Nueva Variable de Entorno
```bash
# Backend .env (/var/www/mitiklive/backend/.env)
MEDIA_URL=https://mitiklive.com  # ⭐ NUEVA - URL base para imágenes
MEDIA_ROOT=/var/www/mitiklive/data
```

### Scripts de Utilidad
```bash
~/reiniciar-backend.sh      # Reinicia FastAPI/Gunicorn
~/reiniciar-todo.sh         # Reinicia todos los servicios
~/comprimir-extension.sh    # Empaqueta extensión para Chrome
~/mitiklive-dev/           # Accesos directos a todo el código
```

---

## 📋 ÍNDICE COMPLETO

### 🐧 [ENTORNO LINUX](#cambios-respecto-a-version-windows)
- [Infraestructura y Rutas](#infraestructura)
- [Comandos Útiles Linux](#comandos-utiles-linux)
- [Configuración Nginx](#configuracion-nginx)
- [Gestión de Servicios](#gestion-de-servicios)
- [🆕 Centralización de URLs de Imágenes](#centralizacion-de-urls-de-imagenes)
- [🆕 WebSocket Multi-Worker con Redis](#websocket-multi-worker-con-redis-pubsub)
- [🆕 Gestión de Tokens Wallapop](#gestion-de-tokens-wallapop)
- [🆕 v5.6: Sistema de Logs y Debug Mode](#sistema-de-logs-y-debug-mode)
- [🆕 v5.6: Refresh Token MitikLive](#refresh-token-mitiklive)
- [🆕 v5.6: Refresh Token Wallapop via Iframe](#refresh-token-wallapop-via-iframe)
- [🆕 v5.6: Prevención Cierre Sesión Wallapop](#prevencion-cierre-sesion-wallapop)
- [🆕 v5.7: Sistema de Verificación de Versión](#sistema-verificacion-version)
- [🆕 v5.7: Layout Centrado Panel Extendido](#layout-centrado-panel-extendido)

### 🎯 [FUNDAMENTOS](#fundamentos)
- [Flujo de Trabajo Obligatorio](#flujo-de-trabajo-obligatorio)
- [Arquitectura Global Multi-Usuario](#arquitectura-global-multi-usuario)
- [Arquitectura de Archivos](#arquitectura-de-archivos)

### 🏛️ [ARQUITECTURAS COMPLETAS](#arquitecturas-completas)
- [Extensión - Panel UI](#extension-panel-ui)
- [Service Worker - Backend Logic](#service-worker-backend-logic)
- [Backend - FastAPI](#backend-fastapi)
- [WebSocket y SSE](#websocket-y-sse)
- [Sistema de Base de Datos](#sistema-de-base-de-datos)

### 📜 [REGLAS DE ORO (1-63)](#reglas-de-oro)
- [Reglas Fundamentales (1-10)](#reglas-fundamentales)
- [Reglas de Centralización (11-16)](#reglas-de-centralizacion)
- [Reglas de Publicación (17-24)](#reglas-de-publicacion)
- [Reglas de Performance (25-28)](#reglas-de-performance)
- [Reglas de Memoria (29-35)](#reglas-de-memoria)
- [Reglas Backend (36-40)](#reglas-backend)
- [Reglas Frontend (41-44)](#reglas-frontend)
- [🆕 Reglas de Imágenes y Multi-Worker (45-46)](#reglas-imagenes-multiworker)
- [🆕 Reglas de Código Limpio (47-48)](#reglas-codigo-limpio)
- [🆕 Reglas de Tokens Wallapop (52-53)](#reglas-tokens-wallapop)
- [🆕 Reglas de Estado Unificado (54-55)](#reglas-estado-unificado)
- [🆕 v5.6: Reglas de Logs y Debug (56-57)](#reglas-logs-debug)
- [🆕 v5.6: Reglas de Tokens MitikLive (58-60)](#reglas-tokens-mitiklive)
- [🆕 v5.7: Reglas de Verificación de Versión (61-63)](#reglas-verificacion-version)

### 🔄 [FLUJOS COMPLETOS](#flujos-completos)
- [Flujo Completo de Publicación](#flujo-completo-de-publicacion)
- [Flujo de Backup](#flujo-de-backup)
- [Flujo de Créditos](#flujo-de-creditos)
- [Flujo de Eliminación](#flujo-de-eliminacion)
- [Flujo de Autenticación](#flujo-de-autenticacion)
- [Flujo Editor de Anuncios](#flujo-editor-de-anuncios)
- [Flujo de Cuentas Wallapop](#flujo-de-cuentas-wallapop)
- [Flujo de WebSocket/SSE](#flujo-de-websocket)
- [🆕 v5.7: Flujo de Verificación de Versión](#flujo-verificacion-version)

### 📦 [MÓDULOS CRÍTICOS](#modulos-criticos)
- [Publishing Monitor](#publishing-monitor)
- [Progress Tab](#progress-tab)
- [Listings Module](#listings-module)
- [Button Manager](#button-manager)

### ✅ [CHECKLISTS Y VALIDACIÓN](#checklists)
- [Checklist CSS](#checklist-css)
- [Checklist JavaScript](#checklist-javascript)
- [Checklist Backend](#checklist-backend)
- [Checklist Publicación](#checklist-publicacion)

### 🚫 [ERRORES COMUNES](#errores-comunes)

---

# FUNDAMENTOS

## 🎯 FLUJO DE TRABAJO OBLIGATORIO

```
1. ❓ ¿Necesito una función/estilo/componente?
   ↓
2. 🔍 Buscar PRIMERO en archivos centralizados
   ↓
3. ✅ ¿Existe? → REUTILIZAR (importar)
   ↓
4. ❌ ¿No existe? → Añadir al archivo centralizado apropiado
   ↓
5. 🚫 NUNCA crear código duplicado inline
   ↓
6. 📝 Documentar en esta guía si es funcionalidad crítica
```

---

## 🏗️ ARQUITECTURA GLOBAL MULTI-USUARIO

### Sistema de Aislamiento Total

```
Usuario A (JWT Token A, user_id: 1)
  └─ walla_accounts
       ├─ account_id: 1 (Cuenta Personal)
       │    └─ listings (100 anuncios)
       │         └─ /data/products/1/1/id_wallapop_xxx/
       │              ├─ image_1.jpg
       │              └─ image_2.jpg
       │
       └─ account_id: 2 (Cuenta Profesional)
            └─ listings (250 anuncios)
                 └─ /data/products/1/2/id_wallapop_yyy/

Usuario B (JWT Token B, user_id: 2)
  └─ walla_accounts  
       └─ account_id: 100
            └─ listings (50 anuncios)
                 └─ /data/products/2/100/id_wallapop_zzz/
```

### 🔐 REGLAS CRÍTICAS DE AISLAMIENTO

**1. TODO query a `listings` DEBE filtrar por `account_id`:**
```sql
-- ✅ CORRECTO
SELECT * FROM listings 
WHERE account_id = :account_id AND status = 'active'

-- ❌ INCORRECTO (vulnerabilidad de seguridad)
SELECT * FROM listings WHERE id_wallapop = :id
```

**2. Backend SIEMPRE verifica propiedad:**
```python
# ✅ Verificar que la cuenta pertenece al usuario
account = db.query(WallaAccount).filter(
    WallaAccount.id == account_id,
    WallaAccount.user_id == current_user.id
).first()

if not account:
    raise HTTPException(403, "No tienes acceso a esta cuenta")
```

**3. Rutas de archivos SIEMPRE incluyen user_id y account_id:**
```python
# ✅ CORRECTO
/data/products/{user_id}/{account_id}/{id_wallapop}/image.jpg

# ❌ INCORRECTO (posible acceso cruzado)
/data/products/{id_wallapop}/image.jpg
```

---

## 🖼️ CENTRALIZACIÓN DE URLS DE IMÁGENES

### Problema Resuelto
Antes había múltiples formas de construir URLs de imágenes dispersas por el código:
- Backend: `/public/`, `/files/`, `/data/`, construcciones manuales
- Extensión: Construcciones manuales con `${apiBase}/public/`, inconsistencias

### Solución Implementada

#### Backend - Helper Centralizado

**Archivo:** `/var/www/mitiklive/backend/app/image_helpers.py`

```python
"""
Helpers centralizados para URLs de imágenes
Proporciona funciones para convertir rutas locales a URLs públicas servidas por Nginx
"""

def build_image_url(path: Optional[str]) -> Optional[str]:
    """
    Convierte una ruta local a URL pública para Nginx
    
    Examples:
        >>> build_image_url("/data/products/1/123/img.jpg")
        "/data/products/1/123/img.jpg"
        
        >>> build_image_url("products/1/123/img.jpg")
        "/data/products/1/123/img.jpg"
    """
    if not path:
        return None
    
    # Si ya tiene /data/, retornar tal cual
    if path.startswith('/data/'):
        return path
    
    # Si empieza con /, retornar tal cual
    if path.startswith('/'):
        return path
    
    # Path relativo → agregar /data/
    return f"/data/{path}"

def build_images_urls(paths: Optional[List[str]]) -> List[str]:
    """Convierte lista de rutas locales a URLs públicas"""
    # ... (ver código completo en image_helpers.py)

def get_first_image_url(images_local_json: Optional[str]) -> Optional[str]:
    """Obtiene URL de la primera imagen desde images_local JSON"""
    # ... (ver código completo en image_helpers.py)
```

#### Endpoints Backend Actualizados

**1. `routers/walla_runs.py` - Endpoint `by-account`**
```python
from ..image_helpers import build_image_url, get_first_image_url

# En el endpoint /listings/by-account:
rel = row[11] or None  # first_image (path from images_local[0])
# ✅ Usar helper centralizado
first_image_url = build_image_url(rel)
```

**2. `routers/listing_editor.py` - Endpoint `GET /{listing_id}`**
```python
from ..image_helpers import build_images_urls

# Al retornar listing:
images_local_urls = build_images_urls(images_local_str)

return {
    "listing": {
        # ...
        "images_local": images_local_urls  # ✅ URLs públicas (/data/...)
    }
}
```

**3. `routers/publish_simple.py` - Endpoint `next-item`**
```python
from ..image_helpers import get_first_image_url

# Para monitor-state (thumbnail):
if not image_url and row["images_local"]:
    image_url = get_first_image_url(row["images_local"]) or ""

# Para publicación (normalizar paths):
for idx, path in enumerate(images_local):
    # Normalizar: quitar /data/ si existe
    normalized_path = path.replace('/data/', '') if path.startswith('/data/') else path
    normalized_path = normalized_path.lstrip('/')
    
    # Construir URL para extensión: /data/products/...
    image_url = f"/data/{normalized_path}"
```

#### Extensión - Función Centralizada

**Archivo:** `scripts/panel/listings/helpers.js`

```javascript
/**
 * Convierte path de imagen a URL completa
 * ÚNICA función para construcción de URLs de imágenes
 * 
 * @param {string} path - Path de imagen (puede ser URL completa o relativa)
 * @param {string} apiBase - API_BASE del servidor
 * @returns {string} URL completa de imagen
 */
export function getImageUrl(path, apiBase = '') {
  if (!path) return '';
  
  // Si ya es URL completa, retornar tal cual
  if (/^https?:\/\//i.test(path)) return path;
  
  // Intentar obtener API_BASE desde config centralizado
  if (!apiBase && isInitialized()) {
    apiBase = getApiBase();
  }
  
  // Si no hay API_BASE, no podemos construir URL
  if (!apiBase) return '';
  
  // Si empieza con /, concatenar directamente con API_BASE
  if (path.startsWith('/')) return `${apiBase}${path}`;
  
  // Path relativo → agregar /data/ (Nginx)
  return `${apiBase}/data/${path}`;
}
```

**Archivos que usan `getImageUrl()`:**
- ✅ `listings/render.js` - Miniaturas en tabla de anuncios
- ✅ `progress-tab.js` - Imágenes en modal de progreso
- ✅ `listing-editor/image-manager.js` - Modal de edición
- ✅ `listings/websocket.js` - Actualizaciones en tiempo real

#### Formato Único de URLs

**Todo el sistema usa UN SOLO formato:**

```
Backend retorna:    /data/products/1/1166/xyz/img_000.jpg
Extensión usa:      https://mitiklive.com/data/products/1/1166/xyz/img_000.jpg
Nginx sirve desde:  /var/www/mitiklive/data/products/1/1166/xyz/img_000.jpg
```

#### Beneficios

1. ✅ **Una sola fuente de verdad** - Cambios en un lugar
2. ✅ **Sin inconsistencias** - Imposible tener URLs rotas por construcción manual
3. ✅ **Fácil mantenimiento** - Modificar formato en helpers.py y helpers.js
4. ✅ **Testing simplificado** - Solo testear 3 funciones
5. ✅ **Migraciones fáciles** - Cambiar servidor → cambiar helpers

#### Regla de Oro #45

**NUNCA construir URLs de imágenes manualmente**

❌ **INCORRECTO:**
```python
# Backend
url = f"/public/{path}"
url = f"{base}/files/{img}"

# JavaScript
url = `${apiBase}/public/${path}`
url = baseUrl + "/data/" + filename
```

✅ **CORRECTO:**
```python
# Backend
from ..image_helpers import build_image_url
url = build_image_url(path)

# JavaScript
import { getImageUrl } from './listings/helpers.js';
const url = getImageUrl(path, apiBase);
```

---

## 🧹 REGLAS DE CÓDIGO LIMPIO (47-48)

### Regla de Oro #47

**NUNCA crear funciones de log personalizadas (logg, _log, etc.)**

Las funciones de debug custom causan errores cuando se eliminan parcialmente.

❌ **PROHIBIDO:**
```javascript
// Esto causa problemas al eliminar los logs
const logg = (...a) => log && console.log('[PUB]', ...a);
const _log = (...a) => { if (log) console.log('[FILL]', ...a); };

logg('mensaje debug');
_log('otro mensaje');
```

✅ **CORRECTO:**
```javascript
// Opción 1: No loggear en producción
// (simplemente no poner logs)

// Opción 2: Usar console directamente si necesitas debug
console.log('[PUB] mensaje debug');

// Opción 3: Usar el logger centralizado del panel
const logger = await getLogger();
logger.info('mensaje', { context: data });
```

**Razón:** Durante la migración v10.x se eliminaron logs de debug pero quedaron llamadas a funciones no definidas (`logg is not defined`, `_log is not defined`). Esto causó múltiples errores en cascada.

---

### Regla de Oro #48

**SIEMPRE verificar sintaxis JavaScript antes de empaquetar**

❌ **PROHIBIDO:**
```bash
# Empaquetar sin verificar
zip -r extension.zip extension/
```

✅ **OBLIGATORIO:**
```bash
# Verificar TODOS los archivos JS antes de empaquetar
cd extension/
for f in $(find . -name "*.js" -type f); do
  result=$(node --check "$f" 2>&1)
  if [ -n "$result" ]; then
    echo "❌ ERROR en $f:"
    echo "$result"
    exit 1
  fi
done
echo "✅ Todos los JS válidos"

# Solo entonces empaquetar
zip -r extension.zip .
```

**Errores que detecta:**
- `SyntaxError: Illegal return statement` - Return fuera de función
- `SyntaxError: Unexpected token` - Objetos o código huérfano
- `SyntaxError: Unexpected identifier` - Variables no declaradas

**Caso real v10.x:** Las funciones `fillBrandCombo`, `fillModelCombo`, `fillStorageCapacityCombo` perdieron sus declaraciones `async function nombre() {` durante una edición. El código tenía el cuerpo pero no la declaración, causando "Illegal return statement".

---

## 🔑 GESTIÓN DE TOKENS WALLAPOP (v10.5.54)

### Problema

Wallapop utiliza tokens de autenticación que pueden:
1. **Expirar por tiempo** - Después de cierto período
2. **Ser invalidados server-side** - Por sesiones cerradas, detección de actividad sospechosa, límites de sesiones activas

El sistema de publicación puede fallar con errores `401 Unauthorized` si el token se invalida durante el proceso.

### Solución Implementada

**Sistema de 3 capas:**

1. **Captura automática de tokens** - Interceptor en content_script.js
2. **Refresh preventivo** - Antes de operaciones críticas si token > 90s
3. **Reintento con token fresco** - Si falla con 401, refresh y reintentar

### Arquitectura de Archivos

```
sw/
├── walla.js                    ⭐⭐⭐ GESTIÓN DE TOKENS
│   ├── wallaCreds              → Token en memoria
│   ├── getTokenDiagnostics()   → Diagnóstico de token
│   ├── getTokenAge()           → Edad del token actual
│   ├── ensureWallaCreds()      → Obtener/refrescar token
│   ├── refreshWallaCredsIfOld()⭐ Refresh si viejo (>90s)
│   └── invalidateWallaCredsInSW() → Invalidar token

service_worker.js
├── import { refreshWallaCredsIfOld } ⭐ Import ESTÁTICO
└── TOKEN.FORCE_REFRESH handler  ⭐ Para CS → SW

sw/handlers/
└── publish-process-next.js
    ├── DELETE con reintento 401 ⭐
    └── PUBLICAR con refresh preventivo

content_script.js
└── Click publicar con TOKEN.FORCE_REFRESH ⭐
```

### Flujo de Token en Publicación

```
┌─────────────────────────────────────────────────────────────────────────┐
│                          FLUJO DELETE                                    │
├─────────────────────────────────────────────────────────────────────────┤
│                                                                          │
│  1. Token fresco? (>90s)                                                │
│     │                                                                    │
│     ├── SÍ viejo → refreshWallaCredsIfOld() → Captura nuevo token       │
│     │                                                                    │
│     └── NO (fresco) → Continuar                                         │
│                                                                          │
│  2. DELETE /api/v3/items/{id}                                           │
│     │                                                                    │
│     ├── 204 OK → Continuar con publicación                              │
│     │                                                                    │
│     ├── 401 → refreshWallaCredsIfOld(tabId, 0) → Reintentar DELETE     │
│     │         │                                                          │
│     │         ├── 204 OK → Continuar                                    │
│     │         │                                                          │
│     │         └── 401 → Error (token inválido persistente)              │
│     │                                                                    │
│     └── 404/410 → OK (ya no existe, continuar)                          │
│                                                                          │
└─────────────────────────────────────────────────────────────────────────┘

┌─────────────────────────────────────────────────────────────────────────┐
│                         FLUJO PUBLICAR (CS)                             │
├─────────────────────────────────────────────────────────────────────────┤
│                                                                          │
│  1. Token fresco? (>90s en SW)                                          │
│     │                                                                    │
│     └── Igual que DELETE                                                │
│                                                                          │
│  2. Click en botón publicar                                             │
│     │                                                                    │
│     ├── Éxito → Captura ID nuevo                                        │
│     │                                                                    │
│     └── Error servidor (401) detectado                                  │
│         │                                                                │
│         ├── Enviar TOKEN.FORCE_REFRESH al SW                            │
│         │   └── SW ejecuta refreshWallaCredsIfOld(tabId, 0)             │
│         │                                                                │
│         ├── Esperar 15s                                                 │
│         │                                                                │
│         └── Reintentar click (máx 3 reintentos)                         │
│                                                                          │
└─────────────────────────────────────────────────────────────────────────┘
```

### Código Crítico

#### sw/walla.js - refreshWallaCredsIfOld()

```javascript
/**
 * Refresca token de Wallapop si es viejo
 * @param {number} tabId - Tab ID de Wallapop
 * @param {number} maxAgeMs - Edad máxima en ms (default 90000 = 90s)
 *                           Usar 0 para forzar refresh siempre
 * @returns {Promise<{ok: boolean, refreshed: boolean, creds: object}>}
 */
export async function refreshWallaCredsIfOld(tabId, maxAgeMs = 90000) {
  const age = getTokenAge();
  const isOld = !age.ms || age.ms > maxAgeMs;
  
  console.log(`[TOKEN] 🔍 Verificando edad: ${age.formatted} | Límite: ${maxAgeMs/1000}s | ¿Viejo?: ${isOld}`);
  
  if (!isOld && wallaCreds.token && wallaCreds.deviceId) {
    // Token es reciente, usarlo
    return { ok: true, refreshed: false, creds: wallaCreds };
  }
  
  // Token viejo o no existe - intentar refrescar
  console.log('[TOKEN] 🔄 Token viejo, intentando refrescar...');
  
  // 1. Invalidar token viejo
  wallaCreds = { token: null, deviceId: null, appVersion: null, deviceOs: null, ts: 0 };
  await chrome.storage.local.remove([WCREDS_KEY]);
  
  // 2. Enviar mensaje al CS para forzar petición que capture token nuevo
  try {
    await chrome.tabs.sendMessage(tabId, { type: 'CAPTURE.FORCE_REFRESH' });
    await new Promise(r => setTimeout(r, 500));
  } catch (e) {
    console.warn('[TOKEN] ⚠️ No se pudo enviar CAPTURE.FORCE_REFRESH:', e.message);
  }
  
  // 3. Hacer kick tradicional también
  try {
    await chrome.tabs.sendMessage(tabId, { type: 'CAPTURE.KICK_ONCE' });
  } catch (e) {
    // Ignorar - en /upload no hay "Cargar más"
  }
  
  // 4. Esperar a que se capture token nuevo (máx 5s)
  const t0 = Date.now();
  while (Date.now() - t0 < 5000) {
    if (wallaCreds.token && wallaCreds.deviceId) {
      console.log(`[TOKEN] ✅ Token refrescado! Nueva edad: ${getTokenAge().formatted}`);
      return { ok: true, refreshed: true, creds: wallaCreds };
    }
    // Verificar en storage
    const st = await stGet([WCREDS_KEY]);
    if (st[WCREDS_KEY]?.token && st[WCREDS_KEY]?.deviceId) {
      wallaCreds = st[WCREDS_KEY];
      return { ok: true, refreshed: true, creds: wallaCreds };
    }
    await new Promise(r => setTimeout(r, 300));
  }
  
  console.error('[TOKEN] ❌ No se pudo refrescar token después de 5s');
  return { ok: false, refreshed: false, creds: null };
}
```

#### service_worker.js - Handler TOKEN.FORCE_REFRESH

```javascript
// ⚠️ CRÍTICO: Import ESTÁTICO (import() dinámico prohibido en SW MV3)
import { refreshWallaCredsIfOld } from './sw/walla.js';

// Handler para forzar refresh de token desde CS
if (message.type === 'TOKEN.FORCE_REFRESH') {
  (async () => {
    try {
      const tabs = await chrome.tabs.query({ url: '*://*.wallapop.com/*', active: true });
      const tabId = tabs[0]?.id || sender.tab?.id;
      
      if (!tabId) {
        sendResponse({ ok: false, error: 'No se encontró tab de Wallapop' });
        return;
      }
      
      // Forzar refresh (0 = ignorar edad)
      const result = await refreshWallaCredsIfOld(tabId, 0);
      console.log(`[SW] 🔄 TOKEN.FORCE_REFRESH resultado:`, result.ok ? '✅' : '❌');
      sendResponse({ ok: result.ok, refreshed: result.refreshed });
    } catch (e) {
      console.error('[SW] ❌ Error en TOKEN.FORCE_REFRESH:', e.message);
      sendResponse({ ok: false, error: e.message });
    }
  })();
  return true;
}
```

#### publish-process-next.js - DELETE con reintento

```javascript
// Función helper para DELETE
const doDelete = async (token, deviceId) => {
  return await fetch(deleteUrl, {
    method: 'DELETE',
    headers: {
      'Accept': 'application/json',
      'Authorization': `Bearer ${token}`,
      'X-DeviceId': deviceId
    }
  });
};

let deleteRes = await doDelete(creds.token, creds.deviceId);

// Si 401, refrescar token y reintentar UNA vez
if (deleteRes.status === 401) {
  console.warn(`[TOKEN] ⚠️ DELETE 401 - Refrescando token y reintentando...`);
  
  // Forzar refresh (ignorar edad)
  const refreshResult = await refreshWallaCredsIfOld(tab.id, 0);
  
  if (refreshResult.ok && refreshResult.creds?.token) {
    console.log(`[TOKEN] 🔄 Token refrescado, reintentando DELETE...`);
    deleteRes = await doDelete(refreshResult.creds.token, refreshResult.creds.deviceId);
    console.log(`[TOKEN] 📊 DELETE reintento: HTTP ${deleteRes.status}`);
  }
}
```

#### content_script.js - Publicar con refresh

```javascript
// Si fue error de servidor, refrescar token y reintentar
if (serverErrorDetected && serverErrorRetries < MAX_SERVER_ERROR_RETRIES) {
  serverErrorRetries++;
  
  // Pedir al SW que refresque el token antes de reintentar
  console.log(`[ML] 🔄 Solicitando refresh de token antes de reintento...`);
  try {
    const refreshRes = await chrome.runtime.sendMessage({ type: 'TOKEN.FORCE_REFRESH' });
    if (refreshRes?.ok) {
      console.log(`[ML] ✅ Token refrescado correctamente`);
    } else {
      console.warn(`[ML] ⚠️ No se pudo refrescar token:`, refreshRes?.error);
    }
  } catch (e) {
    console.warn(`[ML] ⚠️ Error pidiendo refresh de token:`, e.message);
  }
  
  console.log(`[ML] ⏳ Esperando 15s antes de reintentar...`);
  await new Promise(r => setTimeout(r, SERVER_ERROR_WAIT_MS));
  continue; // Reintentar
}
```

### Logs de Diagnóstico

**Token válido (flujo normal):**
```
[TOKEN] 📊 ANTES de DELETE #4: {edad: '2s', token: 'eyJhbG...', anunciosProcesados: 4}
[TOKEN] 🔍 Verificando edad: 2s | Límite: 90s | ¿Viejo?: false
[TOKEN] 📊 DELETE resultado: HTTP 204 | Token edad: 0s | Anuncios: 4
[PUBLISH] ✅ Anuncio #5 publicado | Tiempo ciclo: 193s
```

**Token expirado (401 → refresh → reintento):**
```
[TOKEN] 📊 DELETE resultado: HTTP 401 | Token edad: 95s
[TOKEN] ⚠️ DELETE 401 - Refrescando token y reintentando...
[TOKEN] 🔄 Token viejo, intentando refrescar...
[TOKEN] 🔄 NUEVO token capturado! (eyJhbG...)
[TOKEN] ✅ Token refrescado! Nueva edad: 0s
[TOKEN] 🔄 Token refrescado, reintentando DELETE...
[TOKEN] 📊 DELETE reintento: HTTP 204 | Token edad: 0s
```

**Error de servidor en publicación (CS):**
```
api.wallapop.com/api/v3/items:1 Failed to load resource: 401 (Unauthorized)
[ML] ⚠️ ERROR SERVIDOR detectado después de 1s
[ML] 📊 Hora: 22:52:48 | Reintento: 1/3
[ML] 🔄 Solicitando refresh de token antes de reintento...
[SW] 🔄 TOKEN.FORCE_REFRESH resultado: ✅
[ML] ✅ Token refrescado correctamente
[ML] ⏳ Esperando 15s antes de reintentar...
[ML] 🖱️ Click en publicar (intento 2/4)
```

### Reglas de Tokens

#### REGLA #52: Siempre refrescar token antes de operaciones críticas

```javascript
// ✅ BIEN: Verificar y refrescar si >90s antes de DELETE/PUBLISH
const refreshResult = await refreshWallaCredsIfOld(tab.id, 90000);
if (refreshResult.refreshed) {
  console.log(`[TOKEN] 🔄 Token refrescado antes de operación`);
}

// ❌ MAL: Usar token sin verificar edad
const creds = getWallaCreds();
await fetch(url, { headers: { Authorization: `Bearer ${creds.token}` }});
```

#### REGLA #53: Siempre reintentar con token fresco en 401

```javascript
// ✅ BIEN: En 401, refrescar y reintentar UNA vez
if (response.status === 401) {
  const refreshResult = await refreshWallaCredsIfOld(tabId, 0); // 0 = forzar
  if (refreshResult.ok) {
    response = await retry(refreshResult.creds);
  }
}

// ❌ MAL: Reintentar con mismo token
if (response.status === 401) {
  await sleep(1000);
  response = await retry(sameToken); // ¡Fallará otra vez!
}
```

### Consideraciones Importantes

1. **Import() dinámico prohibido en SW MV3**
   - ❌ `const { fn } = await import('./module.js');`
   - ✅ `import { fn } from './module.js';` (estático)

2. **Delays largos (>90s)**
   - El refresh preventivo de 90s ya cubre delays largos configurados
   - Token se refresca automáticamente si tiene >90s al iniciar operación

3. **Wallapop puede invalidar tokens en cualquier momento**
   - No podemos predecir invalidación server-side
   - Solo podemos detectar (401) y reintentar

4. **Máximo 1 reintento con token nuevo**
   - Si falla 2 veces con tokens frescos, hay otro problema
   - Evita loops infinitos de refresh

---

## 🔄 REGLAS DE ESTADO UNIFICADO (54-55) - v10.5.82

### REGLA #54: Siempre importar selectAllByFilter desde module-state.js

**selectAllByFilter fue MOVIDO de state-manager.js a module-state.js**

```javascript
// ✅ CORRECTO - Importar desde module-state.js
import { 
  getSelectAllByFilter, 
  setSelectAllByFilter, 
  clearSelectAllByFilter 
} from './scripts/panel/listings/module-state.js';

const isSelectAll = getSelectAllByFilter();

// ❌ INCORRECTO - Ya NO existe en state-manager.js
import { getSelectAllByFilter } from './scripts/panel/state-manager.js'; // ❌ ERROR
AppState.getSelectAllByFilter(); // ❌ ERROR - función eliminada
```

**Archivos que DEBEN usar este import:**
- `panel.js` ✅
- `delete-handler.js` ✅  
- `listings/events.js` ✅
- `listings/index.js` (usa `State.getSelectionState()`)

### REGLA #55: Usar getSelectionState() para obtener estado completo

**NUNCA duplicar estado de selección - usar fuente única**

```javascript
// ✅ CORRECTO - Usar getSelectionState() que retorna todo
import * as State from './listings/module-state.js';

function updateSelectionUI() {
  const selState = State.getSelectionState();
  // selState contiene: filter, selectedCount, selectedActive, selectedDeleted,
  //                    totalActive, totalDeleted, isSelectAll, selectAllByFilter
  
  ButtonManager.updateState({
    ...selState,
    dbTotalActive: dbStats.active,
    dbTotalDeleted: dbStats.deleted
  });
}

// ❌ INCORRECTO - Duplicar lógica de conteo
function updateSelectionUI() {
  const selectedIds = State.getSelectedIds();
  const listings = State.getCurrentListings();
  
  // ❌ NO hacer esto - ya existe en getSelectionState()
  let selectedActive = 0;
  selectedIds.forEach(id => {
    const listing = listings.find(l => l.listing_id === id);
    if (listing?.status === 'active') selectedActive++;
  });
}
```

### Funciones ELIMINADAS de state-manager.js (v10.5.82)

| Función eliminada | Reemplazo |
|-------------------|-----------|
| `getSelectAllByFilter()` | `import { getSelectAllByFilter } from './listings/module-state.js'` |
| `setSelectAllByFilter()` | `import { setSelectAllByFilter } from './listings/module-state.js'` |
| `clearSelectAllByFilter()` | `import { clearSelectAllByFilter } from './listings/module-state.js'` |
| `getSelectionInfo()` | `State.getSelectionState()` en module-state.js |

### Checklist de migración de estado

Si modificas código relacionado con selección de listings:

- [ ] ¿Importas `selectAllByFilter` desde `module-state.js`?
- [ ] ¿Usas `getSelectionState()` en lugar de múltiples getters?
- [ ] ¿Llamas `clearAllSavedSelections()` al cambiar de cuenta?
- [ ] ¿NO duplicas estado en otros archivos?
- [ ] ¿El estado fluye: module-state → index.js → button-manager?

---

#### Troubleshooting

**Problema:** WebSocket no recibe eventos
```bash
# Verificar Redis
redis-cli ping

# Ver si workers usan Redis
sudo journalctl -u mitiklive-backend -f | grep Redis

# Reiniciar todo
sudo systemctl restart redis-server
sudo systemctl restart mitiklive-backend
```

**Problema:** "Error publicando en Redis"
```bash
# Ver logs
sudo journalctl -u mitiklive-backend -n 100 | grep Redis

# Verificar permisos
ls -la /var/run/redis/

# Reiniciar Redis
sudo systemctl restart redis-server
```

---

## 📝 NOTAS DE ACTUALIZACIÓN v4.0.2

**Fecha:** 2024-12-03  
**Cambios principales:**

1. ✅ **Botón cancelar backup atascado** - Config → Mantenimiento
2. ✅ **Endpoint `/api/system/maintenance/cancel-stuck-backup/{account_id}`**
3. ✅ **Confirmación antes de cancelar** - Modal de confirmación
4. ✅ **Toast informativo** - Feedback visual del resultado
5. ✅ **Cumple TODAS las reglas de la guía** (REGLA #1, #2, #4, #5)

**Archivos modificados:**
- `panel.html` - Botón `cancel-backup-btn` en sección Mantenimiento (línea ~564)
- `scripts/panel/config.js` - Función `initCancelBackupButton()` (60 líneas)
- `scripts/panel/config.js` - Llamada en `initConfig()` (línea ~54)
- `manifest.json` - Versión actualizada a 4.0.2

**Funcionalidad:**

```javascript
// Botón en panel.html
<button id="cancel-backup-btn" class="btn btn-danger" type="button" data-auth>
  🛑 Cancelar backup atascado
</button>

// Función en config.js
function initCancelBackupButton() {
  // 1. Obtener accountId activo
  // 2. Confirmar con usuario
  // 3. POST /api/system/maintenance/cancel-stuck-backup/{accountId}
  // 4. Toast con resultado
  // 5. Actualizar UI
}
```

**Seguridad:**
- ✅ Requiere autenticación JWT
- ✅ Verifica ownership de cuenta (`verify_account_ownership`)
- ✅ SQL solo afecta backups del usuario: `WHERE account_id = :account_id AND status = 'running'`

**Backend requerido:** v3.0.0+

**Escenario de uso:**
```
Problema: Backup atascado en "running" (backend reinició o proceso murió)
Solución: Config → Mantenimiento → Click "🛑 Cancelar backup atascado"
Resultado: Backup marcado como "cancelled", puede iniciar uno nuevo
```

**Diferencia con "Limpiar colgados":**

| Característica | Limpiar colgados | Cancelar backup |
|----------------|------------------|-----------------|
| **Ámbito** | Publicación | Backup |
| **Tabla** | `walla_listings` | `backup_runs` |
| **Status origen** | `processing` | `running` |
| **Status destino** | `queued_delete` | `cancelled` |
| **Endpoint** | `/api/walla/publish2/reset-stuck` | `/api/system/maintenance/cancel-stuck-backup/{id}` |

**Versiones:**
- Extensión: v4.0.2
- Backend: v3.0.0+ (endpoint ya existe)
- Guía: v5.8 (compatible)

---

## 📝 NOTAS DE ACTUALIZACIÓN v5.8

**Fecha:** 2025-11-30  
**Cambios principales:**

1. ✅ **Verificación de versión SIN autenticación** (v2.2.3) - Se verifica ANTES de todo
2. ✅ **Flujo post-login centralizado** - `onLoginSuccess()` para todos los providers
3. ✅ **Sistema de pausa en publicación** (v2.2.0) - Botón ⏸️/▶️ con 4 checkpoints
4. ✅ **Modal de pausa en Wallapop** (v2.2.2) - Sin fader, sincronización bidireccional
5. ✅ **Reintentos de iframe para token** (v2.2.1) - 3 intentos, 5s cada uno
6. ✅ **Modal para procesos colgados** (v2.2.2) - Limpiar y continuar automáticamente
7. ✅ **Nueva REGLA #64** - Flujo post-login centralizado
8. ✅ **Nueva REGLA #65** - Sistema de pausa en publicación
9. ✅ **REGLA #62 actualizada** - Ahora verificación es ANTES de login

**Archivos modificados:**
- `panel.js` - `checkVersionWithoutAuth()` al inicio de `initPanel()`
- `scripts/panel/auth.js` - `onLoginSuccess()` centralizado, eliminada verificación duplicada
- `scripts/panel/maintenance.js` - `showUpdateModal` exportado
- `scripts/panel/progress-tab.js` - Sistema de pausa con checkpoints
- `sw/handlers/publish-process-next.js` - `checkPauseAndWait()` en 4 puntos estratégicos
- `sw/walla.js` - Reintentos de iframe (3 intentos)
- `content_script.js` - `showPauseOverlay()`, `hidePauseOverlay()`
- `styles/mitiklive.css` - Estilos modal de pausa

**Nuevos mensajes del sistema:**
- `PUBLISH.SET_PAUSED` - Panel/CS → SW para pausar/reanudar
- `PUBLISH.PAUSED_CHANGED` - SW → Panel/CS broadcast de cambio de estado
- `SHOW_PAUSE_OVERLAY` - Panel → CS para mostrar modal en Wallapop
- `HIDE_PAUSE_OVERLAY` - Panel → CS para ocultar modal

**Flujo de verificación de versión (NUEVO):**
```
Panel se abre
    ↓
checkVersionWithoutAuth() ← SIN login, fetch directo
    ↓
¿Desactualizada? → Modal bloqueante (STOP)
    ↓
OK → Continuar init (login, sesión activa, etc.)
```

**Versiones:**
- Backend: v2.9.35
- Extensión: v2.2.3
- Guía: v5.8

---

## 📝 NOTAS DE ACTUALIZACIÓN v5.4

**Fecha:** 2025-01-24  
**Cambios principales:**

1. ✅ **Sistema de gestión de tokens Wallapop** (v10.5.47-v10.5.54)
2. ✅ **Refresh preventivo de token** - Antes de DELETE y PUBLICAR si >90s
3. ✅ **Reintento con token fresco en 401** - DELETE y publicación
4. ✅ **Handler TOKEN.FORCE_REFRESH** - CS puede pedir refresh al SW
5. ✅ **Import estático obligatorio** - Fix error `import() disallowed in ServiceWorker`
6. ✅ **Mensajes de paso en progress-tab** - Feedback visual durante publicación
7. ✅ **Nuevas reglas #52-53** - Tokens Wallapop

**Archivos modificados en v10.5.47-v10.5.54:**
- `sw/walla.js` - `refreshWallaCredsIfOld()`, `getTokenDiagnostics()`, `getTokenAge()`
- `service_worker.js` - Import estático + handler `TOKEN.FORCE_REFRESH`
- `sw/handlers/publish-process-next.js` - DELETE con reintento 401, logs diagnóstico
- `content_script.js` - Envía `TOKEN.FORCE_REFRESH` antes de reintentos
- `scripts/panel/progress-tab.js` - Mensajes de paso dinámicos, interval contador tiempo
- `styles/progress-tab.css` - Estilos para mensajes de paso

**Nuevos mensajes del sistema:**
- `TOKEN.FORCE_REFRESH` - CS → SW para forzar refresh de token

**Logs añadidos:**
```
[TOKEN] 📊 ANTES de DELETE #N: {edad, token, anunciosProcesados}
[TOKEN] 🔍 Verificando edad: Xs | Límite: 90s | ¿Viejo?: true/false
[TOKEN] ⚠️ DELETE 401 - Refrescando token y reintentando...
[TOKEN] 🔄 Token refrescado, reintentando DELETE...
[ML] 🔄 Solicitando refresh de token antes de reintento...
[ML] ✅ Token refrescado correctamente
[SW] 🔄 TOKEN.FORCE_REFRESH resultado: ✅/❌
```

**Versiones:**
- Backend: v2.9.14
- Extensión: v10.5.54
- Guía: v5.4

---

## 📝 NOTAS DE ACTUALIZACIÓN v5.3

**Fecha:** 2025-11-24  
**Cambios principales:**

1. ✅ **success-modal.css → variables CSS (v10.5.28)** - Migrado de colores hardcoded a var(--)
2. ✅ **btnBackup → ButtonManager (v10.5.29)** - Centralizado en publish-button-manager.js
3. ✅ **waitFor unificado (v10.5.30-31)** - Eliminado dom-optimized.js, fusionado en dom.js
4. ✅ **BaseModal centralizado (v10.5.32)** - Nuevo componente reutilizable para modales
5. ✅ **Todos los pendientes completados** - 0 tareas pendientes

**Archivos nuevos:**
- `scripts/panel/base-modal.js` - Componente base para modales (overlay, ESC, animaciones)

**Archivos eliminados:**
- `scripts/dom-optimized.js` - Fusionado en dom.js

**Archivos modificados en v10.5.28-32:**
- `styles/success-modal.css` - Usa variables CSS
- `scripts/panel/ui.js` - Delega btnBackup a ButtonManager
- `scripts/panel/publish-button-manager.js` - Añadido estado backup y setAuthenticated()
- `scripts/dom.js` - Fusionado con dom-optimized.js (waitFor con MutationObserver)
- `scripts/publish/form-filler.js` - Imports actualizados a dom.js
- `scripts/panel/image-modal.js` - Usa base-modal.js
- `scripts/panel/success-modal.js` - Usa base-modal.js
- `manifest.json` - Eliminado dom-optimized.js de web_accessible_resources

**API de base-modal.js:**
```javascript
createModal({ id, content, closeOnEsc, closeOnOverlayClick, autoCloseMs, onOpen, onClose })
closeModal(id)
closeAllModals()
isModalOpen(id)
getModalOverlay(id)
```

**Versiones:**
- Backend: v2.9.14
- Extensión: v10.5.32
- Guía: v5.3

---

## 📝 NOTAS DE ACTUALIZACIÓN v5.2

**Fecha:** 2025-11-24  
**Cambios principales:**

1. ✅ **Reglas de imports ES6 vs Dinámicos (49-51)** - Documentación completa de cuándo usar cada tipo
2. ✅ **Auditoría de código completada** - Identificados duplicados y deuda técnica
3. ✅ **Limpieza de duplicados (v10.5.27)**:
   - showLoader/hideLoader eliminados de ui.js (usar utils.js)
   - escapeHtml centralizado en utils.js
   - waitFor de shared.js marcado como @deprecated
4. ✅ **Tabla de contextos MV3** - Service Worker, Content Script, Panel

**Archivos modificados en v10.5.27:**
- `scripts/panel/ui.js` - Eliminado showLoader/hideLoader duplicados
- `scripts/panel/logger.js` - Usa escapeHtml de utils.js
- `scripts/panel/listing-editor/modal.js` - Usa escapeHtml de utils.js
- `scripts/utils.js` - Añadido escapeHtml centralizado
- `scripts/publish.js` - Eliminado import no usado (sharedWaitFor)
- `scripts/publish/form-filler.js` - Eliminado import no usado (sharedWaitFor)
- `scripts/publish/shared.js` - waitFor marcado @deprecated

**Pendientes completados en v5.3:**
- [x] Unificar btnBackup de ui.js a ButtonManager ✅
- [x] Unificar waitFor de dom.js → usar dom-optimized.js ✅
- [x] Evaluar fusión de dom.js y dom-optimized.js ✅
- [x] success-modal.css usar variables CSS ✅
- [x] Evaluar crear BaseModal reutilizable ✅

**Versiones:**
- Backend: v2.9.14
- Extensión: v10.5.27
- Guía: v5.2

---

## 📝 NOTAS DE ACTUALIZACIÓN v5.1

**Fecha:** 2025-11-24  
**Cambios principales:**

1. ✅ **Flujo de publicación documentado al detalle** (21 pasos en SW + 7 secciones form-filler)
2. ✅ **Modal de éxito simplificado** - Solo aparece si 100% éxito (0 errores)
3. ✅ **Backup simplificado** - No actualiza tabla durante proceso, solo al finalizar
4. ✅ **Click en miniaturas** - Abre modal con imagen ampliada (selector `.listing-thumb`)
5. ✅ **Velocidad de publicación** - No muestra valores absurdos con 1 solo anuncio
6. ✅ **Toast delay mejorado** - Aparece inline junto al slider
7. ✅ **Fix `const` → `let`** en `missingRequired` (form-filler.js línea 860)
8. ✅ **web_accessible_resources** - Añadidos `dom-optimized.js` y `logger.js`

**Archivos modificados en v10.5.18-v10.5.26:**
- `scripts/panel/listings/websocket.js` - Simplificado backup (no toca tabla durante proceso)
- `scripts/panel/listings/render.js` - Añadido import y llamadas a initImageModalListeners
- `scripts/panel/image-modal.js` - Corregido selector `.listing-image` → `.listing-thumb`
- `scripts/panel/success-modal.js` - Solo muestra modal si 100% éxito
- `scripts/panel/progress-tab.js` - Velocidad "Calculando..." con <2 items, toast inline
- `scripts/publish/form-filler.js` - Fix const → let en missingRequired
- `styles/progress-tab.css` - CSS para toast inline
- `panel.html` - Mensaje delay movido dentro del contenedor
- `manifest.json` - Añadidos archivos a web_accessible_resources

**Versiones:**
- Backend: v2.9.14
- Extensión: v10.5.26
- Redis: 7.1.0+

---

## 📝 NOTAS DE ACTUALIZACIÓN v5.0

**Fecha:** 2025-11-23  
**Cambios principales:**
1. ✅ Restauradas declaraciones de funciones faltantes en `form-filler.js`
2. ✅ Eliminados TODOS los logs de debug custom (`logg`, `_log`, `log` locales)
3. ✅ Corregido fallback de `getLogger()` con todos los métodos
4. ✅ Nueva regla #47: No crear funciones de log custom
5. ✅ Nueva regla #48: Verificar sintaxis antes de empaquetar

**Funciones restauradas en `form-filler.js`:**
- `fillBrandCombo(brandValue, opts)` - Línea 56
- `fillModelCombo(modelValue, opts)` - Línea 229
- `fillStorageCapacityCombo(storageValue, opts)` - Línea 499

**Archivos limpiados de logs debug:**
- `scripts/publish.js` - Eliminados `log()` en `selectCategoryPath`
- `scripts/publish/form-filler.js` - Eliminados 127+ llamadas a `_log()`
- `scripts/publish/error-handler.js` - Eliminados `_log()` de retry logic

**Versiones:**
- Backend: v2.9.11+
- Extensión: v10.5.0
- Redis: 7.1.0+

---

## 📝 NOTAS DE ACTUALIZACIÓN v4.2

**Fecha:** 2025-11-22  
**Cambios principales:**
1. ✅ Centralización completa de URLs de imágenes (Backend + Extensión)
2. ✅ Redis Pub/Sub para WebSocket multi-worker
3. ✅ Gunicorn con 8 workers (optimización para 32 cores)
4. ✅ Nueva regla #45: URLs de imágenes centralizadas
5. ✅ Nueva regla #46: Redis obligatorio con múltiples workers

**Versiones:**
- Backend: v2.9.11+
- Extensión: v7.7.0+
- Redis: 7.1.0+

---

## 📁 EXTENSION - PANEL UI

### Estructura Completa de Archivos

```
extension/
│
├─ manifest.json                    ⭐ Configuración Chrome Extension
├─ panel.html                       ⭐ HTML principal del panel
├─ panel.js                         ⭐ Controlador principal
├─ service_worker.js                ⭐ Service Worker (background)
├─ content_script.js                ⭐ Inyectado en páginas Wallapop
│
├─ scripts/
│  │
│  ├─ 🔧 CORE UTILITIES (Reutilización total)
│  │  │
│  │  ├─ utils.js                   ⭐⭐⭐ FUNCIONES GLOBALES
│  │  │   ├─ toast(msg, type, opts)
│  │  │   ├─ showModal({title, html, buttons})
│  │  │   ├─ showLoader(text, opts)
│  │  │   ├─ hideLoader()
│  │  │   ├─ setLoaderText(text)
│  │  │   ├─ sleep(ms)              ⭐ USAR SIEMPRE (no setTimeout)
│  │  │   ├─ normalize(str)
│  │  │   ├─ truncateCP(str, max)
│  │  │   ├─ retryWithBackoff(fn, opts)
│  │  │   ├─ uuidv4()
│  │  │   ├─ cssEscape(s)
│  │  │   └─ aliasFromProfileUrl(url)
│  │  │
│  │  └─ dom.js                     ⭐⭐⭐ DOM MANIPULATION (Content Script)
│  │      ├─ waitFor(fn, opts)      ⭐ Esperar elemento DOM
│  │      ├─ retry(fn, opts)
│  │      ├─ clearThenSetText(el, text)
│  │      ├─ dispatchPointerClick(el)
│  │      ├─ norm(s)
│  │      └─ isOnline()
│  │
│  ├─ 🎛️ PANEL - Módulos Principales
│  │  │
│  │  ├─ logger.js                  ⭐⭐⭐ LOGGING CENTRALIZADO
│  │  │   ├─ logger.debug(msg, context?)
│  │  │   ├─ logger.info(msg, context?)
│  │  │   ├─ logger.warn(msg, context?)
│  │  │   ├─ logger.error(msg, context?)
│  │  │   └─ logger.setLevel(level)
│  │  │   
│  │  │   ❌ NUNCA: console.log() directo
│  │  │   ✅ SIEMPRE: import { logger } from './logger.js'
│  │  │
│  │  ├─ auth.js                    ⭐⭐⭐ AUTENTICACIÓN Y CUENTAS
│  │  ├─ ui.js                      ⭐⭐⭐ INTERFAZ DE USUARIO
│  │  ├─ config.js                  ⭐⭐ CONFIGURACIÓN
│  │  ├─ dom-utils.js               ⭐ DOM Utils (ml-hidden)
│  │  ├─ storage.js                 ⭐ Chrome Storage
│  │  │
│  │  ├─ progress-tab.js            ⭐⭐⭐ TAB MONITOR PUBLICACIÓN
│  │  │   ├─ Polling cada 2s
│  │  │   ├─ Overlay countdown
│  │  │   ├─ Estados en tiempo real
│  │  │   └─ Integrado con ButtonManager
│  │  │
│  │  ├─ publish-button-manager.js  ⭐⭐⭐ GESTIÓN BOTONES CENTRALIZADA
│  │  │   ├─ updateState(partialState)
│  │  │   ├─ getState()
│  │  │   ├─ refreshAllButtons()
│  │  │   └─ init()
│  │  │
│  │  └─ publishing-monitor.js      ⭐⭐ MONITOR LEGACY (WebSocket)
│  │
│  └─ 📦 LISTINGS - Gestión de Anuncios ⭐⭐⭐ MÓDULO CENTRALIZADO
│     │
│     ├─ index.js                   → Controlador principal
│     ├─ module-state.js            → Estado interno (account_id, etc)
│     │
│     ├─ render.js                  ⭐⭐⭐ RENDERIZADO CENTRALIZADO
│     │   ├─ renderListings()       → Renderizar tabla completa
│     │   ├─ renderStats()          → Renderizar estadísticas
│     │   ├─ incrementStatCounter(status) ⭐ Incrementar contador
│     │   ├─ renderListingRow()     → Renderizar fila individual
│     │   └─ updateListingRow()     → Actualizar fila existente
│     │
│     ├─ helpers.js                 ⭐⭐ FUNCIONES AUXILIARES
│     │   ├─ getImageUrl()          → URL imagen principal
│     │   ├─ formatPrice()          → Formato precio EUR
│     │   ├─ truncateTitle()        → Truncar títulos
│     │   ├─ getStatusBadge()       → Badge de estado
│     │   └─ formatDate()           → Formato fechas
│     │
│     ├─ websocket.js               → Eventos WebSocket (importa render.js)
│     ├─ filters.js                 → Filtros y búsqueda
│     ├─ selection.js               → Selección múltiple
│     └─ actions.js                 → Acciones (eliminar, editar)
│
├─ styles/
│  │
│  ├─ variables.css                 ⭐⭐⭐ VARIABLES GLOBALES (ÚNICA FUENTE)
│  │   ├─ Colores (backgrounds, text, brand, semánticos, overlays)
│  │   ├─ Espaciado (space-1 a space-16)
│  │   ├─ Tipografía (tamaños, pesos, line-heights)
│  │   ├─ Bordes (anchos, radios, colores)
│  │   ├─ Sombras (shadow-sm a shadow-2xl)
│  │   ├─ Transiciones (durations, easings)
│  │   └─ Z-index (z-base a z-max)
│  │
│  ├─ panel.css                     ⭐⭐ ESTILOS PRINCIPALES
│  ├─ config.css                    ⭐ Estilos configuración
│  ├─ progress-tab.css              ⭐ Estilos monitor publicación
│  └─ mitiklive.css                 → Estilos generales
│
└─ sw/                              → Service Worker modules
    ├─ api_client.js                ⭐⭐⭐ Cliente API & JWT
    ├─ constants.js                 ⭐⭐⭐ Constantes centralizadas
    ├─ utils.js                     → Utilidades SW
    └─ handlers/                    → Message handlers
```

---

# REGLAS DE ORO

## REGLAS FUNDAMENTALES (1-10)

### REGLA #1: SIEMPRE importar, NUNCA copiar/pegar
```javascript
// ❌ MAL: Copiar función existente
function sleep(ms) { return new Promise(r => setTimeout(r, ms)); }

// ✅ BIEN: Importar desde utils.js
import { sleep } from '../utils.js';
```

### REGLA #2: SIEMPRE usar variables CSS, NUNCA valores fijos
```css
/* ❌ MAL */
.mi-elemento {
  color: #e7ecff;
  background: #0e1628;
  padding: 16px;
}

/* ✅ BIEN */
.mi-elemento {
  color: var(--text-primary);
  background: var(--bg-secondary);
  padding: var(--space-4);
}
```

### REGLA #3: Preferir clases CSS sobre estilos inline

**Preferido - Usar clases CSS:**
```javascript
// ✅ PREFERIDO - Clases para estados
element.classList.add('ml-hidden');
element.classList.remove('ml-hidden');
element.classList.toggle('active');
```

**Estilos inline aceptables para:**
```html
<!-- ✅ ACEPTABLE - Valores dinámicos que no pueden ser clases -->
<div style="width: ${percentage}%"></div>
<img style="background-image: url(${dynamicUrl})">

<!-- ✅ ACEPTABLE - Posicionamiento calculado -->
<div style="top: ${calculatedTop}px"></div>
```

**En panel.html:** Existen ~21 estilos inline, la mayoría para layout base o valores dinámicos. Esto es aceptable.

### REGLA #4: Preferir logger.js sobre console.log

**Para logging estructurado usar logger.js:**
```javascript
// ✅ PREFERIDO - Logger estructurado con contexto
import { logger } from './panel/logger.js';
logger.info('Usuario autenticado', { userId: 123 });
logger.error('Error en publicación', { itemId, error: e.message });
```

**console.log es aceptable para:**
```javascript
// ✅ ACEPTABLE - Debug temporal en desarrollo
console.log('[DEBUG] valor:', variable);

// ✅ ACEPTABLE - Errores críticos en catch blocks
console.error('[CRITICAL] Error:', e);

// ✅ ACEPTABLE - Prefijos claros para identificar origen
console.log('[PUB][cat] mensaje');
```

**Evitar:**
```javascript
// ❌ EVITAR - Logs sin contexto
console.log(variable);
console.log('aquí');
```

### REGLA #5: SIEMPRE filtrar por account_id en queries a listings
```sql
-- ❌ MAL: Vulnerabilidad de seguridad
SELECT * FROM listings WHERE id_wallapop = ?

-- ✅ BIEN
SELECT * FROM listings WHERE account_id = ? AND id_wallapop = ?
```

### REGLA #6: SIEMPRE verificar propiedad del recurso en backend
```python
# ✅ BIEN
account = db.query(WallaAccount).filter(
    WallaAccount.id == account_id,
    WallaAccount.user_id == current_user.id
).first()

if not account:
    raise HTTPException(403, "Acceso denegado")
```

### REGLA #7: SIEMPRE usar funciones de utils.js
```javascript
// ❌ MAL
const result = await new Promise((resolve, reject) => {
  setTimeout(() => resolve(), 1000);
});

// ✅ BIEN
import { sleep } from '../utils.js';
await sleep(1000);
```

### REGLA #8: SIEMPRE preguntar "¿Ya existe?" antes de crear
**Antes de escribir cualquier función/clase/helper:**
1. Buscar en utils.js
2. Buscar en el módulo correspondiente
3. Buscar en helpers.js del módulo
4. Si no existe → Añadir al archivo centralizado

### REGLA #9: NUNCA exponer datos entre usuarios diferentes
**TODO endpoint debe:**
1. Autenticar usuario (JWT)
2. Verificar propiedad del recurso
3. Filtrar por user_id/account_id

### REGLA #10: NUNCA hacer queries globales sin filtro de aislamiento
```sql
-- ❌ MAL: Devuelve datos de TODOS los usuarios
SELECT COUNT(*) FROM listings

-- ✅ BIEN: Solo del usuario actual
SELECT COUNT(*) FROM listings WHERE account_id IN (
    SELECT id FROM walla_accounts WHERE user_id = ?
)
```

---

## REGLAS DE CENTRALIZACIÓN (11-16)

### REGLA #11: Centralización estricta de funciones relacionadas

**Si una función manipula X, TODAS las funciones que manipulan X deben estar en el MISMO módulo**

```javascript
// ❌ MAL: Funciones dispersas
// render.js
function renderStats(stats) { ... }

// websocket.js
function incrementStatCounter(status) { ... }  // ¡DUPLICACIÓN!

// ✅ BIEN: Todo centralizado
// render.js
export function renderStats(stats) { ... }
export function incrementStatCounter(status) { ... }

// websocket.js
import { renderStats, incrementStatCounter } from './render.js';
```

### REGLA #12: Manejo de títulos como objeto/string

**El campo `title` puede ser string u objeto `{original: "..."}`**

```javascript
// ✅ SIEMPRE hacer esto:
let titleText = item.title || '(sin título)';
if (typeof titleText === 'object' && titleText.original) {
  titleText = titleText.original;
}
const displayTitle = titleText.substring(0, 40);
```

### REGLA #13: NO crear registros vacíos en BD

**Al omitir items (coches, inmuebles), NO insertar registros vacíos**

```sql
-- ❌ MAL: Crea registro sin datos
INSERT IGNORE INTO listings (id_wallapop, account_id) VALUES (?, ?)

-- ✅ BIEN: Solo actualizar si ya existe
UPDATE listings 
SET backup_status = 'skipped' 
WHERE id_wallapop = ? AND account_id = ?
```

### REGLA #14: Migración de archivos al cambiar IDs

**Al cambiar `id_wallapop` (publicación), migrar carpeta física y actualizar rutas en BD**

```python
# Al publicar: old_id → new_id
# 1. Renombrar carpeta:
old_path = f"/data/products/{user_id}/{account_id}/{old_id}/"
new_path = f"/data/products/{user_id}/{account_id}/{new_id}/"
shutil.move(old_path, new_path)

# 2. Actualizar images_local en BD
UPDATE listings 
SET images_local = REPLACE(images_local, old_id, new_id)
WHERE id = ?
```

### REGLA #15: Espera activa de renderizado

**Antes de interactuar con combos dinámicos, esperar que tengan opciones**

```javascript
// ✅ BIEN: Esperar que combo tenga opciones cargadas
await waitFor(() => {
  const options = combo.querySelectorAll('li, option');
  return options.length > 0;
}, { timeout: 8000 });
```

### REGLA #16: Flags de estado en storage

**Para operaciones que requieren reload, usar flags en chrome.storage.local**

```javascript
// Marcar necesidad de acción
await chrome.storage.local.set({ needs_form_reset: true });

// Al inicio, verificar flag
const { needs_form_reset } = await chrome.storage.local.get(['needs_form_reset']);
if (needs_form_reset) {
  // Ejecutar acción
  await chrome.storage.local.set({ needs_form_reset: false });
}
```

---

## REGLAS DE PUBLICACIÓN (17-24)

### REGLA #17: Validar publish_idx antes de publicar

**SIEMPRE validar que publish_idx está configurado correctamente**

```python
# ✅ Antes de iniciar publicación
items_sin_idx = db.query(Listing).filter(
    Listing.account_id == account_id,
    Listing.publish_status == 'queued_publish',
    Listing.publish_idx.is_(None)
).count()

if items_sin_idx > 0:
    raise ValueError(f"{items_sin_idx} items sin publish_idx configurado")
```

### REGLA #18: Gestión de envíos - Parámetros exactos

**shipping_allowed debe coincidir con supports_shipping del backend**

```python
# Backend calcula:
supports_shipping = category_id not in SHIPPING_BLACKLIST

# Frontend debe enviar:
shipping_allowed = supports_shipping  # ⭐ Mismo valor

# Parámetros de shipping si shipping_allowed=true:
{
    "shipping_allowed": true,
    "shipping_cost_amount": 0,
    "shipping_cost_currency": "EUR"
}
```

### REGLA #19: Timeouts y reintentos progresivos

```javascript
// ✅ Timeouts según complejidad
const TIMEOUTS = {
  WAIT_FORM: 15000,        // Esperar formulario
  WAIT_COMBO: 8000,        // Esperar combo con opciones
  WAIT_IMAGE_UPLOAD: 30000 // Subir imagen
};

// ✅ Reintentos con backoff
await retryWithBackoff(async () => {
  // Operación que puede fallar
}, {
  maxRetries: 3,
  initialDelay: 1000,
  maxDelay: 5000,
  backoffFactor: 2
});
```

### REGLA #20: NO mezclar flujos de publicación

```javascript
// ❌ MAL: Iniciar nueva publicación si ya hay una activa
async function publishNext() {
  // Directamente publica
}

// ✅ BIEN: Validar estado primero
async function publishNext() {
  const status = await getPublishStatus(accountId);
  
  if (status.processing > 0) {
    return toast('⚠️ Ya hay un anuncio publicándose', 'warn');
  }
  
  // Continuar...
}
```

### REGLA #21: Actualizar publish_status atómicamente

```python
# ✅ Actualización atómica con transacción
with db.begin():
    item = db.query(Listing).filter(
        Listing.id == item_id,
        Listing.account_id == account_id
    ).with_for_update().first()
    
    item.publish_status = 'processing'
    item.publish_error = None
    db.commit()
```

### REGLA #22: Limpiar estado al terminar publicación

```javascript
// ✅ Al terminar (éxito o error)
async function cleanupAfterPublish() {
  // 1. Resetear estado UI
  ButtonManager.updateState({ processingCount: 0 });
  
  // 2. Limpiar flags
  await chrome.storage.local.remove(['current_publish_item']);
  
  // 3. Ocultar overlays
  hideLoader();
  
  // 4. Recargar lista
  await loadListings();
}
```

### REGLA #23: Log completo de cada paso

```python
# ✅ Log detallado para debugging
logger.info(f"[PUBLISH] Iniciando item {item.id}", extra={
    "account_id": account_id,
    "id_wallapop": item.id_wallapop,
    "title": item.title,
    "category_id": category_id
})

# ✅ Log de errores con contexto
logger.error(f"[PUBLISH] Error en item {item.id}", extra={
    "error": str(e),
    "traceback": traceback.format_exc(),
    "item_data": item.json_data
})
```

### REGLA #24: Estados de publicación consistentes

**Estados válidos en BD (columna `publish_status`):**

| Estado | Significado | Acción |
|--------|-------------|--------|
| `null` | No en cola | Sin acción |
| `queued_publish` | En cola | Pendiente de publicar |
| `processing` | Publicando AHORA | No tocar |
| `success` | Publicado OK | Mostrar verde |
| `error_retry` | Error, reintentar | Mostrar rojo |

```python
# ✅ Transiciones válidas:
null → queued_publish        # Añadir a cola
queued_publish → processing  # Empezar publicación
processing → success         # Éxito
processing → error_retry     # Error, reintentar
error_retry → processing     # Reintentar
error_retry → null           # Cancelar reintento
```

---

## REGLAS DE PERFORMANCE (25-28)

### REGLA #25: Polling inteligente con backoff

```javascript
// ✅ BIEN: Polling con backoff exponencial
let pollInterval = 2000; // Empezar en 2s
const MAX_INTERVAL = 10000; // Máximo 10s

async function pollWithBackoff() {
  const data = await fetchData();
  
  if (data.hasActivity) {
    pollInterval = 2000; // Reset si hay actividad
  } else {
    pollInterval = Math.min(pollInterval * 1.5, MAX_INTERVAL);
  }
  
  setTimeout(pollWithBackoff, pollInterval);
}
```

### REGLA #26: Limitar renderizado de items

```javascript
// ✅ BIEN: Limitar items mostrados
const MAX_SUCCESS_SHOWN = 20;
const MAX_ERRORS_SHOWN = 50;

const successItems = allItems
  .filter(i => i.status === 'success')
  .slice(0, MAX_SUCCESS_SHOWN);

const errorItems = allItems
  .filter(i => i.status === 'error_retry')
  .slice(0, MAX_ERRORS_SHOWN);
```

### REGLA #27: Debounce en búsquedas

```javascript
// ✅ BIEN: Debounce de 300ms (usar CONFIG.SEARCH_DEBOUNCE)
let searchTimeout;
searchInput.addEventListener('input', (e) => {
  clearTimeout(searchTimeout);
  searchTimeout = setTimeout(() => {
    performSearch(e.target.value);
  }, CONFIG.SEARCH_DEBOUNCE); // 300ms
});
```

#### Búsqueda Backend Robusta (v2.9.31)

**Problema resuelto:** La búsqueda fallaba cuando `json_data->>'$.title'` era un objeto `{original: "..."}` en lugar de string.

```python
# ✅ CORRECTO en walla_runs.py - Buscar en ambos formatos
search_filter = or_(
    func.lower(
        func.coalesce(
            func.json_unquote(func.json_extract(Listing.json_data, '$.title.original')),
            func.json_unquote(func.json_extract(Listing.json_data, '$.title'))
        )
    ).contains(search_term.lower()),
    func.lower(Listing.title).contains(search_term.lower())
)

# ❌ INCORRECTO - Solo busca un formato
search_filter = func.json_unquote(
    func.json_extract(Listing.json_data, '$.title')
).contains(search_term)  # Falla si $.title es objeto
```

### REGLA #28: Lazy loading de imágenes

```javascript
// ✅ BIEN: Intersection Observer
const imageObserver = new IntersectionObserver((entries) => {
  entries.forEach(entry => {
    if (entry.isIntersecting) {
      const img = entry.target;
      img.src = img.dataset.src;
      imageObserver.unobserve(img);
    }
  });
});

document.querySelectorAll('img[data-src]').forEach(img => {
  imageObserver.observe(img);
});
```

---

## REGLAS DE MEMORIA (29-35)

### REGLA #29: Limpiar event listeners al desmontar

```javascript
// ❌ MAL: Memory leak
function init() {
  window.addEventListener('message', handleMessage);
}

// ✅ BIEN: Guardar referencia y limpiar
let messageHandler;

function init() {
  messageHandler = handleMessage.bind(this);
  window.addEventListener('message', messageHandler);
}

function cleanup() {
  if (messageHandler) {
    window.removeEventListener('message', messageHandler);
    messageHandler = null;
  }
}
```

### REGLA #30: Limpiar intervals y timeouts

```javascript
// ✅ BIEN: Siempre limpiar
let pollInterval;

function startPolling() {
  pollInterval = setInterval(poll, 2000);
}

function stopPolling() {
  if (pollInterval) {
    clearInterval(pollInterval);
    pollInterval = null;
  }
}
```

### REGLA #31: Cerrar WebSocket connections

```javascript
// ✅ BIEN: Cleanup completo
let ws;

function connectWebSocket() {
  ws = new WebSocket(WS_URL);
  ws.onmessage = handleMessage;
}

function disconnectWebSocket() {
  if (ws) {
    ws.close();
    ws = null;
  }
}
```

### REGLA #32: Limpiar DOM references

```javascript
// ✅ BIEN: Nullificar referencias
let cachedElements = {
  modal: null,
  overlay: null
};

function cleanup() {
  Object.keys(cachedElements).forEach(key => {
    cachedElements[key] = null;
  });
}
```

### REGLA #33: AbortController para fetch

```javascript
// ✅ BIEN: Cancelar requests pendientes
let abortController;

async function fetchData() {
  // Cancelar request anterior
  if (abortController) {
    abortController.abort();
  }
  
  abortController = new AbortController();
  
  try {
    const response = await fetch(url, {
      signal: abortController.signal
    });
    return response.json();
  } catch (e) {
    if (e.name === 'AbortError') {
      return; // Request cancelado, OK
    }
    throw e;
  }
}
```

### REGLA #34: Evitar closures innecesarios

```javascript
// ❌ MAL: Closure captura todo el scope
items.forEach(item => {
  button.addEventListener('click', () => {
    // Captura item, button, items, etc.
  });
});

// ✅ BIEN: Función separada
function handleClick(item) {
  // Solo captura lo necesario
}

items.forEach(item => {
  button.addEventListener('click', () => handleClick(item));
});
```

### REGLA #35: Destruir observadores

```javascript
// ✅ BIEN: Cleanup de observers
let resizeObserver;

function init() {
  resizeObserver = new ResizeObserver(handleResize);
  resizeObserver.observe(element);
}

function cleanup() {
  if (resizeObserver) {
    resizeObserver.disconnect();
    resizeObserver = null;
  }
}
```

---

## REGLAS BACKEND (36-40)

### REGLA #36: Service Worker para APIs (Frontend)

**NUNCA usar fetch() directo en panel, SIEMPRE usar Service Worker**

```javascript
// ❌ MAL: fetch() directo
const response = await fetch(`${API_BASE}/api/user/preferences`, {
  method: 'GET',
  headers: {
    'Authorization': `Bearer ${accessToken}`,
    'Content-Type': 'application/json'
  }
});

// ✅ BIEN: Service Worker
const response = await chrome.runtime.sendMessage({
  type: 'API.FETCH_JSON',
  url: '/api/user/preferences',
  method: 'GET'
});
```

**Razones:**
1. Autenticación centralizada (JWT refresh automático)
2. Headers consistentes
3. Versionado automático
4. Error handling unificado

### REGLA #37: Preferir chrome.runtime.sendMessage sobre fetch() directo

**Usar sendMessage para la mayoría de llamadas API:**
```javascript
// ✅ PREFERIDO - A través del Service Worker
const response = await chrome.runtime.sendMessage({
  type: 'API.FETCH_JSON',
  url: '/api/listings/by-account',
  method: 'GET'
});
```

**Excepciones válidas donde fetch() directo es aceptable:**

```javascript
// ✅ EXCEPCIÓN 1: Cargar archivos locales de la extensión
fetch(chrome.runtime.getURL('intro.html'))

// ✅ EXCEPCIÓN 2: Autenticación (auth.js) - requiere manejo especial de cookies/headers
const res = await fetch(`${API_BASE}/api/auth/google/login?${params}`);

// ✅ EXCEPCIÓN 3: Ping de salud del servidor
const pingRes = await fetch(`${API_BASE}/api/version`, { ... });

// ✅ EXCEPCIÓN 4: Operaciones de créditos (credits.js) - requiere headers específicos
const response = await fetch(`${CONFIG.API_BASE}/api/billing/deduct-credit`, { ... });
```

**Razón de las excepciones:** Algunos endpoints requieren manejo especial de cookies, headers de autenticación o son operaciones críticas que necesitan control directo.

### REGLA #38: Limitar items en respuestas

**Backend:**
```python
# ✅ Limitar resultados
MAX_ITEMS = 1000

rows = db.execute(text("""
    SELECT * FROM listings
    WHERE account_id = :acc
    ORDER BY created_at DESC
    LIMIT :limit
"""), {"acc": account_id, "limit": MAX_ITEMS}).mappings().all()
```

**Frontend:**
```javascript
// ✅ Limitar items mostrados
const MAX_SUCCESS_SHOWN = 20;
const MAX_ERRORS_SHOWN = 50;

const successItems = items.filter(i => i.status === 'success').slice(0, MAX_SUCCESS_SHOWN);
const errorItems = items.filter(i => i.status === 'error').slice(0, MAX_ERRORS_SHOWN);
```

### REGLA #39: Usar columnas de BD directamente

**NUNCA parsear `json_data` si el dato ya existe en columna de BD**

```python
# ❌ MAL: Parsear JSON innecesariamente
data = json.loads(row["json_data"])
price = data.get("sale_price", {}).get("amount", 0)

# ✅ BIEN: Usar columna de BD
price = float(row["price_amount"]) if row["price_amount"] else 0
```

**Tabla listings - Columnas disponibles:**
```sql
l.id, l.account_id, l.id_wallapop, l.slug, l.url
l.title                 -- ✅ Usar esto
l.price_amount          -- ✅ Usar esto (NO price_currency, siempre EUR)
l.status                -- ✅ Usar esto
l.publish_status        -- ✅ Usar esto
l.publish_idx           -- Orden de publicación
l.publish_error         -- Error JSON si falla
l.images_local          -- Array JSON rutas locales
l.json_data             -- ⚠️ Solo para datos NO en columnas
l.created_at, l.modified_at
```

**Regla de oro:** Si está en columna BD → Usar columna. Si no está → Parsear json_data.

### REGLA #40: MEDIA_URL para imágenes en Linux ⭐ NUEVO

**En Linux, usar MEDIA_URL (no API_BASE) para construir URLs de imágenes**

**Backend - config.py:**
```python
class Settings(BaseModel):
    media_root: str = os.getenv("MEDIA_ROOT", "/var/www/mitiklive/data")
    media_url: str = os.getenv("MEDIA_URL", "https://mitiklive.com")  # ⭐ NUEVO
```

**Backend - Endpoint /api/config:**
```python
@router.get("/api/config")
def get_public_config():
    return {
        "MEDIA_URL": settings.media_url,  # Para imágenes
        "API_BASE": "/fa",                 # Para API calls
        "APP_ENV": settings.app_env
    }
```

**Extensión - Cargar MEDIA_URL:**
```javascript
// helpers.js - ensureApiBase() carga ambos
const response = await fetch(`${API_BASE}/api/config`);
const config = await response.json();
setMediaUrl(config.MEDIA_URL || 'https://www.mitiklive.com');
```

**Extensión - Usar MEDIA_URL:**
```javascript
// helpers.js - getImageUrl()
export function getImageUrl(path, mediaUrl = '', apiBase = '') {
  if (path.startsWith('/')) {
    // Rutas absolutas usan MEDIA_URL, NO API_BASE
    return `${mediaUrl}${path}`;
    // → https://mitiklive.com/data/products/1/806/123/img.jpg ✅
  }
  // Rutas relativas legacy usan API_BASE
  return `${apiBase}/${path}`;
}
```

**¿Por qué esta separación?**
- ✅ Nginx sirve imágenes directamente (más rápido que FastAPI)
- ✅ API_BASE sigue siendo `/fa/` para llamadas al backend
- ✅ MEDIA_URL puede ser diferente (CDN, otro dominio, etc)
- ✅ Sin hardcodeo de URLs en el código

---

## REGLAS FRONTEND (41-44)

### REGLA #41: Estado de botones centralizado

**NUNCA manipular botones directamente, SIEMPRE usar ButtonManager**

```javascript
// ❌ MAL: Manipulación directa
const btnResume = document.getElementById('btn-resume-pending');
btnResume.disabled = true;
btnResume.classList.add('ml-hidden');

// ✅ BIEN: ButtonManager centralizado
import * as ButtonManager from './scripts/panel/publish-button-manager.js';

ButtonManager.updateState({
  processingCount: 1,  // Automáticamente oculta botones
  pendingCount: 5
});
```

**Estado centralizado:**
```javascript
{
  filter: 'active',
  selectedCount: 0,
  hasPendingProcess: false,
  pendingCount: 0,          // Items en cola
  processingCount: 0,       // ⭐ Items publicándose AHORA
  totalActive: 0,
  availableCredits: 0
}
```

**Funciones públicas:**
```javascript
ButtonManager.updateState({ processingCount: 1 });
ButtonManager.getState();
ButtonManager.refreshAllButtons();
ButtonManager.init();
```

### REGLA #41: Validación de publicación activa

**SIEMPRE validar antes de iniciar publicación**

```javascript
async function handlePublishFlow({ accountId }) {
  // ✅ Validar si hay publicación ACTIVA
  const statusRes = await chrome.runtime.sendMessage({
    type: 'API.FETCH_JSON',
    url: `/api/walla/publish2/status?account_id=${accountId}`,
    method: 'GET'
  });
  
  if (statusRes?.ok && statusRes.data?.processing > 0) {
    return toast('⚠️ Ya hay un anuncio publicándose ahora mismo', 'warn', 4000);
  }
  
  // Continuar...
}
```

**NO confundir:**
- `pending > 0` → Hay items en cola (✅ OK para reanudar)
- `processing > 0` → Hay item publicándose AHORA (❌ BLOQUEAR)

### REGLA #42: Overlay de countdown en panel

**El overlay se muestra en el PANEL, NO en la página de Wallapop**

```javascript
// ✅ CORRECTO: En panel.js
if (msg?.type === 'PUBLISH.COUNTDOWN_START') {
  const delaySeconds = msg.delaySeconds || 30;
  CONFIG.showCountdownToast(delaySeconds);  // Muestra en panel
  sendResponse({ ok: true });
  return;
}

// ❌ INCORRECTO: Intentar mostrar en Wallapop
chrome.tabs.sendMessage(tabId, { type: 'SHOW_COUNTDOWN' });
```

**Razón:** El usuario está mirando el panel lateral, no la página de Wallapop.

### REGLA #43: Preferir variables CSS de variables.css

**TODO CSS nuevo DEBE usar variables de `variables.css`:**

```css
/* ✅ PREFERIDO - 863 usos en panel.css */
.mi-elemento {
  color: var(--text-primary);
  background: var(--bg-secondary);
  padding: var(--space-4);
  border-radius: var(--radius-md);
  font-size: var(--text-sm);
  font-weight: var(--font-semibold);
  box-shadow: var(--shadow-md);
  transition: var(--transition-base);
}
```

**Excepciones aceptables (4 casos en panel.css):**
```css
/* ✅ ACEPTABLE - Colores muy específicos sin variable definida */
.elemento-especial {
  border-color: #1a2744;  /* Color específico de borde */
}

/* ✅ ACEPTABLE - Valores numéricos específicos */
.elemento {
  width: 280px;
  max-height: 600px;
}

/* ✅ ACEPTABLE - Cálculos */
.elemento {
  height: calc(100vh - 280px);
}

/* ✅ ACEPTABLE - Transformaciones */
.elemento {
  transform: translateX(-50%);
}
```

**Nota:** El código actual tiene ~4 colores hardcodeados en panel.css. Esto es aceptable para casos muy específicos donde no existe variable.

---

## REGLAS DE IMÁGENES Y MULTI-WORKER (45-46)

### **REGLA #45: URLS DE IMÁGENES SIEMPRE CENTRALIZADAS** 🔴

**NUNCA construir URLs de imágenes manualmente**

❌ **INCORRECTO:**
```python
# Backend
url = f"/public/{path}"
url = f"{base}/files/{img}"
url = f"/data/{path}"  # Incluso esto está mal

# JavaScript
url = `${apiBase}/public/${path}`
url = baseUrl + "/data/" + filename
```

✅ **CORRECTO:**
```python
# Backend
from ..image_helpers import build_image_url, build_images_urls, get_first_image_url
url = build_image_url(path)
urls = build_images_urls(paths_array)
first = get_first_image_url(json_string)

# JavaScript
import { getImageUrl } from './listings/helpers.js';
const url = getImageUrl(path, apiBase);
```

**Ubicaciones:**
- Backend: `/var/www/mitiklive/backend/app/image_helpers.py`
- Extensión: `/scripts/panel/listings/helpers.js`

**Formato único:**
```
Backend retorna:    /data/products/1/1166/xyz/img_000.jpg
Extensión usa:      https://mitiklive.com/data/products/1/1166/xyz/img_000.jpg
Nginx sirve desde:  /var/www/mitiklive/data/products/1/1166/xyz/img_000.jpg
```

---

### **REGLA #46: REDIS OBLIGATORIO CON MÚLTIPLES WORKERS** 🔴

**SIEMPRE usar Redis Pub/Sub con múltiples workers de Gunicorn**

✅ **Configuración correcta:**
- 1 worker (desarrollo) → Redis opcional
- 2+ workers (producción) → Redis obligatorio

❌ **NUNCA:**
- Múltiples workers sin Redis → WebSocket NO funciona
- WebSocket sin Redis Pub/Sub → Notificaciones perdidas

**Verificar:**
```bash
# Redis activo
redis-cli ping
# Respuesta: PONG

# Conexiones Redis = workers + listeners
redis-cli info clients | grep connected_clients
# Respuesta: connected_clients:9  (8 workers + 1 pubsub)

# Logs backend
sudo journalctl -u mitiklive-backend -f | grep Redis
# Debe mostrar: ✅ Redis Pub/Sub inicializado para WebSocket
```

**Troubleshooting:**
```bash
# Si WebSocket no funciona con múltiples workers:
sudo systemctl status redis-server
sudo systemctl restart redis-server
sudo systemctl restart mitiklive-backend
```

---

## 📦 REGLAS DE IMPORTS ES6 vs DINÁMICOS (49-51) 🆕

### **REGLA #49: CONTEXTOS DE EJECUCIÓN MV3** 🔴

La extensión tiene 3 contextos de ejecución con reglas diferentes:

| Contexto | Archivo entrada | Tipo | Tiene DOM | Import permitido |
|----------|-----------------|------|-----------|------------------|
| **Service Worker** | `service_worker.js` | Módulo ES6 | ❌ NO | Estático |
| **Content Script** | `content_script.js` | Script clásico | ✅ SÍ (Wallapop) | Dinámico |
| **Panel** | `panel.js` | Módulo ES6 | ✅ SÍ (Panel) | Estático |

---

### **REGLA #50: IMPORTS POR UBICACIÓN DE ARCHIVO** 🔴

| Carpeta | Contexto ejecución | Tipo import | Razón |
|---------|-------------------|-------------|-------|
| `sw/*` | Service Worker | **ESTÁTICO** | Es módulo ES6 |
| `scripts/panel/*` | Panel | **ESTÁTICO** | Es módulo ES6 |
| `scripts/publish/*` | Content Script | **DINÁMICO** | Se carga desde CS |
| `scripts/dom*.js` | Content Script | **DINÁMICO** | Se carga desde CS |
| `scripts/delete.js` | Content Script | **DINÁMICO** | Se carga desde CS |
| `scripts/backup.js` | Content Script | **DINÁMICO** | Se carga desde CS |

**Ejemplo Service Worker (ESTÁTICO):**
```javascript
// sw/boot.js - ✅ CORRECTO
import { apiClient } from './api_client.js';
import { storage } from './storage.js';
```

**Ejemplo Content Script (DINÁMICO):**
```javascript
// content_script.js - ✅ CORRECTO
const url = chrome.runtime.getURL('scripts/publish.js');
const { fillListingForm } = await import(url);
```

**Ejemplo Panel (ESTÁTICO):**
```javascript
// panel.js - ✅ CORRECTO
import { toast } from './scripts/utils.js';
import * as UI from './scripts/panel/ui.js';
```

---

### **REGLA #51: CADENA DE IMPORTS EN PUBLICACIÓN** 🔴

Los scripts en `scripts/publish/*` DEBEN usar imports dinámicos porque se ejecutan en contexto de Content Script:

```
Content Script (NO es módulo ES6)
    │
    └── await import(chrome.runtime.getURL('scripts/publish.js'))
                │
                └── publish.js (ahora en contexto CS)
                        │
                        └── await import('../dom-optimized.js')  ✅ DINÁMICO
                        └── import { x } from './constants.js'   ❌ ERROR
```

**✅ CORRECTO en scripts/publish/form-filler.js:**
```javascript
// Import dinámico porque se ejecuta en Content Script
const module = await import('../dom-optimized.js');
const { waitFor } = module;
```

**❌ INCORRECTO (causaría error):**
```javascript
// Esto fallaría porque Content Script no es módulo ES6
import { waitFor } from '../dom-optimized.js';
```

---

### Resumen visual

```
┌─────────────────────────────────────────────────────────────────┐
│                    REGLA DE IMPORTS MV3                         │
├─────────────────────────────────────────────────────────────────┤
│                                                                 │
│  Service Worker ──► import { x } from './y.js'    (ESTÁTICO)   │
│                                                                 │
│  Panel ──────────► import { x } from './y.js'     (ESTÁTICO)   │
│                                                                 │
│  Content Script ─► await import(getURL('y.js'))   (DINÁMICO)   │
│        │                                                        │
│        └──► scripts/publish/* ──► await import()  (DINÁMICO)   │
│                                                                 │
└─────────────────────────────────────────────────────────────────┘
```

---


# FLUJOS COMPLETOS

## 🔄 FLUJO COMPLETO DE PUBLICACIÓN (v5.1 - DETALLADO)

### Vista General del Sistema

```
┌─────────────────────────────────────────────────────────────────────┐
│                        FLUJO DE PUBLICACIÓN                         │
│                    (23 pasos + 7 secciones form)                    │
└─────────────────────────────────────────────────────────────────────┘

┌──────────────┐         ┌──────────────┐         ┌──────────────┐
│   PANEL UI   │ ──1──→  │   SERVICE    │ ──2──→  │   BACKEND    │
│  (panel.js)  │         │   WORKER     │         │  (FastAPI)   │
└──────────────┘         │  (publish-   │         └──────────────┘
       ↑                 │  process-    │                │
       │                 │  next.js)    │                │
      23                 └──────────────┘               3│
       │                        │                        │
       │                       4-22                      ↓
       │                        │              ┌──────────────┐
       │                        ↓              │   BASE DE    │
       └─────────────────┌──────────────┐      │   DATOS      │
                         │   CONTENT    │      │  (MariaDB)   │
                         │   SCRIPT     │      └──────────────┘
                         │  (form-      │              
                         │  filler.js)  │              
                         └──────────────┘
                                │
                                ↓
                         ┌──────────────┐
                         │   WALLAPOP   │
                         │   (DOM)      │
                         └──────────────┘
```

---

### FASE 1: INICIACIÓN (Panel UI)

**Archivo:** `panel.js`

| Paso | Acción | Detalle |
|------|--------|---------|
| 1.1 | Usuario pulsa "Publicar" | Botón en panel lateral |
| 1.2 | Validar no hay publicación activa | Verificar processing === 0 |
| 1.3 | Obtener items seleccionados | O todos los pendientes |
| 1.4 | Enviar PUBLISH.PROCESS_NEXT | Al Service Worker |

---

### FASE 2: ORQUESTACIÓN (Service Worker)

**Archivo:** `sw/handlers/publish-process-next.js` (1394 líneas)

#### PASOS 1-5: PREPARACIÓN

| Paso | Acción | Endpoint/Mensaje | Timeout |
|------|--------|------------------|---------|
| 1 | Obtener siguiente item | GET `/api/walla/publish2/next-item/{accountId}` | - |
| 2 | Verificar categoría | Excluir Coches/Inmobiliaria | - |
| 3 | Marcar como 'processing' | PATCH `/api/walla/publish2/update-status/{itemId}` | - |
| 4 | Buscar pestaña Wallapop | `chrome.tabs.query` | - |
| 5 | Navegar a /upload | Solo si hay datos previos | - |

#### PASOS 6-11: VERIFICACIÓN Y SUMMARY

| Paso | Acción | Mensaje CS | Timeout |
|------|--------|------------|---------|
| 6 | Verificar CS (ping) | CS.PING | 10s |
| 7 | Seleccionar vertical | CS.SELECT_VERTICAL | 30s |
| 8 | Verificar CS (post-vertical) | CS.PING | 10s |
| 9 | Rellenar summary | CS.FILL_SUMMARY | 60s |
| 10 | Click continuar summary | CS.CLICK_CONTINUE_SUMMARY | 30s |
| 11 | Verificar CS (pre-imágenes) | CS.PING | 10s |

#### PASOS 12-18: IMÁGENES Y FORMULARIO

| Paso | Acción | Mensaje CS | Timeout |
|------|--------|------------|---------|
| 12 | Subir imágenes | CS.UPLOAD_IMAGES | 60s |
| 13 | Click continuar fotos | CS.CLICK_CONTINUE_PHOTOS | 30s |
| 14 | Seleccionar categoría | CS.SELECT_CATEGORY | 30s |
| 15 | Escanear formulario | CS.SCAN_FORM | 10s |
| 16 | **Rellenar formulario** | CS.FILL_LISTING_FORM | 60s |
| 17 | Re-escanear | CS.SCAN_FORM | 10s |
| 18 | Verificar botón publicar | Esperar habilitado | 30s |

#### PASOS 19-23: PUBLICACIÓN Y ACTUALIZACIÓN

| Paso | Acción | Detalle | Timeout |
|------|--------|---------|---------|
| 19 | Eliminar anuncio viejo | API Wallapop (si existe id_wallapop) | 10s |
| 20 | Esperar delay usuario | 5-240 segundos configurables | variable |
| 21 | Click publicar + capturar ID | CS.PUBLISH_AND_CAPTURE | 120s |
| 22 | Actualizar BD | PATCH con nuevo id_wallapop | - |
| 23 | Siguiente o finalizar | Loop o modal éxito | - |

---

### FASE 3: RELLENADO DE FORMULARIO

**Archivo:** `scripts/publish/form-filler.js` (2192 líneas)

#### Función Principal: `fillListingForm(snapshot, opts)`

```
ENTRADA: snapshot = { title, description, sale_price, type_attributes, ... }
SALIDA:  { ok: boolean, filled: [], skipped: [], missingRequired: [] }
```

#### SECCIONES DEL FORMULARIO

| # | Campo | Selector | Función |
|---|-------|----------|---------|
| 1 | Formulario | `#step-listing`, `tsl-upload-step-form` | waitFor() |
| 2 | Título | `input#title`, `input[name="title"]` | clearThenSetText() |
| 3 | Descripción | `textarea`, `[data-testid="description"]` | clearThenSetText() |
| 4 | Talla | dropdown size | selectWallaDropdown() |
| 5 | Moneda | dropdown currency | selectWallaDropdown() |
| 6 | Estado/Condición | dropdown condition | tryDD() con mapeo |
| 6.3 | **Marca** | `wallapop-combo-box[data-testid="brand"]` | fillBrandCombo() |
| 6.4 | **Modelo** | `wallapop-combo-box[data-testid="model"]` | fillModelCombo() |
| 6.5 | Capacidad | `wallapop-combo-box[data-testid="storage_capacity"]` | fillStorageCapacityCombo() |
| 6.6 | Precio | input numérico | setValue() |
| 7 | Re-scan | inspectListingForm() | Validación final |

#### Funciones Auxiliares Críticas

**fillBrandCombo(brandValue, opts)** - Línea 56
```javascript
// 1. Buscar combo marca
// 2. Focus para activar listbox
// 3. Esperar opciones (8s timeout)
// 4. Buscar coincidencia case-insensitive
// 5. Click con delay humano
// 6. Verificar selección
// ⚠️ Si falla → ERROR CRÍTICO, detener publicación
```

**fillModelCombo(modelValue, opts)** - Línea 229
```javascript
// 1. Buscar combo modelo
// 2. Esperar 500ms después de marca (v10.5.16)
// 3. Escribir para filtrar
// 4. Esperar opciones con MutationObserver
// 5. Buscar por aria-label
// 6. Si no exacta → primera coincidencia
// ⚠️ Si falla → ERROR CRÍTICO, detener publicación
```

**fillStorageCapacityCombo(storageValue, opts)** - Línea 499
```javascript
// 1. Normalizar: "128gb" → "128 GB"
// 2. Buscar combo capacidad
// 3. Verificar que no esté ya seleccionada
// 4. Click en opción
```

---

### FASE 4: MODAL DE ÉXITO (v10.5.23)

**Archivo:** `scripts/panel/success-modal.js`

```javascript
showSuccessModal({ total, published, errors })

// LÓGICA:
if (published === 0) return;     // No mostrar si nada publicado
if (errors > 0) return;          // No mostrar si hay errores

// Solo si 100% éxito:
// - Mostrar 🎉 ¡Felicidades!
// - Stats: Total | Publicados | % Éxito
// - Confetti animado
// - Auto-cierre: 10 segundos
```

---

### ESTADOS DE PUBLICACIÓN

| Estado | Significado | Color UI | Acción |
|--------|-------------|----------|--------|
| `null` | No en cola | - | - |
| `queued_publish` | En cola | Gris | Esperando turno |
| `processing` | Publicándose AHORA | Amarillo | ⚠️ No tocar |
| `published` | Éxito | Verde | Mostrar nuevo ID |
| `error_retry` | Error | Rojo | Permitir reintento |
| `skipped` | Omitido | Gris | Categoría no soportada |

### Diagrama de Estados

```
null ──┐
       ├─→ queued_publish ──→ processing ──┬─→ published
       │                                   │
       │                                   └─→ error_retry ──┐
       │                                                      │
       │                                   ┌─→ skipped        │
       │                                   │                  │
       └───────────────────────────────────┴──────────────────┘
                         (Reanudar/Reintentar)
```

---

### TIMEOUTS Y DELAYS

**Archivo:** `sw/constants.js`

```javascript
SW_TIMEOUTS = {
  CS_PING: 10000,        // 10s - Verificar Content Script
  CS_OPERATION: 30000,   // 30s - Operación genérica
  FORM_FILL: 60000,      // 60s - Rellenar formulario
  PUBLISH: 120000,       // 2min - Click publicar + captura
}

SW_DELAYS = {
  AFTER_VERTICAL: 2000,  // Después de seleccionar categoría
  AFTER_UPLOAD: 1500,    // Después de subir imágenes
  BETWEEN_ITEMS: 3000,   // Entre items
}

// Delay humano con variación
humanDelay(base, variance) → base * (1 ± random * variance)
// Ej: humanDelay(500, 0.3) → 350-650ms
```

---

### ARCHIVOS CRÍTICOS

```
extension/
├── sw/handlers/publish-process-next.js  ← ORQUESTADOR (1394 líneas)
├── scripts/publish/form-filler.js       ← RELLENADO (2192 líneas)
├── scripts/panel/success-modal.js       ← MODAL ÉXITO
├── scripts/panel/progress-tab.js        ← BARRA PROGRESO
└── content_script.js                    ← PUENTE CS ↔ SW
```

---

### MANEJO DE ERRORES

```javascript
// Error en Content Script → Notificar backend
await apiFetch('/api/walla/publish2/item-error', {
  method: 'POST',
  body: { account_id, item_id, error: String(error) }
});

// Backend marca como error_retry
item.publish_status = 'error_retry'
item.publish_error = JSON.stringify({
  error_type: errorType,
  error_message: message,
  timestamp: new Date().toISOString()
})
```

---

### CHECKLIST PRE-PUBLICACIÓN

- [ ] Items seleccionados o pendientes disponibles
- [ ] No hay publicación activa (processing === 0)
- [ ] Pestaña Wallapop abierta y accesible
- [ ] Content Script responde a ping
- [ ] Categoría soportada (no Coches/Inmobiliaria)
- [ ] Imágenes disponibles (local o URLs)
- [ ] Créditos suficientes

---

## 📦 FLUJO DE BACKUP (v10.5.20 - SIMPLIFICADO)

**Archivo:** `scripts/panel/listings/websocket.js`

```
1. Usuario pulsa "Backup"
2. Panel → SW: BACKUP.START
3. SW → Backend: POST /api/walla/backup/start
4. Backend marca account como backup_in_progress
5. SW abre tab Wallapop en perfil público
6. Content Script scraping:
   - Detecta items en página
   - Extrae datos (título, precio, fotos)
   - Descarga imágenes a /data/products/{user_id}/{account_id}/{id}/
7. Content Script → Backend: POST /api/walla/backup/item
8. Backend guarda en BD
9. WebSocket envía backup_item_saved → Panel solo responde OK (NO toca tabla)
10. Scroll infinito hasta terminar
11. Backend marca backup_in_progress = false
12. WS.BACKUP_COMPLETE → Panel recarga tabla completa desde backend
```

**⚠️ CAMBIO v10.5.20:** Durante el backup, la tabla NO se actualiza en tiempo real.
Solo se actualiza al finalizar para evitar duplicados visuales.

```javascript
// websocket.js - Handler backup_item_saved
if (msg?.type === 'backup_item_saved') {
  sendResponse({ ok: true });  // Solo responder OK
  return true;                 // NO tocar tabla
}

// Al completar backup
if (msg?.type === 'WS.BACKUP_COMPLETE') {
  loadListingsTable(accountId);  // Recargar tabla limpia
}
```

---

## 💳 FLUJO DE CRÉDITOS

**Ubicación:** `scripts/panel/credits.js`

### Conceptos

| Concepto | Descripción |
|----------|-------------|
| `credits` | Créditos extra comprados |
| `publications_used` | Publicaciones usadas del plan |
| `publications_limit` | Límite del plan mensual |
| `is_unlimited` | Plan ilimitado (sin límite) |

### Flujo Completo

```
1. Usuario abre panel
   ↓
2. credits.js → init()
   ↓
3. Verificar usuario logueado (getUserId via JWT)
   ↓
4. fetchCredits(true) → GET /api/billing/check-credits
   ↓
5. Backend retorna:
   {
     "credits": 10,
     "plan": "basic",
     "publications_used": 5,
     "publications_limit": 50,
     "is_unlimited": false
   }
   ↓
6. updateUI() → Mostrar saldo en panel
   ↓
7. Usuario publica anuncio
   ↓
8. hasCredits() verifica:
   - ¿Plan ilimitado? → OK
   - ¿publications_used < publications_limit? → OK
   - ¿credits > 0? → OK
   - Si no → Mostrar "Sin créditos"
   ↓
9. deductCredit() → POST /api/billing/deduct-credit
   {
     "user_id": 123,
     "walla_account_id": 456,
     "listing_id": 789
   }
   ↓
10. Backend descuenta y retorna { "success": true }
    ↓
11. Actualizar estado local: credits-- o publications_used++
    ↓
12. updateUI() → Reflejar nuevo saldo
```

### Endpoints Utilizados

| Método | Endpoint | Uso |
|--------|----------|-----|
| GET | `/api/billing/check-credits` | Consultar saldo |
| POST | `/api/billing/deduct-credit` | Descontar al publicar |

---

## 🗑️ FLUJO DE ELIMINACIÓN

**Ubicación:** `scripts/panel/delete-handler.js`

### Flujo Completo

```
1. Usuario selecciona anuncios (checkboxes)
   ↓
2. Pulsa botón "Eliminar seleccionados"
   ↓
3. handleDeleteSelected() → showDeleteConfirmModal(count)
   ↓
4. Usuario confirma en modal
   ↓
5. requireWallaActive() → Verificar Wallapop abierto
   ↓
6. Validar cuenta:
   - Cuenta panel == Cuenta Wallapop?
   - Si no coincide → Mostrar error, abortar
   ↓
7. Para cada anuncio seleccionado:
   │
   ├─ moveRowToTop() → Mover fila arriba visualmente
   ├─ setRowState('is-deleting') → Rojo
   │
   ├─ deleteOne({ id: id_wallapop })
   │   ↓
   │   SW abre Wallapop → Elimina anuncio real
   │   ↓
   │   Resultado:
   │   ├─ OK (eliminado) → setRowState('is-success') → Verde
   │   ├─ OK (no existía) → setRowState('is-warning') → Amarillo
   │   └─ Error → setRowState('is-error') → Rojo
   │
   └─ updateProgressBar({ current, total })
   ↓
8. showDeleteSummary({
     reallyDeleted: 5,
     notExisted: 2,
     failed: 1
   })
   ↓
9. refreshListings() → Recargar tabla
   ↓
10. clearSelection() → Limpiar checkboxes
```

### Estados Visuales

| Estado | Color | Significado |
|--------|-------|-------------|
| `is-deleting` | 🔴 Rojo | Eliminando... |
| `is-success` | 🟢 Verde | Eliminado OK |
| `is-warning` | 🟡 Amarillo | No existía en Wallapop |
| `is-error` | 🔴 Rojo | Error al eliminar |

---

## 🔐 FLUJO DE AUTENTICACIÓN

**Ubicación:** `scripts/panel/auth.js`

### Métodos de Login

| Método | Función | Uso |
|--------|---------|-----|
| Email/Password | `doLogin()` | Admin y usuarios legacy |
| Google OAuth | `doGoogleLogin()` | Usuarios normales |

### Flujo Login Email/Password

```
1. Usuario introduce email + contraseña
   ↓
2. doLogin() → SW: SESSION.LOGIN_EMAIL
   ↓
3. SW → Backend: POST /api/auth/login
   ↓
4. Backend verifica credenciales
   ├─ OK → Retorna { access_token, refresh_token }
   ├─ Credenciales incorrectas → { error: "invalid_credentials", attempts_left: 4 }
   └─ Bloqueado → { error: "too_many_attempts", retry_after: 120 }
   ↓
5. SW guarda tokens en chrome.storage
   ↓
6. Panel actualiza UI (mostrar cuentas, créditos, etc.)
```

### Flujo Login Google OAuth

```
1. Usuario pulsa "Continuar con Google"
   ↓
2. doGoogleLogin()
   ↓
3. Verificar conexión: GET /api/version
   ↓
4. Obtener URL de Google: GET /api/auth/google/login?device_id=xxx
   ↓
5. Abrir popup de Google (500x650)
   ↓
6. Usuario autoriza en Google
   ↓
7. Google callback → Backend genera tokens
   ↓
8. Backend envía postMessage al popup
   ↓
9. setupGooglePopupListener() recibe tokens
   ↓
10. SW guarda tokens en chrome.storage
    ↓
11. Panel actualiza UI
```

### Flujo Refresh Token

```
1. Access token expira (401)
   ↓
2. SW intercepta error
   ↓
3. SW → Backend: POST /api/auth/refresh
   { "refresh_token": "xxx" }
   ↓
4. Backend retorna nuevo access_token
   ↓
5. SW actualiza storage y reintenta petición original
```

### Protección Brute Force

| Intentos fallidos | Acción |
|-------------------|--------|
| 1-4 | Mostrar intentos restantes |
| 5 | Bloquear 2 minutos |
| Después de bloqueo | Reinicia contador |

---

## ✏️ FLUJO EDITOR DE ANUNCIOS

**Ubicación:** `scripts/panel/listing-editor/`

### Archivos del Módulo

| Archivo | Responsabilidad |
|---------|-----------------|
| `index.js` | Punto de entrada, abre modal |
| `modal.js` | Renderiza modal, campos editables |
| `api.js` | Comunicación con backend |
| `image-manager.js` | Gestión de imágenes |

### Flujo Completo

```
1. Usuario hace click en "Editar" de un anuncio
   ↓
2. openEditorModal({ loading: true })
   ↓
3. fetchListingData(listingId)
   → SW: LISTING.EDITOR.GET
   → Backend: GET /api/walla/listing-editor/{id}
   ↓
4. Backend retorna:
   {
     "listing": {
       "id": 123,
       "title": "iPhone 12",
       "description": "...",
       "price": 450,
       "images_local": ["/data/products/..."],
       "category_name": "Móviles"
     }
   }
   ↓
5. renderEditorModal(listing)
   - Campos editables: título, descripción, precio
   - Campos solo lectura: categoría, ubicación
   - Gestión imágenes: reordenar, eliminar, añadir
   ↓
6. Usuario modifica campos
   - Contadores de caracteres en tiempo real
   - Validación: título ≤50, descripción ≤640
   ↓
7. Usuario pulsa "Guardar"
   ↓
8. saveListingData(listingId, data)
   │
   ├─ Si hay imágenes nuevas:
   │   uploadImage() → POST /api/walla/listing-editor/{id}/upload-image
   │
   └─ Guardar cambios:
       → SW: LISTING.EDITOR.SAVE
       → Backend: PATCH /api/walla/listing-editor/{id}
       {
         "title": "iPhone 12 Pro",
         "description": "...",
         "price": 400,
         "images_order": [...],
         "images_to_delete": [...]
       }
   ↓
9. Backend actualiza BD y retorna OK
   ↓
10. closeEditorModal()
    ↓
11. refreshListings() → Actualizar tabla
```

### Límites de Campos

| Campo | Límite |
|-------|--------|
| Título | 50 caracteres |
| Descripción | 640 caracteres |
| Precio | ≥ 0 EUR |
| Imágenes | Máximo definido por Wallapop |

---

## 👤 FLUJO DE CUENTAS WALLAPOP

**Ubicación:** `scripts/panel/auth.js`

### Conceptos

| Concepto | Descripción |
|----------|-------------|
| Usuario MitikLive | Cuenta en MitikLive (email/Google) |
| Cuenta Wallapop | Perfil en Wallapop (alias) |
| Vinculación | 1 usuario MitikLive → N cuentas Wallapop |

### Flujo Detectar y Vincular Cuenta

```
1. Usuario inicia sesión en MitikLive
   ↓
2. Panel carga cuentas vinculadas:
   loadAccounts() → GET /api/walla/accounts
   ↓
3. Usuario abre Wallapop en otra pestaña
   ↓
4. Usuario inicia acción (backup, publicar, eliminar)
   ↓
5. requireWallaActive()
   → Detecta pestaña Wallapop
   → Extrae info del perfil logueado (alias, user_id)
   ↓
6. validateAccountMatch(wallapopInfo)
   │
   ├─ Cuenta ya vinculada?
   │   → Seleccionar automáticamente en dropdown
   │
   ├─ Cuenta NO vinculada?
   │   → Modal: "Nueva cuenta detectada"
   │   → "¿Añadir cuenta y hacer backup?"
   │   │
   │   └─ Usuario acepta:
   │       createNewWallapopAccount(wallapopInfo)
   │       → POST /api/walla/accounts
   │       → refreshAccountsDropdown()
   │
   └─ Cuenta seleccionada ≠ Wallapop abierto?
       → Modal error: "Las cuentas no coinciden"
       → Abortar acción
```

### Selector de Cuenta (Dropdown)

```html
<select id="sel-account">
  <option value="">-- Selecciona cuenta --</option>
  <option value="1" data-alias="usuario1">usuario1 (50 anuncios)</option>
  <option value="2" data-alias="tienda_pro">tienda_pro (200 anuncios)</option>
</select>
```

### Validación de Coincidencia

Antes de cualquier acción (backup, publicar, eliminar):

| Panel seleccionado | Wallapop abierto | Resultado |
|--------------------|------------------|-----------|
| usuario1 | usuario1 | ✅ OK |
| usuario1 | tienda_pro | ❌ Error |
| (ninguno) | tienda_pro | Preguntar si vincular |

---

# MÓDULOS CRÍTICOS

## 📊 PROGRESS-TAB (Monitor de Publicación)

**Ubicación:** `scripts/panel/progress-tab.js`

**Responsabilidad:** Mostrar progreso de publicación en tiempo real

### Características

- ✅ Polling cada 2s a `/api/walla/publish2/monitor-state`
- ✅ Overlay de countdown configurable
- ✅ Estados en tiempo real (pending, processing, success, error_retry)
- ✅ Integración con ButtonManager
- ✅ Sin fetch() directo (usa Service Worker - REGLA #36)

### API Utilizada

```javascript
// GET cada 2s
const response = await chrome.runtime.sendMessage({
  type: 'API.FETCH_JSON',
  url: `/api/walla/publish2/monitor-state/${accountId}`,
  method: 'GET'
});

// Response:
{
  "items": [
    {
      "id": 123,
      "title": "iPhone 14",
      "price": 450.0,
      "status": "processing",  // pending, processing, success, error_retry
      "images": [...]
    }
  ],
  "stats": {
    "pending": 5,
    "processing": 1,
    "success": 10,
    "errors": 2
  }
}
```

### Estados UI

```javascript
// Mapeo de estados a clases CSS
const STATUS_CLASSES = {
  'pending': 'status-pending',      // Gris
  'processing': 'status-processing',// Amarillo parpadeante
  'success': 'status-success',      // Verde
  'error_retry': 'status-error'     // Rojo
};
```

---

## 🎛️ PUBLISH-BUTTON-MANAGER (Gestión de Botones)

**Ubicación:** `scripts/panel/publish-button-manager.js`

**Responsabilidad:** Centralizar estado y lógica de TODOS los botones

### Estado Centralizado

```javascript
const state = {
  // Filtro
  filter: 'active',
  
  // Selección
  selectedCount: 0,
  selectedActive: 0,
  selectedDeleted: 0,
  isSelectAll: false,
  selectAllByFilter: false,
  
  // Proceso pendiente
  hasPendingProcess: false,
  pendingProcessId: null,
  pendingType: null,        // 'backup' | 'publish'
  pendingCount: 0,          // Items en cola
  processingCount: 0,       // ⭐ Items publicándose AHORA
  
  // Totales
  totalActive: 0,
  totalDeleted: 0,
  dbTotalActive: 0,
  dbTotalDeleted: 0,
  
  // Créditos
  availableCredits: 0
};
```

### API Pública

```javascript
// Actualizar estado (automáticamente refresca UI)
ButtonManager.updateState({
  processingCount: 1,
  pendingCount: 5
});

// Obtener estado actual
const currentState = ButtonManager.getState();

// Refrescar UI manualmente
ButtonManager.refreshAllButtons();

// Inicializar
ButtonManager.init();
```

### Lógica de Visibilidad (REGLA #40)

```javascript
// Botones Reanudar/Cancelar se OCULTAN cuando:
if (processingCount > 0 || !hasPendingProcess) {
  btnResume.classList.add('ml-hidden');
  btnCancel.classList.add('ml-hidden');
}

// Botones Reanudar/Cancelar se MUESTRAN cuando:
if (processingCount === 0 && hasPendingProcess) {
  btnResume.classList.remove('ml-hidden');
  btnCancel.classList.remove('ml-hidden');
}
```

---

## 📦 LISTINGS MODULE (Gestión Centralizada)

**Ubicación:** `scripts/panel/listings/`

### Estructura

```
listings/
├─ index.js          → Controlador principal y sincronización
├─ module-state.js   → ⭐⭐⭐ FUENTE ÚNICA DE VERDAD (v10.5.82)
├─ render.js         → ⭐⭐⭐ RENDERIZADO CENTRALIZADO
├─ helpers.js        → Funciones auxiliares
├─ filters.js        → Filtros y búsqueda
├─ events.js         → Event handlers
├─ progressive.js    → Scroll infinito
└─ websocket.js      → Eventos WebSocket
```

### 🔴 ARQUITECTURA DE ESTADO UNIFICADA (v10.5.82)

**CRÍTICO:** Todo el estado de selección de listings está centralizado en `module-state.js`.

```
┌─────────────────────────────────────────────────────────────┐
│           module-state.js (FUENTE ÚNICA LISTINGS)           │
│                                                             │
│  Estado de selección:                                       │
│  ├─ selectedIds (Set)         → IDs seleccionados           │
│  ├─ currentFilter             → Filtro activo               │
│  ├─ currentListings           → Listings cargados           │
│  ├─ selectAllByFilter         → Flag "todos por filtro"     │
│  ├─ selectionByFilter (Map)   → Selección por filtro        │
│  └─ getSelectionState()       → Retorna TODO el estado      │
└─────────────────────────────────────────────────────────────┘
                              ↓
┌─────────────────────────────────────────────────────────────┐
│           state-manager.js (ESTADO GLOBAL)                  │
│                                                             │
│  ├─ wallapop (credenciales)                                 │
│  ├─ account (cuenta activa)                                 │
│  ├─ operations (flags)                                      │
│  └─ selection.dbStats (SOLO stats de BD)                    │
│                                                             │
│  ❌ NO CONTIENE: selectAllByFilter (movido a module-state)  │
└─────────────────────────────────────────────────────────────┘
                              ↓
┌─────────────────────────────────────────────────────────────┐
│      publish-button-manager.js (SOLO RENDERIZADO)           │
│                                                             │
│  Recibe datos via updateState() desde listings/index.js     │
│  NO mantiene estado propio - solo renderiza botones         │
└─────────────────────────────────────────────────────────────┘
```

### module-state.js - Funciones Clave (v10.5.82)

```javascript
// ⭐ FUNCIÓN PRINCIPAL - Retorna todo el estado de selección
export function getSelectionState() {
  return {
    filter,              // Filtro actual
    selectedCount,       // Total seleccionados
    selectedActive,      // Seleccionados activos
    selectedDeleted,     // Seleccionados eliminados
    totalActive,         // Total activos cargados
    totalDeleted,        // Total eliminados cargados
    isSelectAll,         // Si todos están seleccionados
    selectAllByFilter    // Flag "seleccionar todos por filtro"
  };
}

// Gestión de selectAllByFilter (MOVIDO desde state-manager.js)
export function getSelectAllByFilter() { return selectAllByFilter; }
export function setSelectAllByFilter(value) { selectAllByFilter = !!value; }
export function clearSelectAllByFilter() { selectAllByFilter = false; }

// Persistencia de selección por filtro
export function saveSelectionForCurrentFilter() { ... }
export function restoreSelectionForFilter(filter) { ... }
export function clearAllSavedSelections() { ... }
```

### Flujo de Sincronización de Estado

```
Usuario selecciona checkbox
        ↓
events.js: addSelection(id) / removeSelection(id)
        ↓
module-state.js: selectedIds.add(id) / delete(id)
        ↓
events.js: updateSelectionUI()
        ↓
index.js: State.getSelectionState()
        ↓
index.js: ButtonManager.updateState({...})
        ↓
publish-button-manager.js: Renderiza botones
```

### render.js - Funciones Clave

```javascript
// Renderizar tabla completa
export function renderListings(items) { ... }

// Renderizar estadísticas
export function renderStats(stats) { ... }

// ⭐ Incrementar contador individual (REGLA #11)
export function incrementStatCounter(status) {
  const counterEl = document.getElementById(`stat-${status}`);
  if (counterEl) {
    const current = parseInt(counterEl.textContent) || 0;
    counterEl.textContent = current + 1;
  }
}

// Renderizar fila individual
export function renderListingRow(item) { ... }

// Actualizar fila existente
export function updateListingRow(itemId, updates) { ... }

// ⭐ v10.5.82: Contador inteligente
export function updateListingsCounter(loaded, total, isSearchResult = false) {
  // Si es búsqueda y loaded === total, solo muestra "X resultados"
  // Si no, muestra "X de Y"
}
```

### helpers.js - Utilidades

```javascript
/**
 * Convierte URL de imagen a formato completo
 * ⭐ ACTUALIZADO en Linux: Usa MEDIA_URL para rutas absolutas
 */
export function getImageUrl(path, mediaUrl = '', apiBase = '') {
  if (!path) return '';
  if (/^https?:\/\//i.test(path)) return path;
  if (path.startsWith('/')) {
    const base = mediaUrl || 'https://www.mitiklive.com';
    return `${base}${path}`;
  }
  if (!apiBase) return '';
  return `${apiBase}/${path}`;
}

// Formato precio EUR
export function formatPrice(amount) {
  return new Intl.NumberFormat('es-ES', {
    style: 'currency',
    currency: 'EUR'
  }).format(amount);
}

// Truncar títulos
export function truncateTitle(title, maxLength = 40) {
  let text = title || '(sin título)';
  if (typeof text === 'object' && text.original) {
    text = text.original;
  }
  return text.length > maxLength 
    ? text.substring(0, maxLength) + '...' 
    : text;
}
```

---

# CHECKLISTS

## ✅ CHECKLIST CSS

Antes de modificar/crear CSS:

- [ ] ¿Revisé `variables.css` para ver variables disponibles?
- [ ] ¿Usé SOLO variables CSS (no colores hex)?
- [ ] ¿Usé SOLO variables para espaciado (no px directos)?
- [ ] ¿Usé SOLO variables para tipografía?
- [ ] ¿Usé clases reutilizables (`.ml-hidden`, `.ml-card`)?
- [ ] ¿Evité estilos inline?
- [ ] ¿Las excepciones están justificadas? (valores muy específicos)
- [ ] ¿Validé con grep que no hay hardcoded?

```bash
# Validar CSS (no debe devolver nada)
grep -n "px\|#[0-9a-fA-F]\{3,6\}\|rgba(" archivo.css | grep -v "var(--"
```

---

## ✅ CHECKLIST JAVASCRIPT

Antes de escribir código JS:

- [ ] ¿Busqué en `utils.js` si ya existe?
- [ ] ¿Busqué en el módulo correspondiente?
- [ ] ¿Importé funciones en lugar de copiar?
- [ ] ¿Usé `logger` en lugar de `console.log`?
- [ ] ¿Usé `sleep()` de utils en lugar de `setTimeout`?
- [ ] ¿Usé `chrome.runtime.sendMessage` en lugar de `fetch()`?
- [ ] ¿Limpié event listeners al desmontar?
- [ ] ¿Limpié intervals y timeouts?
- [ ] ¿Usé AbortController para cancelar fetch?
- [ ] ¿Evité closures innecesarios?

---

## ✅ CHECKLIST BACKEND

Antes de crear/modificar endpoint:

- [ ] ¿Filtré por `account_id` en queries a `listings`?
- [ ] ¿Verifiqué propiedad del recurso (user_id)?
- [ ] ¿Usé columnas de BD en lugar de parsear `json_data`?
- [ ] ¿Limité resultados con `LIMIT`?
- [ ] ¿Validé entrada del usuario?
- [ ] ¿Usé transacciones para operaciones atómicas?
- [ ] ¿Logueé con contexto suficiente?
- [ ] ¿Actualicé `VERSION.txt` y `main.py`?

---

## ✅ CHECKLIST PUBLICACIÓN

Antes de modificar flujo de publicación:

- [ ] ¿Validé que NO hay publicación activa? (REGLA #41)
- [ ] ¿Actualicé ButtonManager.updateState()? (REGLA #40)
- [ ] ¿Asigné `publish_idx` correctamente? (REGLA #17)
- [ ] ¿Usé delays aleatorios (humanDelay)?
- [ ] ¿Esperé activamente renderizado de combos? (REGLA #15)
- [ ] ¿Limpié título/descripción con clearThenSetText?
- [ ] ¿Eliminé item viejo ANTES de publicar?
- [ ] ¿Migré carpeta de imágenes al cambiar ID? (REGLA #14)
- [ ] ¿Actualicé rutas en BD?
- [ ] ¿Manejé errores con `error_retry`?
- [ ] ¿Logueé cada paso del proceso?

---

# ERRORES COMUNES

## ❌ Duplicar código

```javascript
// ❌ MAL
// archivo1.js
function sleep(ms) { return new Promise(r => setTimeout(r, ms)); }

// archivo2.js
function sleep(ms) { return new Promise(r => setTimeout(r, ms)); }

// ✅ BIEN
// utils.js
export function sleep(ms) { return new Promise(r => setTimeout(r, ms)); }

// archivo1.js
import { sleep } from '../utils.js';

// archivo2.js
import { sleep } from '../utils.js';
```

---

## ❌ Queries sin filtro de aislamiento

```sql
-- ❌ MAL: Expone datos de otros usuarios
SELECT * FROM listings WHERE id_wallapop = ?

-- ✅ BIEN
SELECT * FROM listings WHERE account_id = ? AND id_wallapop = ?
```

---

## ❌ No verificar propiedad

```python
# ❌ MAL
item = db.query(Listing).filter(Listing.id == item_id).first()
# No verifica si pertenece al usuario

# ✅ BIEN
item = db.query(Listing).filter(
    Listing.id == item_id,
    Listing.account_id == account_id
).first()

account = db.query(WallaAccount).filter(
    WallaAccount.id == account_id,
    WallaAccount.user_id == current_user.id
).first()

if not account:
    raise HTTPException(403, "Acceso denegado")
```

---

## ❌ Parsear JSON innecesariamente

```python
# ❌ MAL
data = json.loads(row["json_data"])
price = data.get("sale_price", {}).get("amount", 0)

# ✅ BIEN
price = float(row["price_amount"]) if row["price_amount"] else 0
```

---

## ❌ Manipular botones directamente

```javascript
// ❌ MAL
const btnResume = document.getElementById('btn-resume-pending');
btnResume.disabled = true;
btnResume.classList.add('ml-hidden');

// ✅ BIEN
ButtonManager.updateState({ processingCount: 1 });
```

---

## ❌ No validar publicación activa

```javascript
// ❌ MAL
async function startPublish() {
  // Directamente publica
}

// ✅ BIEN
async function startPublish() {
  const status = await getPublishStatus(accountId);
  
  if (status.processing > 0) {
    return toast('⚠️ Ya hay un anuncio publicándose', 'warn');
  }
  
  // Continuar...
}
```

---

## ❌ Esperas fijas en lugar de waitFor

```javascript
// ❌ MAL
await sleep(500);
const listbox = combo.querySelector('[role="listbox"]');

// ✅ BIEN
const listbox = await waitFor(() => {
  return combo.querySelector('[role="listbox"]');
}, { timeout: 5000, interval: 100 });
```

---

## ❌ No limpiar event listeners

```javascript
// ❌ MAL: Memory leak
function init() {
  window.addEventListener('message', handleMessage);
}

// ✅ BIEN
let messageHandler;

function init() {
  messageHandler = handleMessage.bind(this);
  window.addEventListener('message', messageHandler);
}

function cleanup() {
  if (messageHandler) {
    window.removeEventListener('message', messageHandler);
    messageHandler = null;
  }
}
```

---

## ⚠️ fetch() directo - Usar con criterio

```javascript
// ✅ PREFERIDO - A través del Service Worker
const response = await chrome.runtime.sendMessage({
  type: 'API.FETCH_JSON',
  url: '/api/user/preferences',
  method: 'GET'
});

// ✅ ACEPTABLE - Excepciones válidas:
// 1. Cargar archivos locales de extensión
fetch(chrome.runtime.getURL('intro.html'))

// 2. Autenticación en auth.js
const res = await fetch(`${API_BASE}/api/auth/google/login?${params}`);

// 3. Operaciones de créditos en credits.js
const response = await fetch(`${CONFIG.API_BASE}/api/billing/deduct-credit`, {...});
```

---

# VARIABLES CSS COMPLETAS

## Colores

```css
/* Backgrounds */
--bg-primary: #0b1220;
--bg-secondary: #0e1628;
--bg-tertiary: #0f1a33;
--bg-elevated: #202a43;

/* Text */
--text-primary: #e7ecff;
--text-secondary: #94a3b8;
--text-muted: #6b7280;
--text-white: #fff;

/* Brand */
--brand-primary: #3b82f6;
--brand-secondary: #60a5fa;
--brand-dark: #2563eb;
--brand-light: #93c5fd;

/* Semánticos */
--success-500: #10b981;
--success-600: #059669;
--error-500: #ef4444;
--error-600: #dc2626;
--warning-500: #f59e0b;
--warning-600: #d97706;
--info-500: #3b82f6;
--info-600: #2563eb;

/* WhatsApp - v10.5.144 */
--whatsapp-500: #25D366;
--whatsapp-600: #128C7E;
--whatsapp-bg: rgba(37, 211, 102, 0.1);
--whatsapp-bg-hover: rgba(37, 211, 102, 0.2);

/* Overlays */
--overlay-light: rgba(255, 255, 255, 0.03);
--overlay-medium: rgba(255, 255, 255, 0.05);
--overlay-strong: rgba(255, 255, 255, 0.1);
--overlay-dark-light: rgba(0, 0, 0, 0.1);
--overlay-dark-medium: rgba(0, 0, 0, 0.2);
--overlay-dark-strong: rgba(0, 0, 0, 0.4);
--brand-overlay-light: rgba(59, 130, 246, 0.1);
--brand-overlay-medium: rgba(59, 130, 246, 0.2);
```

## Espaciado

```css
--space-1: 4px;
--space-2: 8px;
--space-3: 12px;
--space-4: 16px;
--space-5: 20px;
--space-6: 24px;
--space-8: 32px;
--space-10: 40px;
--space-12: 48px;
--space-16: 64px;
```

## Tipografía

```css
/* Tamaños */
--text-xs: 12px;
--text-sm: 14px;
--text-base: 16px;
--text-lg: 18px;
--text-xl: 20px;
--text-2xl: 24px;
--text-3xl: 30px;

/* Pesos */
--font-normal: 400;
--font-medium: 500;
--font-semibold: 600;
--font-bold: 700;

/* Line heights */
--leading-none: 1;
--leading-tight: 1.25;
--leading-normal: 1.5;
--leading-relaxed: 1.75;
```

## Bordes

```css
/* Anchos */
--border-thin: 1px;
--border-medium: 2px;
--border-thick: 4px;

/* Radios */
--radius-sm: 4px;
--radius-base: 6px;
--radius-md: 8px;
--radius-lg: 12px;
--radius-xl: 16px;
--radius-2xl: 20px;
--radius-full: 9999px;

/* Colores */
--border-primary: #1f2a44;
--border-secondary: #2b3658;
```

## Sombras

```css
--shadow-sm: 0 1px 2px rgba(0, 0, 0, 0.05);
--shadow-md: 0 4px 6px rgba(0, 0, 0, 0.1);
--shadow-lg: 0 10px 15px rgba(0, 0, 0, 0.15);
--shadow-xl: 0 20px 25px rgba(0, 0, 0, 0.2);
--shadow-2xl: 0 25px 50px rgba(0, 0, 0, 0.25);
```

## Transiciones

```css
--duration-fast: 150ms;
--duration-base: 200ms;
--duration-medium: 300ms;
--duration-slow: 500ms;

--ease-in: cubic-bezier(0.4, 0, 1, 1);
--ease-out: cubic-bezier(0, 0, 0.2, 1);
--ease-in-out: cubic-bezier(0.4, 0, 0.2, 1);

--transition-base: all var(--duration-base) var(--ease-in-out);
```

## Z-index

```css
--z-base: 1;
--z-dropdown: 1000;
--z-sticky: 1020;
--z-fixed: 1030;
--z-modal: 1040;
--z-popover: 1050;
--z-tooltip: 1060;
--z-max: 9999;
```

---

# RESUMEN FINAL

## ⭐ 55 Reglas de Oro

1-10: Fundamentales (Imports, CSS, Logging, Aislamiento)
11-16: Centralización (Funciones, Títulos, Registros, Migración)
17-24: Publicación (Validación, Envíos, Timeouts, Estados)
25-28: Performance (Polling, Límites, Debounce, Lazy loading)
29-35: Memoria (Event listeners, Intervals, WebSocket, DOM, AbortController)
36-39: Backend (Service Worker, Límites, Columnas BD)
40-43: Frontend (Botones, Validación, Overlay, Variables CSS)
44-46: Imágenes y Multi-Worker (URLs centralizadas, Redis)
47-48: Código Limpio (No logs custom, Verificar sintaxis)
52-53: Tokens Wallapop (Refresh preventivo, Reintento 401)
**54-55: Estado Unificado (module-state.js, getSelectionState) 🆕**

## 🏗️ Arquitecturas Documentadas

- ✅ Extensión completa (Panel UI + Service Worker + Content Script)
- ✅ Backend FastAPI
- ✅ WebSocket y SSE
- ✅ Base de Datos (listings, walla_accounts)
- ✅ **Estado Unificado Listings (module-state.js) 🆕**

## 🔄 Flujos Completos

- ✅ **Flujo publicación DETALLADO (23 pasos + 7 secciones form)**
- ✅ Flujo backup (simplificado v10.5.20)
- ✅ Flujo créditos/billing
- ✅ Flujo eliminación
- ✅ Flujo autenticación (Email + Google OAuth)
- ✅ Flujo editor de anuncios
- ✅ Flujo cuentas Wallapop
- ✅ Flujo WebSocket
- ✅ **Flujo Sincronización Estado Selección 🆕**

## 📦 Módulos Críticos

- ✅ Publishing Monitor (progress-tab.js)
- ✅ Button Manager (publish-button-manager.js)
- ✅ **Listings Module (module-state.js como fuente única) 🆕**
- ✅ Credits Manager (credits.js)
- ✅ Delete Handler (delete-handler.js)
- ✅ Listing Editor (listing-editor/)
- ✅ Success Modal (success-modal.js)
- ✅ Form Filler (form-filler.js - 2192 líneas)

## ✅ 5 Checklists

- CSS, JavaScript, Backend, Publicación
- **Migración Estado Unificado 🆕**

## 🚫 10+ Errores Comunes

- Con ejemplos de mal/bien código
- **Imports de selectAllByFilter desde state-manager.js (ELIMINADO) 🆕**

## 🎨 Variables CSS

- 150+ variables documentadas

---

## 🔴 NUEVAS SECCIONES v5.6 (2025-11-28)

---

## 📋 SISTEMA DE LOGS Y DEBUG MODE (v10.5.115+)

### Problema Resuelto

Los logs de MitikLive (`[INFO]`, `[TOKEN]`, `[PUBLISH]`) aparecían en la consola de Wallapop, ensuciando la vista y revelando información técnica al usuario.

### Solución Implementada

**Sistema de 3 niveles controlado por `debugMode` en chrome.storage:**

| Componente | debugMode OFF | debugMode ON |
|------------|---------------|--------------|
| **Panel logger** (`scripts/panel/logger.js`) | Solo ERROR+ en consola | DEBUG+ en consola |
| **Service Worker** (`sw/walla.js`, `sw/handlers/*`) | NADA en consola | Logs `[TOKEN]` y `[PUBLISH]` |
| **Content Script** (contexto Wallapop) | NADA (return inmediato) | NADA (siempre silenciado) |

### Arquitectura de Logs

```
scripts/panel/logger.js          → Logger principal del panel
├── LOG_LEVELS: DEBUG=0, INFO=1, WARN=2, ERROR=3
├── minLevel: controlado por debugMode
└── En contexto Wallapop: return inmediato (nunca genera logs)

sw/walla.js
├── DEBUG_MODE: leído de chrome.storage.local['debugMode']
├── dlog(): solo si DEBUG_MODE === true
└── dwarn(): solo si DEBUG_MODE === true

sw/handlers/publish-process-next.js
├── Mismo patrón que walla.js
└── Logs [TOKEN] y [PUBLISH] usan dlog/dwarn
```

### Código Crítico

#### sw/walla.js - Control de DEBUG

```javascript
// ✅ v10.5.115: Flag DEBUG controlado por debugMode en storage
let DEBUG_MODE = false;
chrome.storage.local.get(['debugMode'], (r) => { DEBUG_MODE = !!r.debugMode; });
chrome.storage.onChanged.addListener((changes) => {
  if (changes.debugMode) DEBUG_MODE = !!changes.debugMode.newValue;
});
const dlog = (...args) => DEBUG_MODE && console.log(...args);
const dwarn = (...args) => DEBUG_MODE && console.warn(...args);
```

#### scripts/panel/logger.js - Silenciar en Wallapop

```javascript
_log(level, message, context = {}) {
  // ✅ v10.5.114: NO generar logs en absoluto cuando estamos en Wallapop
  const isWallapopContext = typeof window !== 'undefined' && 
    window.location?.hostname?.includes('wallapop');
  
  if (isWallapopContext) {
    return; // No hacer NADA en contexto de Wallapop
  }
  
  // ... resto del código solo se ejecuta en panel/otras páginas
}
```

### REGLA #56: Nunca mostrar logs en consola de Wallapop

```
❌ PROHIBIDO:
- console.log() directo en content_script.js
- logger.info() sin verificar contexto
- Logs que revelen información técnica en Wallapop

✅ CORRECTO:
- Usar dlog/dwarn en SW (controlados por debugMode)
- logger silencia automáticamente en contexto Wallapop
- Todo visible en pestaña Logs del panel
```

### REGLA #57: debugMode controla TODOS los logs de diagnóstico

```
debugMode = false (producción):
├── Consola Wallapop: VACÍA
├── Consola Panel: Solo errores
└── Consola SW: VACÍA

debugMode = true (desarrollo):
├── Consola Wallapop: VACÍA (siempre)
├── Consola Panel: DEBUG + INFO + WARN + ERROR
└── Consola SW: [TOKEN], [PUBLISH], etc.
```

---

## 🔑 REFRESH TOKEN MITIKLIVE (v10.5.116+)

### Problema Resuelto

El token JWT de MitikLive expiraba durante publicaciones largas (15 minutos default), causando errores de WebSocket "Token inválido" cada 30 segundos.

### Solución Implementada

**Sistema de auto-refresh con múltiples capas:**

1. **token_refresher.js** - Verifica cada 3 minutos si el token expira pronto
2. **chrome.alarms** - Alarm `mitiklive.refresh` cada 10 minutos como backup
3. **service_worker.js** - Handler de alarm que ejecuta doRefresh()

### Arquitectura

```
sw/token_refresher.js
├── startAutoRefresh()           → Inicia interval cada 3 min
├── checkAndRefresh()            → Verifica si token expira en <12 min
├── doRefresh()                  → Llama a api_client.doRefresh()
└── Se inicia en boot.js cuando hay sesión activa

sw/keepalive.js
└── setupKeepAlive()             → Crea alarm 'mitiklive.refresh' cada 10 min

service_worker.js
└── chrome.alarms.onAlarm        → Si alarm='mitiklive.refresh' → doRefresh()

sw/api_client.js
├── doRefresh()                  → POST /api/auth/refresh
├── session.access               → Token actual en memoria
├── session.refresh              → Refresh token para renovar
└── saveSession()                → Persiste en chrome.storage
```

### Código Crítico

#### service_worker.js - Handler de Alarm

```javascript
chrome.alarms.onAlarm.addListener(async (alarm) => {
  console.log(`[SW] ⏰ Alarm recibida: ${alarm.name}`);
  
  // Keep-alive periódico (MitikLive token refresh)
  if (alarm.name === 'mitiklive.refresh') {
    await loadSession();
    try { 
      setAuthState(AUTH_STATES.AUTHENTICATING); 
      const result = await doRefresh(); 
      console.log('[SW] ✅ Token MitikLive refrescado via alarm');
    } catch (e) {
      console.warn('[SW] ⚠️ Error refrescando token MitikLive:', e.message || e);
    }
    return;
  }
  // ... otros handlers de alarm
});
```

#### sw/api_client.js - doRefresh()

```javascript
export async function doRefresh() {
  console.log('[AUTH] 🔄 doRefresh llamado | refresh:', !!session.refresh, '| device_id:', !!session.device_id);
  
  if (!session.refresh || !session.device_id) throw new Error('no_refresh_or_device');
  
  // ... validaciones ...
  
  const res = await fetchWithTimeout(`${CONFIG.API_BASE}/api/auth/refresh`, {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ refresh: session.refresh, device_id: session.device_id })
  }, 10000);
  
  // ... manejo de errores ...
  
  const data = await res.json();
  session.access = data.access;
  session.refresh = data.refresh || session.refresh;
  session.exp = data.exp || session.exp;
  await saveSession();
  
  console.log('[AUTH] ✅ Token refrescado OK | exp:', new Date(session.exp * 1000).toLocaleTimeString());
  return session.access;
}
```

### REGLA #58: Token MitikLive debe renovarse automáticamente

```
✅ CORRECTO:
- startAutoRefresh() se llama al cargar sesión en boot.js
- Alarm 'mitiklive.refresh' existe como backup
- doRefresh() loggea resultado para diagnóstico

❌ PROHIBIDO:
- Catch vacío que silencie errores de refresh
- Asumir que el token nunca expira
- No tener fallback si token_refresher falla
```

### Configuración Backend Recomendada

```bash
# .env del backend
ACCESS_MINUTES=60    # ✅ Recomendado: 60 minutos (antes era 15)
REFRESH_DAYS=30      # Token de refresh válido 30 días
```

---

## 🔄 REFRESH TOKEN WALLAPOP VIA IFRAME (v10.5.77+)

### Problema Resuelto

El token de Wallapop puede expirar durante publicaciones largas (delays de 1 hora entre anuncios). Los métodos anteriores (KICK, FORCE_REFRESH) no funcionaban consistentemente.

### Solución Implementada

**Iframe oculto que carga `/app/profile/info`:**

1. Wallapop al cargar esa página hace petición a `/me` con cookies
2. El interceptor de `webRequest` captura el nuevo token
3. El token se actualiza en memoria y storage

### Arquitectura

```
content_script.js
└── TOKEN.REFRESH_VIA_IFRAME handler
    ├── Crea iframe oculto (1x1px, opacity:0)
    ├── Carga https://es.wallapop.com/app/profile/info?_t={timestamp}
    ├── Espera evento 'load' (máx 10s)
    └── Token capturado por interceptor webRequest

sw/walla.js
└── refreshWallaCredsIfOld()
    ├── Envía TOKEN.REFRESH_VIA_IFRAME al content_script
    ├── Espera hasta 8s por token nuevo
    ├── Compara token nuevo vs viejo (por valor o timestamp)
    └── Si no llega nuevo, usa actual si no expiró
```

### Código Crítico

#### content_script.js - Iframe Handler

```javascript
if (msg?.type === 'TOKEN.REFRESH_VIA_IFRAME') {
  try {
    const IFRAME_ID = '__ml_token_refresh_iframe__';
    const IFRAME_BASE_URL = 'https://es.wallapop.com/app/profile/info';
    
    let iframe = document.getElementById(IFRAME_ID);
    
    if (!iframe) {
      iframe = document.createElement('iframe');
      iframe.id = IFRAME_ID;
      iframe.style.cssText = 'position:fixed;width:1px;height:1px;opacity:0;pointer-events:none;left:-9999px;top:-9999px;';
      document.body.appendChild(iframe);
    }
    
    // Crear promesa para esperar carga
    const loadPromise = new Promise((resolve, reject) => {
      const timeout = setTimeout(() => reject(new Error('Iframe timeout (10s)')), 10000);
      iframe.addEventListener('load', () => { clearTimeout(timeout); resolve(true); });
    });
    
    // ✅ Cache-busting para forzar petición nueva
    iframe.src = IFRAME_BASE_URL + '?_t=' + Date.now();
    
    await loadPromise;
    await new Promise(r => setTimeout(r, 1500)); // Esperar peticiones API
    
    sendResponse({ ok: true, method: 'iframe' });
  } catch (e) {
    sendResponse({ ok: false, error: e.message });
  }
  return;
}
```

#### sw/walla.js - refreshWallaCredsIfOld()

```javascript
export async function refreshWallaCredsIfOld(tabId, _maxAgeMs = 90000, forceRefresh = false) {
  const oldToken = wallaCreds.token;
  const oldTs = wallaCreds.ts || 0;
  
  // Usar iframe para refrescar token
  try {
    const refreshResult = await chrome.tabs.sendMessage(tabId, { type: 'TOKEN.REFRESH_VIA_IFRAME' });
  } catch (e) {
    dwarn('[TOKEN] ⚠️ No se pudo ejecutar iframe refresh:', e.message);
  }
  
  // Esperar token nuevo (máx 8s)
  const t0 = Date.now();
  while (Date.now() - t0 < 8000) {
    if (wallaCreds.token && wallaCreds.token !== oldToken) {
      return { ok: true, refreshed: true, creds: wallaCreds };
    }
    await new Promise(r => setTimeout(r, 300));
  }
  
  // Si no llegó nuevo, usar actual si válido
  const finalAge = getTokenRealAge();
  if (wallaCreds.token && !finalAge.isExpired && finalAge.remainingSeconds > 10) {
    return { ok: true, refreshed: false, creds: wallaCreds };
  }
  
  return { ok: false, refreshed: false, creds: null, reason: 'token_expired' };
}
```

### REGLA #59: Siempre usar iframe para refrescar token Wallapop

```
❌ DEPRECADO:
- CAPTURE.KICK_ONCE (scroll para forzar petición)
- CAPTURE.FORCE_REFRESH (click en botones)
- Depender de timing de peticiones

✅ CORRECTO:
- TOKEN.REFRESH_VIA_IFRAME
- Iframe oculto a /app/profile/info
- Cache-busting con ?_t={timestamp}
- TOKEN.CLEANUP_IFRAME al finalizar publicación
```

---

## 🛡️ PREVENCIÓN CIERRE SESIÓN WALLAPOP (v10.5.108+)

### Problema Resuelto

Wallapop cerraba la sesión del usuario cuando:
1. Los logs de la extensión se mostraban en su consola (sospecha de actividad automatizada)
2. Se hacían muchas peticiones sin interacción real del usuario

### Solución Implementada

**Silenciamiento total de logs en contexto Wallapop + Iframe invisible para refresh:**

1. **Logger silenciado** - En contexto Wallapop, `_log()` hace `return` inmediato
2. **Console.log eliminados** - Todos los logs visibles reemplazados por `dlog`/`dwarn`
3. **Iframe invisible** - El refresh de token usa iframe que parece navegación normal
4. **Delays humanizados** - Entre operaciones para simular comportamiento humano

### Señales que Wallapop Podría Detectar (evitadas)

```
❌ SEÑALES PELIGROSAS (ahora evitadas):
- Logs [INFO], [TOKEN], [PUBLISH] en consola
- Peticiones API sin cookies de sesión
- Tokens JWT visibles en logs
- Patrones de timing demasiado regulares
- Headers inusuales en peticiones

✅ AHORA IMPLEMENTADO:
- Consola de Wallapop completamente limpia
- Refresh via navegación real (iframe a /profile/info)
- Delays configurables por usuario (30s a 2h)
- Cookies de sesión siempre incluidas
```

### REGLA #60: Nunca revelar automatización en contexto Wallapop

```
✅ OBLIGATORIO:
- Logs silenciados en hostname wallapop
- Refresh de token simula navegación real
- Delays humanizados entre operaciones
- Sin diferencias detectables vs usuario real

❌ PROHIBIDO:
- console.log() en content_script (excepto errores críticos)
- Headers personalizados que revelen extensión
- Patrones de timing predecibles
- Peticiones sin cookies de sesión
```

### Logs de Diagnóstico (solo en SW con debugMode ON)

```
[TOKEN] 🔄 Refrescando via iframe... (actual: 4m 30s)
[TOKEN] 📦 Solicitando refresh via iframe...
[TOKEN] 📦 Resultado iframe: ✅
[TOKEN] ✅ Token NUEVO! Restante: 4m 58s
```

---

## 🆕 REGLAS DE VERIFICACIÓN DE VERSIÓN (v5.7)

### REGLA #61: NUNCA usar cache local para verificación de versión

```javascript
// ❌ MAL - Cache local en sessionStorage/localStorage
const VERSION_CHECK_KEY = 'ml_version_check';
const cached = sessionStorage.getItem(VERSION_CHECK_KEY);
if (cached) return JSON.parse(cached).result;

// ✅ BIEN - Siempre consultar backend
export async function checkVersion() {
  const manifestData = chrome.runtime.getManifest();
  const currentVersion = manifestData.version;
  
  const response = await chrome.runtime.sendMessage({
    type: 'API.FETCH_JSON',
    url: `/api/system/version-check?current_version=${currentVersion}`,
    method: 'GET'
  });
  
  if (response?.ok && response.data?.needs_update) {
    showUpdateModal(response.data.required_version, response.data.download_url);
    return true;
  }
  return false;
}
```

**Razones:**
- Usuarios cambian de PC/navegador
- Usuario borra datos del navegador
- Modo incógnito no tiene cache
- Backend ya tiene la lógica optimizada (1 query ligera)

### REGLA #62: Verificar versión ANTES de todo, SIN requerir login (v2.2.3)

```javascript
// ✅ CORRECTO - Flujo en panel.js (inicio de initPanel)
(async function initPanel() {
  try {
    // PRIMERO: Verificar versión SIN auth
    const needsUpdate = await checkVersionWithoutAuth();
    if (needsUpdate) {
      return;  // Modal bloqueante, NO continuar init
    }
    
    // Solo si versión OK, continuar inicialización normal
    if (UI?.init) await UI.init();
    if (AUTH?.init) await AUTH.init();
    // ... resto de inits
  }
})();

// Función que usa fetch directo (no necesita sesión)
async function checkVersionWithoutAuth() {
  const currentVersion = chrome.runtime.getManifest().version;
  const apiBase = /* obtener de storage o default */;
  
  const response = await fetch(
    `${apiBase}/api/system/version-check?current_version=${currentVersion}`
  );
  
  const data = await response.json();
  if (data.needs_update) {
    const { showUpdateModal } = await import('./scripts/panel/maintenance.js');
    showUpdateModal(data.required_version, data.download_url);
    return true;
  }
  return false;
}
```

**Flujo completo (v2.2.3):**
```
Panel se abre
    ↓
checkVersionWithoutAuth() ← SIN login, fetch directo
    ↓
GET /api/system/version-check (endpoint público)
    │
    ├─ check_extension_version = 0 → needs_update: false → Continuar
    │
    └─ check_extension_version = 1
            │
            ├─ Versiones iguales → Continuar init
            │
            └─ Versiones diferentes → Modal bloqueante (STOP)
                                      No hay botón cerrar
                                      Usuario DEBE actualizar
```

**❌ ANTES (bug):** Solo verificaba en login → usuarios con sesión activa se saltaban el check
**✅ AHORA:** Verifica SIEMPRE al abrir panel, independiente del estado de sesión

### REGLA #63: Backend controla versión via system_settings

**Tabla `system_settings`:**
```sql
-- Flag para activar/desactivar verificación
INSERT INTO system_settings (`key`, value, description) VALUES 
('check_extension_version', '1', 'Activar/desactivar verificación de versión (0=desactivado, 1=activado)');

-- Versión EXACTA requerida
INSERT INTO system_settings (`key`, value, description) VALUES 
('required_extension_version', '10.5.144', 'Versión EXACTA de extensión requerida');
```

**Endpoint `/api/system/version-check`:**
```python
@router.get("/version-check")
def check_extension_version(
    current_version: str = Query(...),
    db: Session = Depends(get_db)
):
    # 1. Verificar si check está activado
    check_enabled = db.execute(text(
        "SELECT value FROM system_settings WHERE `key` = 'check_extension_version'"
    )).scalar()
    
    if not check_enabled or check_enabled == '0':
        return {"needs_update": False, ...}  # Bypass
    
    # 2. Comparar versión EXACTA
    required_version = db.execute(text(
        "SELECT value FROM system_settings WHERE `key` = 'required_extension_version'"
    )).scalar() or "10.5.144"
    
    needs_update = (current_version != required_version)
    
    return {
        "needs_update": needs_update,
        "required_version": required_version,
        "download_url": "https://www.mitiklive.com/wallapop/assets/downloads/extension.zip"
    }
```

**⚠️ IMPORTANTE:** Actualizar `required_extension_version` en BD ANTES de desplegar nueva versión.

### REGLA #64: Flujo post-login centralizado con onLoginSuccess() (v2.2.3)

```javascript
// ✅ CORRECTO - auth.js tiene función centralizada
async function onLoginSuccess({ 
  isNewUser = false, 
  lastSelectedAccountId = null, 
  provider = 'unknown'  // 'email', 'google', 'facebook', 'apple'
} = {}) {
  const logPrefix = `[AUTH-${provider.toUpperCase()}]`;
  
  // 1. Cargar cuentas de Wallapop
  await loadAccounts();
  
  // 2. Seleccionar última cuenta usada (si aplica)
  if (lastSelectedAccountId) { /* ... */ }
  
  // 3. Activar filtro "ACTIVE" por defecto
  // 4. Cargar preferencias de usuario
  // 5. Inicializar sistema de créditos
  // 6. Refrescar botones de publicación
  // 7. Mostrar modal de bienvenida (si nuevo)
  // 8. Detectar sesión de Wallapop
  
  return true;
}

// ✅ Login email usa onLoginSuccess
if (newAuth.state === 'AUTHENTICATED') {
  await onLoginSuccess({ isNewUser, provider: 'email' });
}

// ✅ Login Google usa onLoginSuccess
if (r?.ok) {
  await onLoginSuccess({ isNewUser, lastSelectedAccountId, provider: 'google' });
}

// ✅ Futuro: Facebook, Apple usarán lo mismo
await onLoginSuccess({ isNewUser, provider: 'facebook' });
```

**❌ MAL - Código duplicado por provider:**
```javascript
// Login email
if (authenticated) {
  await loadAccounts();
  await loadPreferences();
  // ... 20 líneas más
}

// Login Google (DUPLICADO!)
if (googleOk) {
  await loadAccounts();
  await loadPreferences();
  // ... mismas 20 líneas
}
```

**Beneficios:**
- Un solo lugar para mantener el flujo post-login
- Logs consistentes: `[AUTH-EMAIL]`, `[AUTH-GOOGLE]`, `[AUTH-FACEBOOK]`
- Fácil añadir nuevos providers

### REGLA #65: Sistema de pausa en publicación (v2.2.0)

```javascript
// ✅ Estado en progress-tab.js
const STATE = {
  isPaused: false,
  // ...
};

// ✅ Persistencia en chrome.storage
await chrome.storage.local.set({ 'publish.paused': true });

// ✅ Checkpoints en publish-process-next.js (4 puntos estratégicos)
async function checkPauseAndWait(step) {
  const stored = await chrome.storage.local.get(['publish.paused']);
  if (!stored['publish.paused']) return false;
  
  console.log(`[PUBLISH] ⏸️ PAUSADO en checkpoint: ${step}`);
  
  // Esperar hasta que se reanude (polling cada 500ms)
  while (true) {
    await new Promise(r => setTimeout(r, 500));
    const check = await chrome.storage.local.get(['publish.paused']);
    if (!check['publish.paused']) {
      console.log(`[PUBLISH] ▶️ REANUDADO desde: ${step}`);
      return true;
    }
  }
}

// ✅ Llamar en puntos estratégicos
// 1. Antes de procesar siguiente anuncio
if (await checkPauseAndWait('pre_process')) { /* reanudado */ }

// 2. Después de eliminar, antes de navegar a upload
if (await checkPauseAndWait('post_delete')) { /* reanudado */ }

// 3. Después de cargar imágenes, antes de rellenar formulario
if (await checkPauseAndWait('post_images')) { /* reanudado */ }

// 4. Después de rellenar formulario, antes de publicar
if (await checkPauseAndWait('pre_submit')) { /* reanudado */ }
```

**Modal de pausa en Wallapop (sin fader):**
```javascript
// content_script.js
function showPauseOverlay(pendingCount) {
  const overlay = document.createElement('div');
  overlay.className = 'pause-overlay-wallapop';
  overlay.innerHTML = `
    <div class="pause-box-wallapop">
      <div class="pause-icon-wallapop">⏸️</div>
      <div class="pause-label-wallapop">Publicación pausada</div>
      <div class="pause-pending-wallapop">${pendingCount} anuncios pendientes</div>
      <button class="pause-resume-btn-wallapop">▶️ Reanudar</button>
    </div>
  `;
  document.body.appendChild(overlay);
}
```

**Sincronización bidireccional:**
```
Panel ⏸️ → SW guarda estado → CS muestra modal
CS ▶️ Reanudar → SW actualiza → Panel actualiza botón
```

---

## 🆕 SISTEMA DE VERIFICACIÓN DE VERSIÓN (v5.8)

### Archivos Involucrados

| Archivo | Rol |
|---------|-----|
| `panel.js` | `checkVersionWithoutAuth()` al inicio (SIN login) |
| `scripts/panel/maintenance.js` | `showUpdateModal()` exportado |
| `scripts/panel/auth.js` | `onLoginSuccess()` centralizado |
| `routers/system.py` | Endpoint `/api/system/version-check` (público) |
| BD `system_settings` | Flags `check_extension_version` + `required_extension_version` |

### Despliegue de Nueva Versión

1. **Subir nueva extensión** al servidor
2. **Actualizar BD:**
   ```sql
   UPDATE system_settings 
   SET value = '2.2.4', updated_at = NOW() 
   WHERE `key` = 'required_extension_version';
   ```
3. **TODOS los usuarios** → Modal bloqueante al abrir panel (sin importar estado de sesión)

### Desactivar Verificación (emergencia)

```sql
UPDATE system_settings 
SET value = '0', updated_at = NOW() 
WHERE `key` = 'check_extension_version';
```

---

## 🆕 LAYOUT CENTRADO PANEL EXTENDIDO (v5.7)

### Problema

Cuando el panel se abre en ventana separada (popout) o se extiende mucho, los elementos se estiraban demasiado:
- Pestañas muy separadas
- Stats ocupando todo el ancho
- Tabla desproporcionada

### Solución: Opción A - Contenido centrado con max-width

```css
/* Tab panel con ancho máximo */
.tab-panel {
  max-width: 1200px;
  margin: 0 auto;
}

/* Stats centrados */
.ml-listings-stats {
  display: flex;
  justify-content: center;
  gap: var(--space-2);
  max-width: 700px;
  margin-left: auto;
  margin-right: auto;
}

/* Pestañas logout/popout sin margin-left: auto */
.tab-button.tab-logout,
.tab-button.tab-popout {
  padding: var(--space-2);
  min-width: auto;
  /* SIN margin-left: auto - esto las separaba */
}
```

### Resultado Visual

```
┌─────────────────────────────────────────────────────────────┐
│  [Anuncios] [Actividad] [Config] [🔐] [🪟]                  │
│                                                             │
│         ┌────────────────────────────────────┐              │
│         │  ACTIVOS │ DESTACADOS │ RESERVADOS │              │
│         │   154    │     0      │     0      │              │
│         └────────────────────────────────────┘              │
│                                                             │
│    ┌──────────────────────────────────────────────┐         │
│    │ ☑ │ IMG │ TÍTULO │ PRECIO │ CATEGORÍA │ ... │         │
│    │───────────────────────────────────────────────│         │
│    │ □ │ 🖼️ │ Botas  │ 17,00€ │ Botas     │ ... │         │
│    └──────────────────────────────────────────────┘         │
│                                                             │
│  [Términos · Política · 📱 WhatsApp]  © 2025    v10.5.144  │
└─────────────────────────────────────────────────────────────┘
```

---

## 🆕 FOOTER CON WHATSAPP (v5.7)

### Variables CSS añadidas

```css
/* variables.css */
--whatsapp-500: #25D366;
--whatsapp-600: #128C7E;
--whatsapp-bg: rgba(37, 211, 102, 0.1);
--whatsapp-bg-hover: rgba(37, 211, 102, 0.2);
```

### HTML del Footer

```html
<footer id="ml-footer" class="ml-footer">
  <div class="footer-links">
    <a href="#" data-legal="terminos">Términos</a>
    <span class="separator">·</span>
    <a href="#" data-legal="cookies">Política</a>
    <span class="separator">·</span>
    <a href="https://api.whatsapp.com/send/?phone=34692034818&text=..." 
       target="_blank" class="whatsapp-link">
      <svg class="whatsapp-icon">...</svg>
      <span class="whatsapp-text">WhatsApp</span>
    </a>
  </div>
  <span class="copy">© 2025 MitikLive</span>
  <span class="version" id="extension-version">v—</span>
</footer>
```

### CSS del Footer

```css
.ml-footer .whatsapp-link {
  display: inline-flex;
  align-items: center;
  gap: var(--space-1);
  color: var(--whatsapp-500);
  padding: var(--space-1) var(--space-2);
  background: var(--whatsapp-bg);
  border-radius: var(--radius-sm);
  transition: var(--transition-base);
}

.ml-footer .whatsapp-link:hover {
  background: var(--whatsapp-bg-hover);
  transform: translateY(-1px);
}
```

### Archivos Legales Actualizados

- `legal/terminos.html` - Botón WhatsApp en contacto
- `legal/cookies.html` - Eliminado "chrome-extension://", botón WhatsApp

---
