/*  Cabecera obligatoria: NO BORRAR NI MODIFICAR este bloque inicial en ningún fichero.
  Archivo: content_script.js — Rol: Router DOM de Wallapop (fuente “en vivo”: alias/avatar/profile; sin acciones automáticas; solo responde a mensajes)
  Descripcion de todo el proyecto: 
       MitikLive Wallapop (MV3): panel lateral en Wallapop para gestionar anuncios conectado a FastAPI 
          (login JWT, export/import, backups/renovaciones).  
  Aviso: NO BORRAR los comentarios descriptivos situados encima de cada función.
  En Rol (linea mas arriba): si está vacía o el fichero se modifica o reestructura, modificar esa liena de rol  
*/

// content_script.js — Router DOM (sin acciones automáticas; solo bajo petición del panel/SW)
(() => {
  if (window.top !== window) return;
  if (window.__ML_WALLA_PING__) {
    return;
  }
  window.__ML_WALLA_PING__ = true;
  

  // ===== SISTEMA DE CAPTURA AUTOMÁTICA (2 CAPAS) =====
  window.__ml_captured_ids = [];
  window.__ml_capture_methods = [];
  
  const captureId = (data, method = 'unknown') => {
    if (!data || (!data.itemId && !data.itemSlug)) return false;
    
    // Evitar duplicados
    const exists = window.__ml_captured_ids.find(
      item => item.itemId === data.itemId || item.itemSlug === data.itemSlug
    );
    if (exists) {
      return false;
    }
    
    const captured = {
      ...data,
      capturedBy: method,
      timestamp: Date.now()
    };
    
    window.__ml_captured_ids.push(captured);
    window.__ml_last_captured = captured;
    window.__ml_capture_methods.push(method);
    
    
    return true;
  };

  // ===== CAPA 6: PerformanceObserver (extrae ID de URL) - RESTAURADO v10.5.107 =====
  (() => {
    const API_PATTERNS = [
      { name: 'Legacy pattern', regex: /\/([a-z0-9]{12,})\?language=/i },
      { name: 'Items v3', regex: /\/api\/v3\/items\/([a-z0-9]{12,})/i },
      { name: 'General items', regex: /\/v3\/general\/items\/([a-z0-9]{12,})/i },
      { name: 'Simple items', regex: /\/items\/([a-z0-9]{12,})/i },
      { name: 'Item with query', regex: /\/item\/([a-z0-9]{12,})\?/i }
    ];
    
    try {
      const observer = new PerformanceObserver((list) => {
        for (const entry of list.getEntries()) {
          if (entry.initiatorType === 'xmlhttprequest' || entry.initiatorType === 'fetch') {
            const url = entry.name;
            
            if (/https:\/\/api\.wallapop\.(?:com|es)\//i.test(url)) {
              for (const pattern of API_PATTERNS) {
                const match = url.match(pattern.regex);
                if (match) {
                  const itemId = match[1];
                  captureId({
                    source: 'performance-url',
                    itemId: itemId,
                    itemSlug: null,
                    detectedBy: pattern.name,
                    fullUrl: url
                  }, 'CAPA-6-URL');
                  break;
                }
              }
            }
          }
        }
      });
      
      observer.observe({ entryTypes: ['resource'] });
    } catch (e) {
      console.error('[ML] ❌ CAPA 6 error:', e.message);
    }
  })();



  // ===== Lazy import de helpers de DOM (sin backup.js) =====
  let dom = null;
  async function ensureDom() {
    if (dom && typeof dom.getProfileInfoSimple === 'function') return dom;
    
    try { 
      const url = chrome.runtime.getURL('scripts/dom.js');
      dom = await import(url);
    }
    catch (e) { 
      dom = {}; 
    }
    return dom;
  }

  // ===== Helpers menores =====
  const ML_DEBUG = false;
  const log = (...a) => ML_DEBUG && console.debug('[ML][CS]', ...a);
  const sleep = (ms) => new Promise(r => setTimeout(r, ms));
  
  // 🆕 Helper para delays aleatorios (simula comportamiento humano)
  const randomSleep = (min, max) => {
    const delay = Math.floor(Math.random() * (max - min + 1)) + min;
    return sleep(delay);
  };
  
  // Evita dobles “Siguiente” en paralelo si el SW reintenta
  let __ml_click_continue_lock = false;


  // ===== Utilidad: localizar y (opcionalmente) clicar "Ver más productos" =====
  async function findLoadMoreBtn({ timeout = 8000 } = {}) {
    const deadline = Date.now() + timeout;
    let lastHeight = -1;

    while (Date.now() < deadline) {
      try {
        const h = document.documentElement.scrollHeight;
        if (h !== lastHeight) {
          window.scrollTo({ top: h, behavior: 'auto' });
          lastHeight = h;
          await randomSleep(800, 1500);
        }
      } catch {}

      const hosts = Array.from(document.querySelectorAll('walla-button.hydrated, walla-button'));
      for (const h of hosts) {
        const attrText = (h.getAttribute('text') || '').toLowerCase();
        const btn = h.shadowRoot?.querySelector('button') || null;
        const shadowText = (btn?.textContent || '').toLowerCase();
        const match =
          /ver\s+m[aá]s/.test(attrText) || /ver\s+m[aá]s/.test(shadowText) ||
          /m[aá]s\s+productos/.test(attrText) || /m[aá]s\s+productos/.test(shadowText);

        if (match) {
          let tries = 0;
          while (btn && (btn.disabled || h.getAttribute('aria-disabled') === 'true') && tries++ < 15) {
            await randomSleep(300, 500);
          }
          if (btn && !btn.disabled && h.getAttribute('aria-disabled') !== 'true') {
            return btn;
          }
        }
      }
      await randomSleep(400, 600);
    }
    return null;
  }

  // ===== Fallback local para avatar (por si no importamos dom.js) =====
  function pickAvatarFromDoc(doc = document) {
    try {
      const sel = [
        'img[data-testid="user-avatar"]',
        'header img.avatar-rounded',
        'header img[class*="avatar"]',
        'img.avatar-rounded',
        'img[class*="avatar"][src*="cdn.wallapop.com"]',
        'img[class*="avatar"][src*="cloudfront"]'
      ].join(', ');
      const img = doc.querySelector(sel);
      const src = img?.currentSrc || img?.src || null;
      return (src && !src.startsWith('data:')) ? src : null;
    } catch { return null; }
  }

  // ===== Resolver URL pública /user/... priorizando DOM local y después iframe =====
  async function resolveProfileUrlBest() {
    // SIEMPRE usar iframe a /app/profile/info (única fuente de verdad)
    const m = await ensureDom();
    
    if (typeof m.resolvePublicProfileUrlViaIframe !== 'function') {
      return null;
    }
    
    try {
      const url = await m.resolvePublicProfileUrlViaIframe();
      return url || null;
    } catch (e) {
      return null;
    }
  }


  // ===== Perfil actual (href + alias + avatar) — SIEMPRE usa iframe a /app/profile/info =====
  async function getCurrentProfile() {
    // Importar dom.js
    const m = await ensureDom();
    
    // Verificar que getProfileInfoSimple esté disponible
    if (typeof m.getProfileInfoSimple !== 'function') {
      return { href: null, alias: null, avatar: null };
    }
    
    // ✅ v10.3.2: SIEMPRE usar iframe a /app/profile/info (única fuente de verdad)
    try {
      const info = await m.getProfileInfoSimple();
      
      if (info?.href && info?.alias) {
        return { 
          href: info.href, 
          alias: info.alias, 
          avatar: info.avatar || pickAvatarFromDoc(document)
        };
      }
      
      // No se pudo obtener perfil del iframe - retornar solo avatar si existe
      return { href: null, alias: null, avatar: pickAvatarFromDoc(document) };
      
    } catch (e) {
      return { href: null, alias: null, avatar: pickAvatarFromDoc(document) };
    }
  }

  async function clickMoreOnce() {
    const btn = await findLoadMoreBtn({ timeout: 7000 });
    if (!btn) return false;
    try { btn.scrollIntoView({ block: 'center' }); } catch {}
    btn.click();
    btn.dispatchEvent(new MouseEvent('click', { bubbles: true, composed: true }));
    return true;
  }

  // ===== Router de mensajes CS ⇄ Panel / SW =====

  // ✅ v7.2.51: Mostrar overlay de countdown en página Wallapop
  function showCountdownOverlay(secondsRemaining, isLongBreak = false) {
    // Eliminar overlay previo si existe
    const existing = document.querySelector('.countdown-overlay-wallapop');
    if (existing) existing.remove();

    // ✅ v10.5.34: Mensaje diferente para pausa anti-detección
    const icon = isLongBreak ? '☕' : '⏱️';
    const label = isLongBreak ? 'Pausa anti-detección' : 'Próximo anuncio en';
    const boxClass = isLongBreak ? 'countdown-box-wallapop countdown-long-break' : 'countdown-box-wallapop';

    // Crear overlay
    const overlay = document.createElement('div');
    overlay.className = 'countdown-overlay-wallapop';
    overlay.innerHTML = `
      <div class="${boxClass}">
        <div class="countdown-icon-wallapop">${icon}</div>
        <div class="countdown-label-wallapop">${label}</div>
        <div class="countdown-timer-wallapop">${secondsRemaining}s</div>
      </div>
    `;

    document.body.appendChild(overlay);

    // Actualizar countdown cada segundo
    const interval = setInterval(() => {
      secondsRemaining--;
      const timer = overlay.querySelector('.countdown-timer-wallapop');
      if (timer) {
        timer.textContent = `${secondsRemaining}s`;
      }

      if (secondsRemaining <= 0) {
        clearInterval(interval);
        overlay.remove();
      }
    }, 1000);

    // Auto-remover después del tiempo total
    setTimeout(() => {
      clearInterval(interval);
      overlay.remove();
    }, secondsRemaining * 1000);
  }

  // ✅ v2.2.2: Mostrar/ocultar modal de pausa (sin fader)
  function showPauseOverlay(pendingCount = 0) {
    // Eliminar overlay previo si existe
    hidePauseOverlay();

    const overlay = document.createElement('div');
    overlay.className = 'pause-overlay-wallapop';
    overlay.innerHTML = `
      <div class="pause-box-wallapop">
        <div class="pause-icon-wallapop">⏸️</div>
        <div class="pause-label-wallapop">Publicación pausada</div>
        <div class="pause-pending-wallapop">${pendingCount} anuncios pendientes</div>
        <button class="pause-resume-btn-wallapop" id="pause-resume-btn">
          <span class="pause-resume-icon">▶️</span>
          <span class="pause-resume-text">Reanudar</span>
        </button>
      </div>
    `;

    document.body.appendChild(overlay);

    // Listener para botón reanudar
    const resumeBtn = overlay.querySelector('#pause-resume-btn');
    if (resumeBtn) {
      resumeBtn.addEventListener('click', async () => {
        // Mostrar spinner en botón
        resumeBtn.disabled = true;
        resumeBtn.querySelector('.pause-resume-icon').textContent = '⏳';
        resumeBtn.querySelector('.pause-resume-text').textContent = 'Reanudando...';
        
        // Notificar al SW para reanudar
        try {
          await chrome.runtime.sendMessage({ type: 'PUBLISH.SET_PAUSED', paused: false });
          hidePauseOverlay();
        } catch (e) {
          console.error('[CS] Error reanudando:', e);
          resumeBtn.disabled = false;
          resumeBtn.querySelector('.pause-resume-icon').textContent = '▶️';
          resumeBtn.querySelector('.pause-resume-text').textContent = 'Reanudar';
        }
      });
    }
  }

  function hidePauseOverlay() {
    const existing = document.querySelector('.pause-overlay-wallapop');
    if (existing) existing.remove();
  }

  chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
    (async () => {
      try {
        // ✅ v7.2.51: Mostrar overlay de countdown
        // ✅ v10.5.34: Soporte para pausa anti-detección (isLongBreak)
        if (msg?.type === 'SHOW_COUNTDOWN_OVERLAY') {
          showCountdownOverlay(msg.delaySeconds || 30, msg.isLongBreak || false);
          sendResponse({ ok: true });
          return;
        }

        // ✅ v2.2.2: Mostrar modal de pausa
        if (msg?.type === 'SHOW_PAUSE_OVERLAY') {
          showPauseOverlay(msg.pendingCount || 0);
          sendResponse({ ok: true });
          return;
        }

        // ✅ v2.2.2: Ocultar modal de pausa
        if (msg?.type === 'HIDE_PAUSE_OVERLAY') {
          hidePauseOverlay();
          sendResponse({ ok: true });
          return;
        }

        // (1) ¿Logueado en Wallapop? (DOM)
        if (msg?.type === 'DOM.IS_LOGGED_IN') {
          const m = await ensureDom();
          const logged = !!(await (m.isLoggedInWallapop?.() ?? isLoggedInDoc()));
          sendResponse({ ok: true, logged }); return;
        }

        // (DELETE.ONE) — Ejecuta eliminación API-only en esta pestaña
        if (msg?.type === 'DELETE.ONE') {
          try {
            const mod = await import(chrome.runtime.getURL('scripts/delete.js'));
            const res = await mod.deleteOne(msg.item || {}); // ← delete.js ya es API-only + verificación
            sendResponse(res || { ok:false, error:'no_result' });
          } catch (e) {
            sendResponse({ ok:false, error: String(e?.message || e) });
          }
          return;
        }

        // (2) Click “Ver mi perfil público” (si existe, devuelve href) — opcional
        if (msg?.type === 'DOM.CLICK_PUBLIC_PROFILE') {
          const m = await ensureDom();
          const href = await (m.clickPublicProfileLink?.() ?? Promise.resolve(null));
          sendResponse({ ok: !!href, href }); return;
        }

        // (3) Esperar a que la lista de items esté montada (DOM)
        if (msg?.type === 'DOM.WAIT_ITEMS_LISTED') {
          const m = await ensureDom();
          const node = await m.waitItemsListed?.();
          sendResponse({ ok: true, found: !!node }); return;
        }

        // (4) BACKUP.START — localizar URL pública del perfil (sin importar backup.js)
        if (msg?.type === 'BACKUP.START') {
          try {
            const href = await resolveProfileUrlBest();
            if (href && href !== location.href) {
              location.assign(href);
              sendResponse({ ok: true, href });
            } else {
              sendResponse({ ok: /\/user\//.test(location.pathname), href: href || location.href });
            }
          } catch (e) {
            sendResponse({ ok: false, error: String(e?.message || e) });
          }
          return;
        }

        // === PUBLICACIÓN: verificar si formulario tiene datos previos ===
        if (msg?.type === 'PUBLISH.CHECK_FORM_DIRTY') {
          (async () => {
            try {
              // ✅ v6.8.0: Verificación robusta del estado del formulario
              // La idea es detectar si el wizard ha avanzado del paso inicial
              let dirty = false;
              let reason = '';
              
              // 1. VERIFICACIÓN PRINCIPAL: ¿Hay categoría/vertical seleccionada?
              // Si estamos en el paso de selección de categoría, el formulario está limpio
              // Si ya pasamos ese paso, el formulario tiene datos
              
              // Detectar si hay botones de vertical (Inmobiliaria, Motor, etc) → Paso inicial limpio
              const verticalButtons = document.querySelectorAll('[data-testid^="vertical-"]');
              const hasVerticalSelection = verticalButtons.length > 0;
              
              if (!hasVerticalSelection) {
                // No hay selector de vertical → Ya se pasó el primer paso
                dirty = true;
                reason = 'no_vertical_buttons';
              }
              
              // 2. Verificar si hay un step activo más allá del inicial
              const uploadWizardSteps = document.querySelectorAll('[class*="step"], [class*="wizard"]');
              const editContainer = document.querySelector('[data-testid="edit-listing-container"]');
              if (editContainer) {
                dirty = true;
                reason = 'edit_container_present';
              }
              
              // 3. Verificar si hay texto en campos del formulario (summary/title)
              const titleInput = document.querySelector('textarea[data-testid="summary"], input[formcontrolname="title"], textarea[formcontrolname="description"]');
              if (titleInput && titleInput.value && titleInput.value.trim().length > 0) {
                dirty = true;
                reason = 'title_has_text';
              }
              
              // 4. Verificar si hay imágenes cargadas
              const uploadedImages = document.querySelectorAll('[data-testid^="image-slot-"] img[src*="blob:"], [data-testid^="image-slot-"] img[src*="wallapop"]');
              if (uploadedImages.length > 0) {
                dirty = true;
                reason = 'images_uploaded';
              }
              
              // 5. Verificar si hay breadcrumb de categoría visible
              const categoryBreadcrumb = document.querySelector('[data-testid="category-breadcrumb"], [class*="breadcrumb"]');
              if (categoryBreadcrumb && categoryBreadcrumb.textContent && categoryBreadcrumb.textContent.trim().length > 0) {
                dirty = true;
                reason = 'category_breadcrumb_visible';
              }
              
              // 6. Verificar si estamos en URL de edición (no upload inicial)
              if (/\/upload\/\d+/.test(location.pathname)) {
                dirty = true;
                reason = 'edit_url_pattern';
              }
              
              sendResponse({ ok: true, dirty, reason, hasVerticalSelection });
            } catch (e) {
              console.error('[CS] ❌ Error verificando formulario:', e);
              // En caso de error, asumir dirty para forzar reload (más seguro)
              sendResponse({ ok: false, dirty: true, error: String(e?.message || e) });
            }
          })();
          
          return true; // Mantener canal abierto
        }

        // === PUBLICACIÓN: Esperar a que la página de upload esté completamente lista ===
        // ✅ v6.6.4: Handler inteligente que verifica URL, ausencia de modales y presencia de elementos
        if (msg?.type === 'PUBLISH.WAIT_UPLOAD_READY') {
          (async () => {
            try {
              
              const maxWaitMs = 25000; // ✅ v6.6.5: Aumentado a 25 segundos
              const startTime = Date.now();
              
              // 1. Verificar URL correcta
              while (Date.now() - startTime < maxWaitMs) {
                if (/\/app\/catalog\/upload(?:\/|$)/.test(location.pathname)) {
                  break;
                }
                await new Promise(r => setTimeout(r, 300));
              }
              
              if (!/\/app\/catalog\/upload(?:\/|$)/.test(location.pathname)) {
                console.error('[CS] ❌ URL incorrecta después de esperar:', location.pathname);
                sendResponse({ 
                  ok: false, 
                  error: 'URL incorrecta',
                  current_url: location.pathname,
                  elapsed_ms: Date.now() - startTime
                });
                return;
              }
              
              // 2. Esperar a que desaparezcan modales de Wallapop
              let modalCheckCount = 0;
              while (Date.now() - startTime < maxWaitMs) {
                const hasModal = document.querySelector('[role="dialog"], .modal, [class*="Modal"], [class*="modal"]');
                const hasOverlay = document.querySelector('.overlay, [class*="overlay"], [class*="Overlay"]');
                
                if (!hasModal && !hasOverlay) {
                  break;
                }
                
                modalCheckCount++;
                if (modalCheckCount % 5 === 0) { // Log cada 5 checks (cada ~1.25s)
                }
                await new Promise(r => setTimeout(r, 250));
              }
              
              // Dar un poco más de margen después de que desaparece el modal
              await new Promise(r => setTimeout(r, 500));
              
              // 3. Esperar a que el contenedor de vertical esté presente Y visible
              let verticalContainer = null;
              let checkCount = 0;
              while (Date.now() - startTime < maxWaitMs) {
                const container = document.querySelector('.UploadStepVertical__container, tsl-upload-step-vertical, [data-testid="UploadStepVertical"]');
                
                if (container && container.offsetParent !== null) {
                  verticalContainer = container;
                  break;
                }
                
                checkCount++;
                if (checkCount % 5 === 0) {
                }
                await new Promise(r => setTimeout(r, 200));
              }
              
              if (!verticalContainer) {
                console.error('[CS] ❌ Contenedor de vertical no apareció después de esperar');
                sendResponse({ 
                  ok: false, 
                  error: 'Contenedor de vertical no apareció',
                  elapsed_ms: Date.now() - startTime,
                  url: location.pathname
                });
                return;
              }
              
              // 4. Verificar que hay botones de categoría presentes
              const categoryButtons = verticalContainer.querySelectorAll('button, [role="button"], a');
              if (categoryButtons.length === 0) {
                console.error('[CS] ❌ No se encontraron botones de categoría en contenedor');
                sendResponse({ 
                  ok: false, 
                  error: 'No se encontraron botones de categoría',
                  elapsed_ms: Date.now() - startTime,
                  container_found: true
                });
                return;
              }
              
              
              sendResponse({ 
                ok: true, 
                buttons_found: categoryButtons.length,
                elapsed_ms: Date.now() - startTime
              });
              
            } catch (e) {
              console.error('[CS] ❌ Error en WAIT_UPLOAD_READY:', e);
              sendResponse({ 
                ok: false, 
                error: String(e?.message || e)
              });
            }
          })();
          
          return true; // Mantener canal abierto
        }

        // === PUBLICACIÓN: seleccionar vertical en /app/catalog/upload ===
        if (msg?.type === 'PUBLISH.SELECT_VERTICAL') {
          (async () => {
            try {
              const { selectVerticalForUpload } =
                await import(chrome.runtime.getURL('scripts/publish.js'));
              const result = await selectVerticalForUpload(msg.snapshot || {});
              
              // Responder con el resultado real
              sendResponse({
                ok: result?.ok !== false,  // true si no hay error
                step: 'vertical',
                result: result
              });
            } catch (e) {
              console.error('[CS] ❌ Error en selectVertical:', e);
              sendResponse({ 
                ok: false, 
                error: String(e?.message || e),
                step: 'vertical'
              });
            }
          })();
          
          return true; // ✅ Mantener canal abierto para respuesta asíncrona
        }


        if (msg?.type === 'PUBLISH.FILL_BASIC_FIELDS') {
          (async () => {
            const { fillSummary } =
              await import(chrome.runtime.getURL('scripts/publish.js'));

            const rSummary = await fillSummary(msg.snapshot || {});

            // NO hacemos click aquí. Solo rellenamos.
            sendResponse({
              ok: !!rSummary?.ok,
              step: 'basic',
              fields: { summary: rSummary },
              code: rSummary?.code  // ✅ v10.4.0: Propagar código de error
            });
          })();
          return true;
        }

        // let __ml_click_continue_lock = false;

        if (msg?.type === 'PUBLISH.CLICK_CONTINUE') {
          (async () => {
            try {
              const { clickContinue } = await import(chrome.runtime.getURL('scripts/publish.js'));
              const which   = (msg.which === 'title' || msg.which === 'photo') ? msg.which : null;
              const delay   = Number(msg.delay)   || 0;
              const timeout = Number(msg.timeout) || 12000;
              const log     = msg.log !== false;
              const r = await clickContinue({ which, delay, timeout, log });
              sendResponse(r);
            } catch (e) {
              sendResponse({ ok:false, code:'cs_click_continue_error', error:String(e?.message||e) });
            }
          })();
          return true;
        }

        // ❌ ELIMINADO v6.5.3: PUBLISH.ITEM - Handler obsoleto, nadie lo envía
        // El SW usa PUBLISH.CLICK_AND_CAPTURE_ID en su lugar

        if (msg?.type === 'PUBLISH.UPLOAD_IMAGES') {
          (async () => {
            const { uploadImagesFromSnapshot } =
              await import(chrome.runtime.getURL('scripts/publish.js'));
            const r = await uploadImagesFromSnapshot(msg.snapshot || {}, { log:true, apiBase: msg.apiBase });
            sendResponse(r);
          })();
          return true;
        }

        if (msg?.type === 'PUBLISH.SELECT_CATEGORY') {
          (async () => {
            try {
              const { selectCategoryPath } = await import(chrome.runtime.getURL('scripts/publish.js'));
              const r = await selectCategoryPath(msg);
              sendResponse(r);
            } catch (e) {
              sendResponse({ ok:false, code:'cs_select_category_error', error:String(e?.message||e) });
            }
          })();
          return true;
        }

        if (msg?.type === 'PUBLISH.CHECK_BUTTON') {
          (async () => {
            try {
              const { checkPublishButton } = await import(chrome.runtime.getURL('scripts/publish.js'));
              const r = checkPublishButton();
              sendResponse(r);
            } catch (e) {
              sendResponse({ ok:false, visible:false, error:String(e?.message||e) });
            }
          })();
          return true;
        }

        // ===============================================
        // FORM.SCAN_WIDGETS — inspecciona el formulario #step-listing
        // ===============================================
        if (msg?.type === 'FORM.SCAN_WIDGETS') {
          (async () => {
            try {
              const root = document.querySelector('#step-listing');
              if (!root) return sendResponse({ ok:false, code:'no_form_found' });

              const widgets = [];

              // --- Helper para obtener label asociado ---
              const getLabel = el => {
                let lbl = '';
                if (el.labels?.length) lbl = el.labels[0].innerText.trim();
                else {
                  const lab = el.closest('label') || el.parentElement?.querySelector('label');
                  if (lab) lbl = lab.innerText.trim();
                }
                return lbl.replace(/\*/g, '').trim();
              };

              // --- Inputs y textareas ---
              root.querySelectorAll('input, textarea').forEach(el => {
                const type = el.tagName.toLowerCase() === 'textarea' ? 'textarea' : 'input';
                const name = el.name || el.id || '';
                const label = getLabel(el);
                const visible = el.offsetParent !== null;
                widgets.push({
                  type,
                  name,
                  label,
                  value: el.value || '',
                  visible,
                  required: el.hasAttribute('required'),
                });
              });

              // --- Dropdowns (walla-dropdown / walla-floating-area) ---
              root.querySelectorAll('walla-dropdown, walla-floating-area').forEach(dd => {
                const label = dd.getAttribute('aria-label') || dd.querySelector('label')?.innerText || '';
                const hiddenInput = dd.querySelector('input[type=hidden]'); // .walla-dropdown__inner-input__hidden-input
                const options = Array.from(dd.querySelectorAll('[role="option"]'))
                  .map(o => o.getAttribute('aria-label') || o.textContent.trim());
                const value = hiddenInput?.value || '';
                widgets.push({ type:'dropdown', name:hiddenInput?.name||'', label:label.replace(/\*/g,'').trim(), options, value, visible: dd.offsetParent !== null });
              });


              // --- Toggles ---
              root.querySelectorAll('wallapop-toggle input[type=checkbox]').forEach(el => {
                widgets.push({
                  type: 'toggle',
                  name: el.name || '',
                  label: getLabel(el),
                  checked: el.checked,
                  visible: el.offsetParent !== null
                });
              });

              // --- Radios ---
              root.querySelectorAll('input[type=radio]').forEach(el => {
                widgets.push({
                  type: 'radio',
                  name: el.name || '',
                  label: el.getAttribute('aria-label') || getLabel(el),
                  checked: el.checked,
                  visible: el.offsetParent !== null
                });
              });

              sendResponse({ ok:true, widgets });
            } catch (e) {
              sendResponse({ ok:false, code:'scan_error', error:String(e?.message||e) });
            }
          })();
          return true;
        }

        // ===============================================
        // FORM.FILL_WIDGETS — rellena (básicos) y valida
        // ===============================================
        if (msg?.type === 'FORM.FILL_WIDGETS') {
          (async () => {
            try {
              const { fillListingForm } = await import(chrome.runtime.getURL('scripts/publish.js'));
              const r = await fillListingForm(msg?.snapshot || {}, { log: true });
              sendResponse(r);
            } catch (e) {
              console.error('[CONTENT] ❌ Error en fillListingForm:', e);
              sendResponse({ ok:false, code:'fill_exception', error:String(e?.message||e) });
            }
          })();
          return true;
        }

        // ===============================================
        // FORM.SHOW_ERROR_MODAL — Mostrar error al usuario
        // ✅ v4.81.0: NUEVO
        // ===============================================
        if (msg?.type === 'FORM.SHOW_ERROR_MODAL') {
          (async () => {
            try {
              // Importar showModal desde utils.js
              const utilsUrl = chrome.runtime.getURL('scripts/utils.js');
              const { showModal } = await import(utilsUrl);
              
              const { error, details } = msg;
              
              // Extraer información del item si está disponible
              const itemTitle = details?.item?.title || 'Anuncio';
              const itemImage = details?.item?.image_url || null;
              const errorType = details?.error_type || 'unknown_error';
              const isRetryable = !['missing_required_fields', 'critical_field_error'].includes(errorType);
              
              // Construir mensaje HTML limpio y visual
              let htmlContent = `<div style="display: flex; align-items: flex-start; gap: 16px; margin-bottom: 20px;">`;
              
              // Miniatura de imagen (si existe)
              if (itemImage) {
                htmlContent += `<img src="${itemImage}" alt="Miniatura" style="width: 80px; height: 80px; object-fit: cover; border-radius: 8px; flex-shrink: 0;" />`;
              } else {
                // Placeholder si no hay imagen
                htmlContent += `<div style="width: 80px; height: 80px; background: rgba(255,255,255,0.1); border-radius: 8px; flex-shrink: 0; display: flex; align-items: center; justify-content: center;">
                  <span style="font-size: 32px;">📦</span>
                </div>`;
              }
              
              // Título del anuncio
              htmlContent += `<div style="flex: 1;">
                <p style="font-size: 16px; font-weight: 600; margin: 0 0 8px 0; color: #fff;">${itemTitle}</p>
                <p style="font-size: 14px; color: rgba(255,255,255,0.7); margin: 0;">ID: ${details?.item?.wallapop_id || 'N/A'}</p>
              </div>`;
              
              htmlContent += `</div>`;
              
              // Mensaje de error principal
              htmlContent += `<div style="padding: 16px; background: rgba(239, 68, 68, 0.15); border-left: 4px solid #ef4444; border-radius: 4px; margin-bottom: 16px;">
                <p style="margin: 0; font-size: 15px; color: #fca5a5;"><strong>No se pudo completar:</strong></p>
                <p style="margin: 8px 0 0 0; color: #fff;">→ ${error}</p>
              </div>`;
              
              // Detalles adicionales (si hay campo específico)
              if (details?.details?.fields && details.details.fields.length > 0) {
                htmlContent += `<div style="padding: 12px; background: rgba(255,255,255,0.05); border-radius: 8px; font-size: 14px; margin-bottom: 16px;">`;
                htmlContent += `<p style="margin: 0 0 8px 0; font-weight: 600; color: rgba(255,255,255,0.9);">Campos afectados:</p>`;
                htmlContent += `<ul style="margin: 0; padding-left: 20px; color: rgba(255,255,255,0.8);">`;
                details.details.fields.forEach(field => {
                  const fieldName = field.field_name || field.name || 'Campo desconocido';
                  htmlContent += `<li style="margin-bottom: 4px;">${fieldName}</li>`;
                });
                htmlContent += `</ul>`;
                htmlContent += `</div>`;
              }
              
              // Estado y acciones
              if (isRetryable) {
                htmlContent += `<div style="display: flex; align-items: center; gap: 8px; padding: 12px; background: rgba(59, 130, 246, 0.15); border-radius: 8px;">
                  <span style="font-size: 20px;">🔄</span>
                  <p style="margin: 0; color: #93c5fd; font-size: 14px;"><strong>Estado:</strong> En cola para reintentar automáticamente</p>
                </div>`;
              } else {
                htmlContent += `<div style="display: flex; align-items: center; gap: 8px; padding: 12px; background: rgba(245, 158, 11, 0.15); border-radius: 8px;">
                  <span style="font-size: 20px;">⚠️</span>
                  <p style="margin: 0; color: #fcd34d; font-size: 14px;"><strong>Estado:</strong> Requiere corrección en los datos del anuncio</p>
                </div>`;
              }
              
              // Mostrar modal
              await showModal({
                title: '❌ Error de Publicación',
                html: htmlContent,
                buttons: [
                  { text: 'Cerrar', value: 'ok', primary: true }
                ]
              });
              
              sendResponse({ ok: true });
            } catch (e) {
              console.error('[CONTENT] ❌ Error mostrando modal:', e);
              sendResponse({ ok: false, error: String(e?.message || e) });
            }
          })();
          return true;
        }

        // ❌ ELIMINADO v6.5.3: PUBLISH.CLICK_PUBLISH - Handler obsoleto
        // El SW usa PUBLISH.CLICK_AND_CAPTURE_ID que maneja todo el flujo

        // NUEVO: Publicar y capturar ID (SOLO CAPA 6, con Shadow DOM)
        if (msg?.type === 'PUBLISH.CLICK_AND_CAPTURE_ID') {
          (async () => {
            try {
              
              // Resetear capturas
              window.__ml_captured_ids = [];
              window.__ml_last_captured = null;
              
              // ✅ v10.5.45: Configuración de reintentos por error de servidor
              const MAX_SERVER_ERROR_RETRIES = 3;
              const SERVER_ERROR_WAIT_MS = 15000; // 15s entre reintentos
              let serverErrorRetries = 0;
              
              // Función para detectar toast de error de Wallapop
              const detectServerError = () => {
                const snackbar = document.querySelector('.SnackbarWrapper__text, [class*="Snackbar"]');
                if (snackbar) {
                  const text = snackbar.textContent?.toLowerCase() || '';
                  if (text.includes('error') && (text.includes('servidor') || text.includes('server'))) {
                    return true;
                  }
                }
                return false;
              };
              
              // ✅ v3.0.5: Detectar error de formulario (campos requeridos faltantes)
              const detectFormError = () => {
                const snackbar = document.querySelector('.SnackbarWrapper__text, [class*="Snackbar"]');
                if (snackbar) {
                  const text = snackbar.textContent?.toLowerCase() || '';
                  // "¡ojo! revisa los campos en rojo"
                  if (text.includes('ojo') || text.includes('revisa') || text.includes('campos') || text.includes('rojo')) {
                    return true;
                  }
                }
                return false;
              };
              
              // Función para cerrar el toast de error si hay botón
              const closeSnackbar = () => {
                const closeBtn = document.querySelector('.SnackbarWrapper__close, [class*="Snackbar"] button');
                if (closeBtn) closeBtn.click();
              };
              
              // Buscar botón de publicar (Wallapop usa Web Components)
              const hostSel = 'walla-button[data-testid="continue-action-button"]';
              let host = document.querySelector(hostSel);
              
              // Esperar hasta 10s si no está visible aún
              if (!host) {
                const maxWait = 10000;
                const start = Date.now();
                while (!host && (Date.now() - start) < maxWait) {
                  await new Promise(r => setTimeout(r, 200));
                  host = document.querySelector(hostSel);
                }
              }
              
              if (!host) {
                console.error('[ML] ❌ No se encontró walla-button');
                sendResponse({ ok: false, code: 'no_publish_button' });
                return;
              }
              
              // El botón real está dentro del Shadow DOM
              const btn = host.shadowRoot?.querySelector('button');
              if (!btn) {
                console.error('[ML] ❌ No se encontró botón dentro del Shadow DOM');
                sendResponse({ ok: false, code: 'no_inner_button' });
                return;
              }
              
              // Verificar que no esté deshabilitado
              if (btn.disabled || host.getAttribute('aria-disabled') === 'true') {
                console.error('[ML] ❌ Botón deshabilitado');
                sendResponse({ ok: false, code: 'button_disabled' });
                return;
              }
              
              // ✅ v10.5.45: Loop de publicación con reintentos por error de servidor
              while (serverErrorRetries <= MAX_SERVER_ERROR_RETRIES) {
                
                // Resetear captura antes de cada intento
                window.__ml_captured_ids = [];
                window.__ml_last_captured = null;
                
                // ✅ v10.5.46: Timestamp para diagnóstico
                const clickTime = new Date().toLocaleTimeString();
                console.log(`[ML] 🖱️ Click en publicar (intento ${serverErrorRetries + 1}/${MAX_SERVER_ERROR_RETRIES + 1}) a las ${clickTime}`);
                btn.click();
                
                // Esperar captura CAPA 6 (máximo 60s)
                const maxWait = msg.maxWaitMs || 60000;
                const start = Date.now();
                let serverErrorDetected = false;
                
                while (!window.__ml_last_captured?.itemId && (Date.now() - start) < maxWait) {
                  await new Promise(r => setTimeout(r, 500));
                  
                  // ✅ v3.0.5: Detectar error de formulario PRIMERO (más común)
                  if (detectFormError()) {
                    const elapsed = Math.round((Date.now() - start) / 1000);
                    console.error(`[ML] ❌ ERROR DE FORMULARIO detectado después de ${elapsed}s`);
                    console.error(`[ML] 📋 Wallapop indica: "Revisa los campos en rojo"`);
                    closeSnackbar();
                    sendResponse({
                      ok: false,
                      code: 'form_validation_error',
                      error: 'Wallapop rechazó el anuncio. Revisa los campos en rojo.'
                    });
                    return;
                  }
                  
                  // ✅ v10.5.45: Detectar error de servidor de Wallapop
                  if (detectServerError()) {
                    const elapsed = Math.round((Date.now() - start) / 1000);
                    console.warn(`[ML] ⚠️ ERROR SERVIDOR detectado después de ${elapsed}s`);
                    console.warn(`[ML] 📊 Hora: ${new Date().toLocaleTimeString()} | Reintento: ${serverErrorRetries + 1}/${MAX_SERVER_ERROR_RETRIES}`);
                    serverErrorDetected = true;
                    closeSnackbar();
                    break;
                  }
                }
                
                // Si se capturó el ID, éxito
                if (window.__ml_last_captured?.itemId) {
                  sendResponse({
                    ok: true,
                    itemId: window.__ml_last_captured.itemId,
                    itemSlug: window.__ml_last_captured.itemSlug,
                    capturedBy: window.__ml_last_captured.capturedBy
                  });
                  return;
                }
                
                // Si fue error de servidor, refrescar token y reintentar
                if (serverErrorDetected && serverErrorRetries < MAX_SERVER_ERROR_RETRIES) {
                  serverErrorRetries++;
                  
                  // ✅ v10.5.54: Pedir al SW que refresque el token antes de reintentar
                  console.log(`[ML] 🔄 Solicitando refresh de token antes de reintento...`);
                  try {
                    const refreshRes = await chrome.runtime.sendMessage({ type: 'TOKEN.FORCE_REFRESH' });
                    if (refreshRes?.ok) {
                      console.log(`[ML] ✅ Token refrescado correctamente`);
                    } else {
                      console.warn(`[ML] ⚠️ No se pudo refrescar token:`, refreshRes?.error);
                    }
                  } catch (e) {
                    console.warn(`[ML] ⚠️ Error pidiendo refresh de token:`, e.message);
                  }
                  
                  console.log(`[ML] ⏳ Esperando ${SERVER_ERROR_WAIT_MS/1000}s antes de reintentar...`);
                  await new Promise(r => setTimeout(r, SERVER_ERROR_WAIT_MS));
                  continue; // Reintentar
                }
                
                // Timeout sin error de servidor detectado, o máximo de reintentos alcanzado
                break;
              }
              
              // Si llegamos aquí, falló
              if (serverErrorRetries > 0) {
                console.error(`[ML] ❌ Falló después de ${serverErrorRetries} reintentos por error de servidor`);
                // ✅ v10.5.59: Código específico para sesión expirada de Wallapop
                sendResponse({
                  ok: false,
                  code: 'wallapop_session_expired',
                  error: 'Wallapop ha cerrado tu sesión. Cierra sesión en Wallapop y vuelve a entrar.'
                });
              } else {
                console.error('[ML] ❌ Timeout: No se capturó ID después de', msg.maxWaitMs || 60000, 'ms');
                sendResponse({
                  ok: false,
                  code: 'timeout',
                  error: `No se capturó ID en ${msg.maxWaitMs || 60000}ms`
                });
              }
              
            } catch (e) {
              console.error('[ML] ❌ Error en publicación:', e);
              
              // Último intento: verificar si hay captura
              if (window.__ml_last_captured?.itemId) {
                sendResponse({
                  ok: true,
                  itemId: window.__ml_last_captured.itemId,
                  itemSlug: window.__ml_last_captured.itemSlug,
                  capturedBy: window.__ml_last_captured.capturedBy
                });
              } else {
                sendResponse({ ok: false, code: 'exception', error: e.message });
              }
            }
          })();
          return true;
        }









        // (5) CAPTURE.KICK_ONCE — provocar una request liviana
        if (msg?.type === 'CAPTURE.KICK_ONCE') {
          const clicked = await clickMoreOnce();
          sendResponse({ ok: true, clicked });
          return;
        }
        
        // ✅ v10.5.77: TOKEN.REFRESH_VIA_IFRAME — Refrescar token usando iframe oculto
        // El iframe carga /app/profile/info que hace petición a /me con token fresco
        if (msg?.type === 'TOKEN.REFRESH_VIA_IFRAME') {
          try {
            const IFRAME_ID = '__ml_token_refresh_iframe__';
            const IFRAME_BASE_URL = 'https://es.wallapop.com/app/profile/info';
            
            let iframe = document.getElementById(IFRAME_ID);
            const isNew = !iframe;
            
            if (!iframe) {
              // Crear iframe singleton
              iframe = document.createElement('iframe');
              iframe.id = IFRAME_ID;
              iframe.style.cssText = 'position:fixed;width:1px;height:1px;opacity:0;pointer-events:none;left:-9999px;top:-9999px;';
              document.body.appendChild(iframe);
              // Log silenciado v10.5.108 - visible en consola del panel
            } else {
              // ✅ v2.4.4: Forzar recarga limpiando caché del iframe
              // Limpiar con about:blank primero para evitar caché
              iframe.src = 'about:blank';
              await new Promise(r => setTimeout(r, 100)); // Esperar limpieza
            }
            
            // Crear promesa para esperar carga
            const loadPromise = new Promise((resolve, reject) => {
              const timeout = setTimeout(() => {
                reject(new Error('Iframe timeout (10s)'));
              }, 10000);
              
              const onLoad = () => {
                clearTimeout(timeout);
                iframe.removeEventListener('load', onLoad);
                iframe.removeEventListener('error', onError);
                // Log silenciado v10.5.108
                resolve(true);
              };
              
              const onError = (e) => {
                clearTimeout(timeout);
                iframe.removeEventListener('load', onLoad);
                iframe.removeEventListener('error', onError);
                // Error silenciado v10.5.108
                reject(new Error('Iframe load error'));
              };
              
              iframe.addEventListener('load', onLoad);
              iframe.addEventListener('error', onError);
            });
            
            // ✅ v10.5.79: Cache-busting para forzar petición nueva
            const IFRAME_URL = IFRAME_BASE_URL + '?_t=' + Date.now();
            // Log silenciado v10.5.108
            iframe.src = IFRAME_URL;
            
            await loadPromise;
            
            // ✅ v2.1.1: Detectar si la sesión de Wallapop expiró
            // Si el iframe redirigió a login, la sesión expiró
            let sessionExpired = false;
            try {
              const iframeUrl = iframe.contentWindow?.location?.href || '';
              // Wallapop redirige a /login cuando no hay sesión
              if (iframeUrl.includes('/login') || iframeUrl.includes('/auth')) {
                sessionExpired = true;
              }
            } catch (e) {
              // Error de cross-origin es normal, no podemos leer la URL
              // Pero si el iframe cargó sin token nuevo, asumimos sesión expirada
            }
            
            // Esperar un poco más para que las peticiones API se completen
            await new Promise(r => setTimeout(r, 1500));
            
            // Log silenciado v10.5.108
            sendResponse({ ok: true, method: 'iframe', isNew, sessionExpired });
            
          } catch (e) {
            // Error silenciado v10.5.108
            sendResponse({ ok: false, error: e.message });
          }
          return;
        }
        
        // ✅ v10.5.77: TOKEN.CLEANUP_IFRAME — Limpiar iframe al finalizar publicación
        if (msg?.type === 'TOKEN.CLEANUP_IFRAME') {
          const iframe = document.getElementById('__ml_token_refresh_iframe__');
          if (iframe) {
            iframe.remove();
            // Log silenciado v10.5.108
          }
          sendResponse({ ok: true, removed: !!iframe });
          return;
        }
        
        // ✅ v2.4.6: WALLAPOP.DISABLE_BUTTON — Deshabilitar botón "Subir producto"
        if (msg?.type === 'WALLAPOP.DISABLE_BUTTON') {
          try {
            const timeout = msg?.timeout || 5000;
            const startTime = Date.now();
            let wallaButton = null;
            
            // Esperar a que el botón exista
            while (Date.now() - startTime < timeout) {
              wallaButton = document.querySelector('walla-button[data-testid="continue-action-button"]');
              if (wallaButton && wallaButton.shadowRoot) {
                const btn = wallaButton.shadowRoot.querySelector('button');
                if (btn) break;
              }
              await new Promise(r => setTimeout(r, 100));
            }
            
            if (!wallaButton || !wallaButton.shadowRoot) {
              // ✅ v2.5.2: Silenciado - no es un error crítico
              sendResponse({ ok: false, reason: 'button_not_found' });
              return;
            }
            
            const btn = wallaButton.shadowRoot.querySelector('button');
            if (!btn) {
              console.warn('[CS] ⚠️ Botón interno no encontrado');
              sendResponse({ ok: false, reason: 'inner_button_not_found' });
              return;
            }
            
            // Deshabilitar
            btn.disabled = true;
            btn.style.opacity = '0.5';
            btn.style.cursor = 'not-allowed';
            btn.style.pointerEvents = 'none';
            wallaButton.setAttribute('aria-disabled', 'true');
            
            console.log('[CS] ✅ Botón Wallapop deshabilitado');
            sendResponse({ ok: true });
            
          } catch (e) {
            console.error('[CS] ❌ Error deshabilitando botón:', e);
            sendResponse({ ok: false, error: e.message });
          }
          return;
        }
        
        // ✅ v2.4.6: WALLAPOP.ENABLE_BUTTON — Habilitar botón "Subir producto"
        if (msg?.type === 'WALLAPOP.ENABLE_BUTTON') {
          try {
            const wallaButton = document.querySelector('walla-button[data-testid="continue-action-button"]');
            if (!wallaButton || !wallaButton.shadowRoot) {
              sendResponse({ ok: false, reason: 'button_not_found' });
              return;
            }
            
            const btn = wallaButton.shadowRoot.querySelector('button');
            if (!btn) {
              sendResponse({ ok: false, reason: 'inner_button_not_found' });
              return;
            }
            
            // Habilitar
            btn.disabled = false;
            btn.style.opacity = '1';
            btn.style.cursor = 'pointer';
            btn.style.pointerEvents = 'auto';
            wallaButton.setAttribute('aria-disabled', 'false');
            
            console.log('[CS] ✅ Botón Wallapop habilitado');
            sendResponse({ ok: true });
            
          } catch (e) {
            console.error('[CS] ❌ Error habilitando botón:', e);
            sendResponse({ ok: false, error: e.message });
          }
          return;
        }
        
        // ✅ v10.5.47: CAPTURE.FORCE_REFRESH — Forzar petición para capturar token fresco
        // DEPRECADO: Usar TOKEN.REFRESH_VIA_IFRAME en su lugar
        if (msg?.type === 'CAPTURE.FORCE_REFRESH') {
          try {
            // Redirigir al nuevo método de iframe
            // Log silenciado v10.5.108
            
            // Crear/recargar iframe
            const IFRAME_ID = '__ml_token_refresh_iframe__';
            const IFRAME_BASE_URL = 'https://es.wallapop.com/app/profile/info';
            
            let iframe = document.getElementById(IFRAME_ID);
            if (!iframe) {
              iframe = document.createElement('iframe');
              iframe.id = IFRAME_ID;
              iframe.style.cssText = 'position:fixed;width:1px;height:1px;opacity:0;pointer-events:none;left:-9999px;top:-9999px;';
              document.body.appendChild(iframe);
            }
            
            // ✅ v10.5.79: Cache-busting
            iframe.src = IFRAME_BASE_URL + '?_t=' + Date.now();
            await new Promise(r => setTimeout(r, 2000));
            
            // Log silenciado v10.5.108
            sendResponse({ ok: true, method: 'iframe' });
          } catch (e) {
            console.error('[TOKEN] ❌ Error en FORCE_REFRESH:', e);
            sendResponse({ ok: false, error: e.message });
          }
          return;
        }

        // (6) utilidades DOM simples
        if (msg?.type === 'DOM.GET_PROFILE_URL') {
          const url = await resolveProfileUrlBest();
          sendResponse({ ok: !!url, url });
          return;
        }

        if (msg?.type === 'DOM.CURRENT_PROFILE') {
          const info = await getCurrentProfile();
          sendResponse({
            ok: !!(info.alias || info.href || info.avatar),
            alias: info.alias || null,
            profileUrl: info.href || null,
            avatar: info.avatar || null
          });
          return;
        }

        // 🆕 Verificar si hay login activo (detecta avatar)
        if (msg?.type === 'DOM.CHECK_LOGIN') {
          const info = await getCurrentProfile();
          const hasLogin = !!(info.alias || info.avatar || info.href);
          sendResponse({ 
            ok: true, 
            hasLogin,
            alias: info.alias || null,
            avatar: info.avatar || null,
            profileUrl: info.href || null  // ✅ v6.4.1: Añadir URL completa del perfil
          });
          return;
        }

        // (7) EXPORT.PROFILE_FULL — delega en SW con URL pública        
        if (msg?.type === 'EXPORT.PROFILE_FULL') {
          const href = await resolveProfileUrlBest();
          if (!href) { sendResponse({ ok:false, error:'profile_url_not_found' }); return; }

          const r = await new Promise((res) => {
            try {
              chrome.runtime.sendMessage(
                { type:'LISTINGS.FULL_EXPORT_BY_PROFILE', profileUrl: href },
                (reply) => {
                  // Evita "Unchecked runtime.lastError: Could not establish connection..."
                  void chrome.runtime.lastError;
                  res(reply || { ok:false, error:'no_response' });
                }
              );
            } catch (e) {
              res({ ok:false, error:String(e) });
            }
          });

          sendResponse(r);
          return;
        }


        // (7b) EXPORT.PROFILE_AND_IMPORT — delega en SW
        if (msg?.type === 'EXPORT.PROFILE_AND_IMPORT') {
          const href = await resolveProfileUrlBest();
          if (!href) { sendResponse({ ok:false, error:'profile_url_not_found' }); return; }
          chrome.runtime.sendMessage({ type: 'EXPORT.PROFILE_AND_IMPORT', profileUrl: href }, (r) => {
            if (chrome.runtime.lastError) {
              sendResponse({ ok:false, error: chrome.runtime.lastError.message });
            } else {
              sendResponse(r || { ok:false, error:'no_response' });
            }
          });
          return;
        }

        // (8) Ping
        if (msg?.type === 'SYS.PING' || msg?.type === 'PING') {
          sendResponse({ ok: true, data: { now: Date.now() } }); return;
        }

        sendResponse({ ok: false, code: 'unknown' });
      } catch (e) {
        sendResponse({ ok: false, error: String(e?.message || e) });
      }
    })();
    return true; // respuesta asíncrona
  });

  // ===== Emisor ligero de cambios (login/alias/avatar/URL de perfil) → Panel =====
  // ✅ v10.3.5: Simplificado - NO usa iframes automáticamente para evitar bucles
  (function setupLiveEmitter(){
    let last = { logged: null, alias: null, avatar: null, profileUrl: null };
    let debounce = null;

    function fire(reason, state) {
      try {
        chrome.runtime.sendMessage(
          { type: 'WALLA.SESSION_CHANGED', reason, ...state },
          () => void chrome.runtime.lastError
        );
      } catch {}
    }

    // ✅ Solo detección RÁPIDA del DOM local (sin iframes)
    function detectLoginFast() {
      try {
        // Detectar si hay avatar (indica login)
        const avatarImg = document.querySelector(
          'img[data-testid="user-avatar"], header img.avatar-rounded, img.avatar-rounded'
        );
        const avatar = avatarImg?.src || null;
        const logged = !!avatar;

        // Intentar obtener href/alias del DOM local (sin iframe)
        let href = null;
        let alias = null;

        if (/^\/user\/[^\/]+-\d+/.test(location.pathname)) {
          href = location.href;
        } else {
          const profileLink = document.querySelector('a.header-link[href*="/user/"], a[href*="/user/"]');
          if (profileLink?.href && /\/user\/[^\/]+-\d+/.test(profileLink.href)) {
            href = profileLink.href;
          }
        }

        if (href) {
          const lastSeg = new URL(href).pathname.split('/').filter(Boolean).pop() || '';
          alias = lastSeg.replace(/-\d+$/, '');
        }

        return { logged, alias, avatar, profileUrl: href };
      } catch {
        return { logged: false, alias: null, avatar: null, profileUrl: null };
      }
    }

    function checkAndNotify(reason) {
      const state = detectLoginFast();

      const changed =
        state.logged !== last.logged ||
        state.alias !== last.alias ||
        state.avatar !== last.avatar ||
        state.profileUrl !== last.profileUrl;

      if (changed) {
        last = state;
        fire(reason, state);
      }
    }

    function schedule(reason = 'debounce') {
      clearTimeout(debounce);
      debounce = setTimeout(() => checkAndNotify(reason), 300);
    }

    // ✅ v10.5.105: MutationObserver restaurado pero solo en header (menos intrusivo)
    try {
      const mo = new MutationObserver(() => schedule('mut'));
      const target = document.querySelector('header') || document.body;
      mo.observe(target, { childList: true, subtree: true, attributes: true, attributeFilter: ['src', 'href'] });
    } catch {}

    // ✅ v10.5.105: Popstate restaurado (navegación SPA)
    try {
      window.addEventListener('popstate', () => schedule('popstate'));
    } catch {}

    // ✅ v10.5.109: Polling ELIMINADO - MutationObserver ya es tiempo real
    // El MutationObserver detecta cambios de avatar instantáneamente

    // ✅ v10.5.105: Disparo inicial con delay
    setTimeout(() => checkAndNotify('init'), 3000);
  })();

  // ✅ v10.5.105: Listener de visibilidad restaurado
  document.addEventListener('visibilitychange', () => {
    try {
      chrome.runtime.sendMessage({
        type: 'TAB_VISIBILITY_CHANGED',
        hidden: document.hidden,
        timestamp: Date.now()
      }).catch(() => {});
    } catch (e) {
      // Ignorar errores de comunicación
    }
  });

  // ✅ v10.5.44: Protección contra navegación durante publicación
  let _publicacionEnCurso = false;
  let _navegacionInterna = false; // Flag para permitir navegación dentro de Wallapop
  
  // ✅ v10.5.102: Limpiar estado al cargar (evita bloqueos persistentes)
  try {
    chrome.storage.local.remove(['mitiklive_publishing_active']);
  } catch (e) {}
  
  // Escuchar cuando el panel activa/desactiva publicación
  chrome.runtime.onMessage.addListener((msg) => {
    if (msg.type === 'PUBLISH.SET_ACTIVE') {
      _publicacionEnCurso = msg.active;
      console.log(`[CS] Publicación ${msg.active ? 'ACTIVA' : 'INACTIVA'}`);
      
      // Persistir en storage para sobrevivir navegaciones internas
      try {
        if (msg.active) {
          chrome.storage.local.set({ mitiklive_publishing_active: true });
        } else {
          chrome.storage.local.remove(['mitiklive_publishing_active']);
        }
      } catch (e) {}
    }
    
    // ✅ v10.5.44: Permitir navegación interna (ej: a /upload)
    if (msg.type === 'PUBLISH.ALLOW_NAVIGATION') {
      _navegacionInterna = true;
      // Reset después de 5 segundos
      setTimeout(() => { _navegacionInterna = false; }, 5000);
    }
  });
  
  // Avisar antes de navegar/refrescar si hay publicación en curso
  window.addEventListener('beforeunload', (e) => {
    // No bloquear si es navegación interna permitida
    if (_navegacionInterna) {
      return;
    }
    
    if (_publicacionEnCurso) {
      // Mensaje estándar (navegadores modernos ignoran texto personalizado)
      e.preventDefault();
      e.returnValue = '¿Seguro? Perderás la publicación en curso';
      return e.returnValue;
    }
  });

})();